/****************************************************************************
 *
 * File:            Length.java
 *
 * Description:     Length Class
 *
 * Author:          PDF Tools AG
 *
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.geometry.units;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;

/**
 * <p>Class that represents a measurable length. Provides conversion between units, parsing of string representations of lengths, and basic mathematical operations.</p>
 * 
 * <p>The unit used in PDF documents is {@link Units.POINT}, which is also used internally by the {@link Length} object to store length values.
 * For that reason, when converting units to and from the internally used unit, minor numerical differences may occur due to floating-point arithmetic.</p>
 */
public class Length implements Comparable<Length>
{
    public static enum Units {
        /**
         * The metre, symbol m, is the SI unit of length.
         */
        METRE("m"),
        /**
         * The point, symbol pt, is the default unit used in PDF documents, equal to <code>1/72in</code> or <code>25.4/72mm</code>.
         * 
         * This is the unit used internally by the {@link Length} object.
         */
        POINT("pt"),
        /**
         * The kilometre, symbol km, equal to <code>1000m</code>.
         */
        KILOMETRE("km"),
        /**
         * The centimetre, symbol cm, equal to <code>0.01m</code>.
         */
        CENTIMETRE("cm"),
        /**
         * The millimetre, symbol mm, equal to <code>0.001m</code>.
         */
        MILLIMETRE("mm"),
        /**
         * The inch, symbol in, equal to <code>25.4mm</code> or <code>72pt</code>.
         */
        INCH("in");

        private String unit;

        private Units(String unit) {
            this.unit = unit;
        }

        /**
         * Get the unit's symbol name.
         */
        public String getSymbol() {
            return unit;
        }

        @Override
        public String toString() {
            return getSymbol();
        }

        private static final Map<String, Units> symbols;
        static {
            Map<String,Units> map = new ConcurrentHashMap<String, Units>();
            for (Units instance : Units.values()) {
                map.put(instance.getSymbol(), instance);
            }
            symbols = Collections.unmodifiableMap(map);
        }

        /**
         * Get the unit enum value from its symbol, e.g. m, or in.
         * @param symbol the symbol, e.g. m, or in.
         * @return
         */
        public static Units get(String symbol) {
            Units u = symbols.get(symbol.toLowerCase());
            if (u == null)
                throw new IllegalArgumentException(String.format("Invalid unit symbol %s.", symbol));
            return u;
        }
    }

    /**
     * Constructs a newly allocated {@link Length} object using the specified value and unit.
     * @param value the numerical value.
     * @param unit the unit.
     * @return
     */
    public Length(double value, Units unit) {
        switch (unit)
        {
        case KILOMETRE:  points = value * 2834645.6692913385826771653543307; break;
        case METRE:      points = value * 2834.6456692913385826771653543307; break;
        case CENTIMETRE: points = value * 28.346456692913385826771653543307; break;
        case MILLIMETRE: points = value * 2.8346456692913385826771653543307; break;
        case POINT:      points = value; break;
        case INCH:       points = value * 72; break;
        default:
            throw new IllegalArgumentException(String.format("Invalid unit %s.", unit));
        }
    }

    /**
     * Get the numerical length value of the specified unit.
     * @param unit the unit.
     * @return
     */
    public double to(Units unit) {
        switch (unit)
        {
        case KILOMETRE:  return points / 2834645.6692913385826771653543307;
        case METRE:      return points / 2834.6456692913385826771653543307;
        case CENTIMETRE: return points / 28.346456692913385826771653543307;
        case MILLIMETRE: return points / 2.8346456692913385826771653543307;
        case POINT:      return points;
        case INCH:       return points / 72;
        default:
            throw new IllegalArgumentException(String.format("Invalid unit %s.", unit));
        }
    }

    public static final Length ZERO      = new Length(0);
    public static final Length MAX_VALUE = new Length(Double.MAX_VALUE);

    private double points;

    /**
     * @hidden
     */
    public Length(double points) {
        this.points = points;
    }

    /**
     * @hidden
     */
    public double getValue()
    {
        return points;
    }

    /**
     * Creates a {@link Length} object by parsing a String representation of a length with its unit.
     * @param value Value-unit pair of the form "&lt;value>&lt;unit>". Examples: "12.3cm" or "23.9mm". Allowed units are "um", "mm", "cm", "m", "km", "pt", and "in".
     * @throws IllegalArgumentException if unit given in {@code value} is invalid.
     */
    public static Length parse(String value) {
        int i = 0;
        while (i < value.length() && (Character.isDigit(value.charAt(i)) || value.charAt(i) == '.' || value.charAt(i) == '-'))
            ++i;
        double val = Double.parseDouble(value.substring(0, i));
        if (val == 0)
            return ZERO;
        return new Length(val, Units.get(value.substring(i, value.length()).trim()));
    }

    /**
     * Creates a string representation with an associated suitable metric unit, "m", "cm" or "mm".
     */
    @Override
    public String toString()
    {
        double metre = to(Units.METRE);
        return metre >= 1    ? String.valueOf(metre     ) + "m" :
               metre >= 0.01 ? String.valueOf(metre*100 ) + "cm" :
                               String.valueOf(metre*1000) + "mm";
    }

    /**
     * Creates a string representation with the specified unit.
     */
    public String toString(Units unit)
    {
        return String.valueOf(to(unit)) + unit.getSymbol();
    }

    /**
     * Creates an array of {@link Length} by parsing a String representation of lengths with units.
     * @param value A group of value-unit pairs of the form "&lt;value1>&lt;unit1> &lt;value2>&lt;unit2>...". Example: "12.3cm 23.9mm  0.25in" etc. Allowed units are "um", "mm", "cm", "m", "km", "pt" and "in".
     * @param size Number of value-unit pairs represented by the {@code value} parameter.
     * @throws IllegalArgumentException if parameter {@code size} doesn't fit the number of values in {@code value}.
     * @throws IllegalArgumentException if a unit is invalid.
     */
    public static Length[] parseArray(String value, int size)
    {
        Length[] arr = parseArray(value);
        if (arr.length != size)
            throw new IllegalArgumentException("value must contain exactly " + size + " components");
        return arr;
    }

    /**
     * Creates an array of {@link Length} by parsing a String representation of lengths with units.
     * @param value A group of value-unit pairs of the form "&lt;value1>&lt;unit1> &lt;value2>&lt;unit2>...". Example: "12.3cm 23.9mm  0.25in" etc. Allowed units are "um", "mm", "cm", "m", "km", "pt" and "in".
     * @throws IllegalArgumentException if a unit is invalid.
     */
    public static Length[] parseArray(String value)
    {
        ArrayList<Length> lengthArray = new ArrayList<Length>();
        for (String string : value.trim().split("\\s+")) {
            lengthArray.add(parse(string));
        }
        Length[] arr = new Length[lengthArray.size()];
        return lengthArray.toArray(arr);
    }

    public Length add(Length addend) {
        return new Length(points + addend.points);
    }

    public Length subtract(Length subtrahend) {
        return new Length(points - subtrahend.points);
    }

    public Length multiply(double multiplicand) {
        return new Length(points * multiplicand);
    }

    public Length divide(double divisor) {
        return new Length(points / divisor);
    }

    public double divide(Length divisor) {
        return points / divisor.points;
    }

    public static Length min(Length lhs, Length rhs) {
        return new Length(Math.min(lhs.points, rhs.points));
    }

    public static Length max(Length lhs, Length rhs) {
        return new Length(Math.max(lhs.points, rhs.points));
    }

    public static Length abs(Length length) {
        return new Length(Math.abs(length.points));
    }

    @Override
    public boolean equals(Object obj) {
        if (obj != null && obj instanceof Length) 
        {
            return ((Length) obj).points == points;
        }

        return false;
    }

    @Override
    public int hashCode() {
        return Objects.hash(points);
    }

    @Override
    public int compareTo(Length o) {
        return Double.compare(points, o.points);
    }
}