/****************************************************************************
 *
 * File:            Point.java
 *
 * Description:     Point class
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.geometry.units;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;

import com.pdftools.geometry.units.*;
import com.pdftools.geometry.units.Length.Units;

/**
 * <p>Class that represents a point that is based on a X-coordinate and a Y-coordinate of type {@link Length}. </p>
 * <p>Creates a {@link Point} object by parsing a string representation and creates a string representation of a {@link Point} object.</p>
 */
public class Point
{
    private double x;
    private double y;

    /**
     * @hidden
     */
    public Point(double x, double y) {
        this.x = x;
        this.y = y;
    }

    /**
     * @hidden
     */
    public double getXValue() {
        return this.x;
    }

    /**
     * @hidden
     */
    public double getYValue() {
        return this.y;
    }

    /**
     * Construct {@link Point} object from x and y.
     * @param x as {@link Length} type
     * @param y as {@link Length} type
     */
    public Point(Length x, Length y) {
        this(x.getValue(), y.getValue());
    }

    /**
     * Create a {@link Point} object from the string representation of a x-y-coordinate pair.
     * @param value Allowed values are value-unit pairs of the form "&lt;x_value>&lt;x_unit> &lt;y_value>&lt;y_unit>". Allowed associated units are "um", "mm", "cm", "m", "km", "pt" and "in". Example: "12.3cm 23.9mm".
     * @return
     */
    public static Point parse(String value) {
        Length[] lengths = Length.parseArray(value, 2);
        return new Point(lengths[0], lengths[1]);
    }

    /**
     * Creates a string representation as x-y-pair with associated suitable metric units, "m", "cm" or "mm".
     */
    @Override
    public String toString() {
        return this.getX().toString() + " " + this.getY().toString();
    }

    /**
     * Creates a string representation as x-y-pair with the specified unit.
     */
    public String toString(Units unit) {
        return this.getX().toString(unit) + " " + this.getY().toString(unit);
    }

    /**
     * <p>Gets the horizontal coordinate as {@link Length} object.</p>
     * <p>The horizontal axis is oriented from left to right.</p>
     */
    public Length getX() {
        return new Length(x, Units.POINT);
    }

    /**
     * <p>Gets the vertical coordinate as {@link Length} object.</p>
     * <p>The vertical axis is oriented from bottom to top.</p>
     */
    public Length getY() {
        return new Length(y, Units.POINT);
    }

    @Override
    public boolean equals(Object obj)
    {
        if (obj != null && obj instanceof Point)
        {
            if (((Point) obj).x != x)
                return false;
            if (((Point) obj).y != y)
                return false;

            return true;
        }

        return false;
    }

    @Override
    public int hashCode() {
        return Objects.hash(x, y);
    }
}