/****************************************************************************
 *
 * File:            Resolution.java
 *
 * Description:     Resolution Struct
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.geometry.units;

import java.util.Objects;

/**
 * The resolution defines the spatial dot density, e.g. of images.
 */
public class Resolution
{
    public Resolution(double xDpi, double yDpi)
    {
        this.xDpi = xDpi;
        this.yDpi = yDpi;
    }

    public Resolution(double dpi)
    {
        this.xDpi = dpi;
        this.yDpi = dpi;
    }

    /**
     * Gets {@link Resolution#xDpi }
     */
    public double getXDpi()
    {
        return this.xDpi;
    }

    /**
     * Sets {@link Resolution#xDpi }
     */
    public void setXDpi(double xDpi)
    {
        this.xDpi = xDpi;
    }

    /**
     * Gets {@link Resolution#yDpi }
     */
    public double getYDpi()
    {
        return this.yDpi;
    }

    /**
     * Sets {@link Resolution#yDpi }
     */
    public void setYDpi(double yDpi)
    {
        this.yDpi = yDpi;
    }

    /**
     * <h1>Horizontal resolution in DPI (dots per inch)</h1>
     */
    public double xDpi;
    /**
     * <h1>Vertical resolution in DPI (dots per inch)</h1>
     */
    public double yDpi;

    /**
     * Calculate the physical size of an object, e.g. an image, at that resolution.
     * @param pixels The size of the image in pixels
     * @return The physical size
     */
    public Size calculateSize(com.pdftools.geometry.integer.Size pixels) {
        return new Size(pixels.getWidth() / xDpi * 72, pixels.getHeight() / yDpi * 72, Length.Units.POINT);
    }

    @Override
    public boolean equals(Object obj)
    {
        if (obj != null && obj instanceof Resolution)
        {
            if (((Resolution) obj).xDpi != xDpi)
                return false;
            if (((Resolution) obj).yDpi != yDpi)
                return false;

            return true;
        }

        return false;
    }

    @Override
    public int hashCode() {
        return Objects.hash(xDpi, yDpi);
    }

    @Override
    public String toString() {
        return String.format("%fDPI %fDPI", xDpi, yDpi);
    }
}
