/****************************************************************************
 *
 * File:            Size.java
 *
 * Description:     Size Class
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.geometry.units;

import java.util.Collections;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.lang.Math;

import com.pdftools.geometry.units.Length.Units;

/**
 * <p>Class that represents a size that is based on a width and height of type {@link Length}.</p>
 * <p>Constructs from standard paper sizes, generates a string representation, and provides conversions to portrait and landscape mode.</p>
 */
public class Size
{
    public static enum PaperSizes {
        /**
         * Paper size with dimensions <code>0.297m x 0.420m</code>, which is part of the A series format paper size defined by the ISO 216 standard. 
         */
        A3("A3"),
        /**
         * Paper size with dimensions <code>0.210m x 0.297</code>, which is part of the A series format paper size defined by the ISO 216 standard. 
         */
        A4("A4"),
        /**
         * Paper size with dimensions <code>0.148m x 0.210m</code>, which is part of the A series format paper size defined by the ISO 216 standard. 
         */
        A5("A5"),
        /**
         * Paper size with dimensions <code>8.5in x 11in</code>, which is mainly used in North America.
         */
        LETTER("Letter"),
        /**
         * Paper size with dimensions <code>8.5in x 14in</code>, which is mainly used in North America.
         */
        LEGAL("Legal");

        private String paperSize;

        private PaperSizes(String paperSize) {
            this.paperSize = paperSize;
        }

        /**
         * Get the paper size's name.
         */
        public String getName() {
            return paperSize;
        }

        @Override
        public String toString() {
            return getName();
        }

        private static final Map<String, PaperSizes> names;
        static {
            Map<String,PaperSizes> map = new ConcurrentHashMap<String, PaperSizes>();
            for (PaperSizes instance : PaperSizes.values()) {
                map.put(instance.getName(), instance);
            }
            names = Collections.unmodifiableMap(map);
        }

        /**
         * Get the paper size enum value from its name, e.g. A4, or Letter.
         * @param name the name, e.g. A4, or Letter.
         * @return
         */
        public static PaperSizes get(String name) {
            PaperSizes ps = names.get(name);
            if (ps == null)
                throw new IllegalArgumentException(String.format("Invalid paper size name %s.", name));
            return ps;
        }
    }    

    private double width;
    private double height;

    /**
     * @hidden
     */
    public Size(double width, double height) {
        this.width = width;
        this.height = height;
    }

    /**
     * @hidden
     */
    public double getWidthValue() {
        return this.width;
    }

    /**
     * @hidden
     */
    public double getHeightValue() {
        return this.height;
    }

    /**
     * Construct {@link Size} object from width and height.
     * @param width as {@link Length} type
     * @param height as {@link Length} type
     */
    public Size(Length width, Length height) {
        this(width.getValue(), height.getValue());
    }

    /**
     * Construct {@link Size} object from width, height and a common unit.
     * @param width as numerical value
     * @param height as numerical value
     * @param unit the unit
     */
    public Size(double width, double height, Units unit) {
        this(new Length(width, unit), new Length(height, unit));
    }

    /**
     * Construct a {@link Size} object from standard paper sizes defined in enum {@link PaperSizes}.
     * @param paperSize enumeration item e.g. {@link PaperSizes#A4}, or {@link PaperSizes#LETTER}.
     */
    public Size(PaperSizes paperSize) {
        Size size = getStandardSize(paperSize);
        this.width = size.width;
        this.height = size.height;
    }

    private static Size getStandardSize(PaperSizes paperSize) {
        switch (paperSize) {
            case A3:
                return new Size(new Length(0.297, Units.METRE), new Length(0.420, Units.METRE));

            case A4:
                return new Size(new Length(0.210, Units.METRE), new Length(0.297, Units.METRE));

            case A5:
                return new Size(new Length(0.148, Units.METRE), new Length(0.210, Units.METRE));

            case LETTER:
                return new Size(new Length(8.5, Units.INCH), new Length(11, Units.INCH));

            case LEGAL:
                return new Size(new Length(8.5, Units.INCH), new Length(14, Units.INCH));

            default:
                throw new IllegalArgumentException("PaperSize " + paperSize.toString() + " not supported.");
        }
    }

    /**
     * <p>Create a {@link Size} object from a string representation.</p>
     * 
     * <p>Allowed are value-unit pairs of the form "&lt;width_value>&lt;width_unit> &lt;height_value>&lt;height_unit>" or the name of a standard paper size.</p>
     * 
     * <p>Examples: "12.3cm 23.9mm" or "A4"</p>
     * 
     * @param value the string value
     * @return
     */
    public static Size parse(String value) {
        try
        {
            return new Size(PaperSizes.get(value));
        }
        catch (IllegalArgumentException ex)
        {
            Length[] lengths = Length.parseArray(value, 2);
            return new Size(lengths[0], lengths[1]);
        }
    }

    /**
     * Creates a string representation as width-height-pair with associated suitable metric units, "m", "cm" or "mm".
     */
    @Override
    public String toString() {
        return this.getWidth().toString() + " " + this.getHeight().toString();
    }

    /**
     * Creates a string representation as width-height-pair with the specified unit.
     */
    public String toString(Units unit) {
        return this.getWidth().toString(unit) + " " + this.getHeight().toString(unit);
    }

    /**
     * Switches width and height if the absolute value of the width is greater than the absolute value of the height.
     */
    public Size toPortrait() {
        if (Math.abs(this.width) > Math.abs(this.height))
            return new Size(this.height, this.width);
        else
            return this;
    }

    /**
     * Switches width and height if the absolute value of height is greater than the the absolute value of width.
     */
    public Size toLandscape() {
        if (Math.abs(this.height) > Math.abs(this.width))
            return new Size(this.height, this.width);
        else
            return this;
    }

    /**
     * Calculate ratio between height and width.
     */
    public double getAspectRatio() {
        return Math.abs(this.height) / Math.abs(this.width);
    }

    /**
     * Gets the width (horizontal size) as {@link Length} object.
     */
    public Length getWidth() {
        return new Length(width, Units.POINT);
    }

    /**
     * Gets the height (vertical size) as {@link Length} object.
     */
    public Length getHeight() {
        return new Length(height, Units.POINT);
    }

    @Override
    public boolean equals(Object obj)
    {
        if (obj != null && obj instanceof Size)
        {
            if (((Size) obj).width != width)
                return false;
            if (((Size) obj).height != height)
                return false;

            return true;
        }

        return false;
    }

    @Override
    public int hashCode() {
        return Objects.hash(width, height);
    }
}