/****************************************************************************
 *
 * File:            Document.java
 *
 * Description:     PDFTOOLS Document Class
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.image;

import com.pdftools.sys.*;
import com.pdftools.internal.*;
import java.util.EnumSet;
import java.time.OffsetDateTime;
/**
 * <h1>The base class for image documents</h1>
 * Image documents are either opened using {@link Document#open } or the result of an operation, e.g. of PDF to image conversion using {@link com.pdftools.pdf2image.Converter#convertPage pdftools.pdf2image.Converter.convertPage}.
 */
public abstract class Document extends NativeObject implements AutoCloseable 
{
    protected Document(long handle) 
    {
        super(handle);
    }

    /**
     * @hidden
     */
    public static Document createDynamicObject(long handle)
    {
        int type = getType(handle);
        switch (type)
        {
            case 1:
                return com.pdftools.image.SinglePageDocument.createDynamicObject(handle);
            case 2:
                return com.pdftools.image.MultiPageDocument.createDynamicObject(handle);
            default:
                return null;
        }
    }



    /**
     * <h1>Open an image document</h1>
     * Documents opened with this method are read-only and cannot be modified.
     * @param stream 
     * The stream from which the image is read.
     * @return 
     * The newly created document instance
     *
     * @throws com.pdftools.LicenseException The license check has failed.
     * @throws com.pdftools.UnknownFormatException The document has a not recognized image format.
     * @throws com.pdftools.CorruptException The document is corrupt or not an image.
     * @throws IllegalArgumentException if {@code stream} is {@code null}
     */
    public static com.pdftools.image.Document open(com.pdftools.sys.Stream stream) 
        throws 
            com.pdftools.LicenseException,
            com.pdftools.UnknownFormatException,

            com.pdftools.CorruptException
    {
        if (stream == null)
            throw new IllegalArgumentException("Argument 'stream' must not be null.", new NullPointerException("'stream'"));

        long retHandle = openNative(stream);

        if (retHandle == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                case 12: throw new com.pdftools.LicenseException(getLastErrorMessage());
                case 15: throw new com.pdftools.UnknownFormatException(getLastErrorMessage());
                case 16: throw new com.pdftools.CorruptException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }

        return com.pdftools.image.Document.createDynamicObject(retHandle);
    }




    private static native int getType(long handle);
    private static native long openNative(com.pdftools.sys.Stream stream);

    /**
     * <h1>Close the object.</h1>
     * Release all resources associated with the object.
     * @throws com.pdftools.PdfToolsException only explicitly stated in a superclass
     */
    public void close() 
        throws com.pdftools.PdfToolsException, 
               java.io.IOException
    {
        try
        {
            if (!close(getHandle()))
                throwLastError();
        }
        finally
        {
            setHandle(0);
        }
    }

    private native boolean close(long hObject);
}

