/****************************************************************************
 *
 * File:            Archive.java
 *
 * Description:     PDFTOOLS Archive Class
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.image2pdf.profiles;

import com.pdftools.sys.*;
import com.pdftools.internal.*;
import java.util.EnumSet;
import java.time.OffsetDateTime;
/**
 * <h1>The profile for image to PDF/A conversion for archiving</h1>
 * This profile is suitable for archiving images as PDFs.
 */
public class Archive extends com.pdftools.image2pdf.profiles.Profile 
{
    protected Archive(long handle) 
    {
        super(handle);
    }

    /**
     * @hidden
     */
    public static Archive createDynamicObject(long handle)
    {
        return new Archive(handle);
    }


    /**
     * 
     */
    public Archive()
    {
        this(newHelper());
    }

    private static long newHelper()
    {
        long handle = newNative();
        if (handle == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }

        return handle;
    }



     /**
     * <h1>The PDF/A conformance of the output document (Getter)</h1>
     * <p>
     * The supported PDF/A conformance are:
     * <ul>
     * <li>
     * "PDF/A-1b"</li>
     * <li>
     * "PDF/A-1a"</li>
     * <li>
     * "PDF/A-2b"</li>
     * <li>
     * "PDF/A-2u"</li>
     * <li>
     * "PDF/A-2a"</li>
     * <li>
     * "PDF/A-3b"</li>
     * <li>
     * "PDF/A-3u"</li>
     * <li>
     * "PDF/A-3a"</li>
     * </ul>
     * With level A conformances (PDF/A-1a, PDF/A-2a, PDF/A-3a),
     * the properties {@link Archive#getAlternateText }
     * and {@link Archive#getLanguage } must be set.</p>
     * <p>
     * Default value: "PDF/A-2b"
     * </p>
     */
    public com.pdftools.pdf.Conformance getConformance()
    {
        int retVal = getConformanceNative(getHandle());
        if (retVal == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return new com.pdftools.pdf.Conformance(retVal);

    }

     /**
     * <h1>The PDF/A conformance of the output document (Setter)</h1>
     * <p>
     * The supported PDF/A conformance are:
     * <ul>
     * <li>
     * "PDF/A-1b"</li>
     * <li>
     * "PDF/A-1a"</li>
     * <li>
     * "PDF/A-2b"</li>
     * <li>
     * "PDF/A-2u"</li>
     * <li>
     * "PDF/A-2a"</li>
     * <li>
     * "PDF/A-3b"</li>
     * <li>
     * "PDF/A-3u"</li>
     * <li>
     * "PDF/A-3a"</li>
     * </ul>
     * With level A conformances (PDF/A-1a, PDF/A-2a, PDF/A-3a),
     * the properties {@link Archive#getAlternateText }
     * and {@link Archive#getLanguage } must be set.</p>
     * <p>
     * Default value: "PDF/A-2b"
     * </p>
     *
     * @throws IllegalArgumentException The conformance is PDF but must be PDF/A for this profile.
     *         Use the profile {@link Default } to create PDF documents.
     * @throws IllegalArgumentException if {@code value} is {@code null}
     */
    public void setConformance(com.pdftools.pdf.Conformance value)
    {
        if (value == null)
            throw new IllegalArgumentException("Argument 'value' must not be null.", new NullPointerException("'value'"));
        boolean retVal = setConformanceNative(getHandle(), value.getValue());
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                case 3: throw new IllegalArgumentException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }
    }


     /**
     * <h1>The alternate text describing the image (Getter)</h1>
     * <p>
     * The alternate text provides a meaningful description of the images.
     * For example, "This color image shows a small sailing boat at sunset".
     * This information can be used to read the document to the visually impaired.</p>
     * <p>
     * The list must contain a description for each page of the input image document.
     * For the conversion of {@link com.pdftools.image.SinglePageDocument pdftools.image.SinglePageDocument}, a single description
     * is sufficient. For {@link com.pdftools.image.MultiPageDocument pdftools.image.MultiPageDocument}, multiple descriptions may be
     * required.</p>
     * <p>
     * Alternate text is required for PDF/A level A conformance.
     * It is not advisable to add "dummy" tagging solely for the purpose of achieving level A
     * conformance. Instead, for scanned text documents, the Conversion Service can be used to
     * recognize the characters in the documents (OCR) and tag the image with the recognized structure and text.
     * For other types of images, such as photos, logos or graphics, alternate text descriptions
     * should be written manually by a user.</p>
     * <p>
     * Default: empty list
     * </p>
     */
    public com.pdftools.StringList getAlternateText()
    {
        long retHandle = getAlternateTextNative(getHandle());
        if (retHandle == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
        return com.pdftools.StringList.createDynamicObject(retHandle);

    }

     /**
     * <h1>The alternate text describing the image (Setter)</h1>
     * <p>
     * The alternate text provides a meaningful description of the images.
     * For example, "This color image shows a small sailing boat at sunset".
     * This information can be used to read the document to the visually impaired.</p>
     * <p>
     * The list must contain a description for each page of the input image document.
     * For the conversion of {@link com.pdftools.image.SinglePageDocument pdftools.image.SinglePageDocument}, a single description
     * is sufficient. For {@link com.pdftools.image.MultiPageDocument pdftools.image.MultiPageDocument}, multiple descriptions may be
     * required.</p>
     * <p>
     * Alternate text is required for PDF/A level A conformance.
     * It is not advisable to add "dummy" tagging solely for the purpose of achieving level A
     * conformance. Instead, for scanned text documents, the Conversion Service can be used to
     * recognize the characters in the documents (OCR) and tag the image with the recognized structure and text.
     * For other types of images, such as photos, logos or graphics, alternate text descriptions
     * should be written manually by a user.</p>
     * <p>
     * Default: empty list
     * </p>
     * @throws IllegalArgumentException if {@code value} is {@code null}
     */
    public void setAlternateText(com.pdftools.StringList value)
    {
        if (value == null)
            throw new IllegalArgumentException("Argument 'value' must not be null.", new NullPointerException("'value'"));
        boolean retVal = setAlternateTextNative(getHandle(), getHandle(value), value);
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
    }


     /**
     * <h1>The language of the output PDF (Getter)</h1>
     * <p>
     * The language code that specifies the language of the PDF and specifically
     * its {@link Archive#getAlternateText } descriptions.
     * Specifying the language is highly recommended for PDF/A level A conformance.</p>
     * <p>
     * The codes are defined in BCP 47 and ISO 3166:2013 and can
     * be obtained from the Internet Engineering Task Force and
     * the International Organization for Standardization.</p>
     * <p>
     * If no code is set, the language will be specified as
     * unknown.</p>
     * <p>
     * Examples:
     * <ul>
     * <li>
     * "en"</li>
     * <li>
     * "en-US"</li>
     * <li>
     * "de"</li>
     * <li>
     * "de-CH"</li>
     * <li>
     * "fr-FR"</li>
     * <li>
     * "zxx" (for non linguistic content)</li>
     * </ul></p>
     * <p>
     * Default: {@code null} (unknown)
     * </p>
     */
    public String getLanguage()
    {
        String retVal = getLanguageNative(getHandle());
        if (retVal == null)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }

     /**
     * <h1>The language of the output PDF (Setter)</h1>
     * <p>
     * The language code that specifies the language of the PDF and specifically
     * its {@link Archive#getAlternateText } descriptions.
     * Specifying the language is highly recommended for PDF/A level A conformance.</p>
     * <p>
     * The codes are defined in BCP 47 and ISO 3166:2013 and can
     * be obtained from the Internet Engineering Task Force and
     * the International Organization for Standardization.</p>
     * <p>
     * If no code is set, the language will be specified as
     * unknown.</p>
     * <p>
     * Examples:
     * <ul>
     * <li>
     * "en"</li>
     * <li>
     * "en-US"</li>
     * <li>
     * "de"</li>
     * <li>
     * "de-CH"</li>
     * <li>
     * "fr-FR"</li>
     * <li>
     * "zxx" (for non linguistic content)</li>
     * </ul></p>
     * <p>
     * Default: {@code null} (unknown)
     * </p>
     */
    public void setLanguage(String value)
    {
        boolean retVal = setLanguageNative(getHandle(), value);
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
    }




    private static native long newNative();
    private native int getConformanceNative(long handle);
    private native boolean setConformanceNative(long handle, int value);
    private native long getAlternateTextNative(long handle);
    private native boolean setAlternateTextNative(long handle, long value, com.pdftools.StringList valueObj);
    private native String getLanguageNative(long handle);
    private native boolean setLanguageNative(long handle, String value);

}

