/****************************************************************************
 *
 * File:            Default.java
 *
 * Description:     PDFTOOLS Default Class
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.image2pdf.profiles;

import com.pdftools.sys.*;
import com.pdftools.internal.*;
import java.util.EnumSet;
import java.time.OffsetDateTime;
/**
 * <h1>The default profile for image to PDF conversion</h1>
 * This profile is suitable for the conversion of input images to PDF.
 */
public class Default extends com.pdftools.image2pdf.profiles.Profile 
{
    protected Default(long handle) 
    {
        super(handle);
    }

    /**
     * @hidden
     */
    public static Default createDynamicObject(long handle)
    {
        return new Default(handle);
    }


    /**
     * 
     */
    public Default()
    {
        this(newHelper());
    }

    private static long newHelper()
    {
        long handle = newNative();
        if (handle == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }

        return handle;
    }



     /**
     * <h1>The PDF conformance of the output document (Getter)</h1>
     * <p>
     * All PDF conformances are supported.
     * For PDF/A the {@link Archive } profile must be used.</p>
     * <p>
     * Default value: "PDF 1.7"
     * </p>
     */
    public com.pdftools.pdf.Conformance getConformance()
    {
        int retVal = getConformanceNative(getHandle());
        if (retVal == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return new com.pdftools.pdf.Conformance(retVal);

    }

     /**
     * <h1>The PDF conformance of the output document (Setter)</h1>
     * <p>
     * All PDF conformances are supported.
     * For PDF/A the {@link Archive } profile must be used.</p>
     * <p>
     * Default value: "PDF 1.7"
     * </p>
     *
     * @throws IllegalArgumentException The conformance is PDF/A but must be PDF for this profile.
     *         Use the profile {@link Archive } to create PDF/A documents.
     * @throws IllegalArgumentException if {@code value} is {@code null}
     */
    public void setConformance(com.pdftools.pdf.Conformance value)
    {
        if (value == null)
            throw new IllegalArgumentException("Argument 'value' must not be null.", new NullPointerException("'value'"));
        boolean retVal = setConformanceNative(getHandle(), value.getValue());
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                case 3: throw new IllegalArgumentException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }
    }




    private static native long newNative();
    private native int getConformanceNative(long handle);
    private native boolean setConformanceNative(long handle, int value);

}

