/****************************************************************************
 *
 * File:            NativeBase.java
 *
 * Description:     Internal Native Base abstract Class
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.internal;

import com.pdftools.internal.Flag;
import com.pdftools.*;
import java.util.EnumSet;

public abstract class NativeBase
{
    protected NativeBase()
    { }

    protected static void throwLastRuntimeException()
    {
        throwLastRuntimeException(true);
    }

    protected static void throwLastRuntimeException(boolean allowSuccess)
    {
        switch (getLastErrorCode())
        {
            case 1: throw new UnsupportedOperationException(getLastErrorMessage());
            case 2: throw new IllegalStateException(getLastErrorMessage());
            case 3: throw new IllegalArgumentException(getLastErrorMessage());

            case 0:  if (allowSuccess) return;
            case 10: throw new RuntimeException(new com.pdftools.GenericException(getLastErrorMessage()));
            case 12: throw new RuntimeException(new com.pdftools.LicenseException(getLastErrorMessage()));
            case 15: throw new RuntimeException(new com.pdftools.UnknownFormatException(getLastErrorMessage()));
            case 16: throw new RuntimeException(new com.pdftools.CorruptException(getLastErrorMessage()));
            case 17: throw new RuntimeException(new com.pdftools.PasswordException(getLastErrorMessage()));
            case 18: throw new RuntimeException(new com.pdftools.ConformanceException(getLastErrorMessage()));
            case 19: throw new RuntimeException(new com.pdftools.UnsupportedFeatureException(getLastErrorMessage()));
            case 21: throw new RuntimeException(new com.pdftools.ProcessingException(getLastErrorMessage()));
            case 22: throw new RuntimeException(new com.pdftools.ExistsException(getLastErrorMessage()));
            case 23: throw new RuntimeException(new com.pdftools.PermissionException(getLastErrorMessage()));
            case 24: throw new RuntimeException(new com.pdftools.HttpException(getLastErrorMessage()));
            case 25: throw new RuntimeException(new com.pdftools.RetryException(getLastErrorMessage()));
            case 5: throw new RuntimeException(new com.pdftools.NotFoundException(getLastErrorMessage()));
            case 4: throw new RuntimeException(new java.io.IOException(getLastErrorMessage()));
            default: throw new RuntimeException(getLastErrorMessage());
        }
    }

    protected static void throwLastError(boolean allowSuccess) throws PdfToolsException, java.io.IOException
    {
        switch (getLastErrorCode())
        {
            case 1: throw new UnsupportedOperationException(getLastErrorMessage());
            case 2: throw new IllegalStateException(getLastErrorMessage());
            case 3: throw new IllegalArgumentException(getLastErrorMessage());

            case 0:  if (allowSuccess) return; else throw new RuntimeException("An unexpected error occurred");
            case 10: throw new com.pdftools.GenericException(getLastErrorMessage());
            case 12: throw new com.pdftools.LicenseException(getLastErrorMessage());
            case 15: throw new com.pdftools.UnknownFormatException(getLastErrorMessage());
            case 16: throw new com.pdftools.CorruptException(getLastErrorMessage());
            case 17: throw new com.pdftools.PasswordException(getLastErrorMessage());
            case 18: throw new com.pdftools.ConformanceException(getLastErrorMessage());
            case 19: throw new com.pdftools.UnsupportedFeatureException(getLastErrorMessage());
            case 21: throw new com.pdftools.ProcessingException(getLastErrorMessage());
            case 22: throw new com.pdftools.ExistsException(getLastErrorMessage());
            case 23: throw new com.pdftools.PermissionException(getLastErrorMessage());
            case 24: throw new com.pdftools.HttpException(getLastErrorMessage());
            case 25: throw new com.pdftools.RetryException(getLastErrorMessage());
            case 5: throw new com.pdftools.NotFoundException(getLastErrorMessage());
            case 4: throw new java.io.IOException(getLastErrorMessage());
            default: return;
        }
    }

    protected static Exception createError(int errorCode, String errorMessage, boolean allowSuccess)
    {
        switch (errorCode)
        {
            case 1: return new UnsupportedOperationException(errorMessage);
            case 2: return new IllegalStateException(errorMessage);
            case 3: return new IllegalArgumentException(errorMessage);

            case 0:  if (allowSuccess) return null; else return new RuntimeException("An unexpected error occurred");
            case 10: return new com.pdftools.GenericException(errorMessage);
            case 12: return new com.pdftools.LicenseException(errorMessage);
            case 15: return new com.pdftools.UnknownFormatException(errorMessage);
            case 16: return new com.pdftools.CorruptException(errorMessage);
            case 17: return new com.pdftools.PasswordException(errorMessage);
            case 18: return new com.pdftools.ConformanceException(errorMessage);
            case 19: return new com.pdftools.UnsupportedFeatureException(errorMessage);
            case 21: return new com.pdftools.ProcessingException(errorMessage);
            case 22: return new com.pdftools.ExistsException(errorMessage);
            case 23: return new com.pdftools.PermissionException(errorMessage);
            case 24: return new com.pdftools.HttpException(errorMessage);
            case 25: return new com.pdftools.RetryException(errorMessage);
            case 5: return new com.pdftools.NotFoundException(errorMessage);
            case 4: return new java.io.IOException(errorMessage);
            default: return null;
        }
    }

    protected static void throwLastError() throws PdfToolsException, java.io.IOException
    {
        throwLastError(true);
    }

    protected static <E extends Enum<E> & Flag> int getFlags(EnumSet<E> eset)
    {
        int flags = 0;
        for (E e : eset)
            flags |= e.getFlag();
        return flags;
    }

    protected static <E extends Enum<E> & Flag> EnumSet<E> getEnumSet(int flags, Class<E> clazz)
    {
        EnumSet<E> eset = EnumSet.noneOf(clazz);
        for(E x : clazz.getEnumConstants())
        {
            if ((x.getFlag() & flags) != 0)
                eset.add(x);
        }
        return eset;
    }

    protected static long getHandle(NativeObject object)
    {
        if (object == null)
            return 0;

        return object.getHandle();
    }

    protected static void addRef(NativeObject object)
    {
        if (object == null)
            return;

        object.addRef();
    }

    protected long getHandle() throws Exception
    {
        throw new Exception("Not implemented.");
    }

    private static void addShutdownHook() {
        Runtime.getRuntime().addShutdownHook(new Thread() {
            public void run() {
                uninitialize();
            }
        });
    }

    static
    {
        try
        {
            initialize();
            addShutdownHook();
        } 
        catch (UnsatisfiedLinkError ue) 
        {
            System.loadLibrary("PdfToolsSdk");
            initialize();
            addShutdownHook();
        }
    }

    protected static native int getLastErrorCode();
    protected static native String getLastErrorMessage();
    protected static native void setLastErrorNative(int errorCode, String errorMessage);
    private static native void initialize();
    private static native void uninitialize();
}
