/****************************************************************************
 *
 * File:            FontOptions.java
 *
 * Description:     PDFTOOLS FontOptions Class
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.optimization;

import com.pdftools.sys.*;
import com.pdftools.internal.*;
import java.util.EnumSet;
import java.time.OffsetDateTime;
/**
 * <h1>The parameters for font optimization</h1>
 */
public class FontOptions extends NativeObject 
{
    protected FontOptions(long handle) 
    {
        super(handle);
    }

    /**
     * @hidden
     */
    public static FontOptions createDynamicObject(long handle)
    {
        return new FontOptions(handle);
    }




     /**
     * <h1>Whether to merge fonts and font programs (Getter)</h1>
     * <p>
     * A PDF document can have the same font, or a subset of it,
     * embedded multiple times.
     * This commonly occurs, when multiple PDFs are merged into
     * one large PDF.
     * Such fonts can be merged into one font.</p>
     * <p>
     * Merging fonts and font programs can greatly reduce the file size.
     * However, it is computationally complex and can increase file processing time
     * and memory usage substantially.</p>
     * <p>
     * Default: {@code true}.
     * </p>
     */
    public boolean getMerge()
    {
        boolean retVal = getMergeNative(getHandle());
        if (!retVal)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }

     /**
     * <h1>Whether to merge fonts and font programs (Setter)</h1>
     * <p>
     * A PDF document can have the same font, or a subset of it,
     * embedded multiple times.
     * This commonly occurs, when multiple PDFs are merged into
     * one large PDF.
     * Such fonts can be merged into one font.</p>
     * <p>
     * Merging fonts and font programs can greatly reduce the file size.
     * However, it is computationally complex and can increase file processing time
     * and memory usage substantially.</p>
     * <p>
     * Default: {@code true}.
     * </p>
     */
    public void setMerge(boolean value)
    {
        boolean retVal = setMergeNative(getHandle(), value);
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
    }


     /**
     * <h1>Whether to remove standard fonts (Getter)</h1>
     * <p>
     * Enable or disable un-embedding of the font programs of all embedded
     * standard fonts, such as Arial, Courier, CourierNew, Helvetica, Symbol,
     * Times, TimesNewRoman and ZapfDingbats.
     * This decreases the file size.</p>
     * <p>
     * The fonts are replaced with one of the 14 PDF Standard Fonts, all of
     * which have no associated font program.
     * A PDF viewer must be able to display these 14 PDF Standard Fonts correctly.
     * Therefore, enabling this property usually does not visually alter the PDF
     * when it is displayed.</p>
     * <p>
     * Un-embedding the font works based on the font's Unicode information,
     * i.e. the un-embedded font's characters are mapped to those of the
     * original font with the same Unicode.
     * Therefore, only fonts with Unicode information are un-embedded.</p>
     * <p>
     * If a font's Unicode information is incorrect, un-embedding may lead
     * to visual differences.
     * The correctness of a Unicode information can be verified by extracting
     * text that uses the font.</p>
     * <p>
     * If the extracted text is meaningful, the font's Unicode information is
     * correct, and un-embedding of the font does not cause visual differences.</p>
     * <p>
     * Default: {@code false} (disabled) except in the profile {@link com.pdftools.optimization.profiles.MinimalFileSize profiles.MinimalFileSize}.
     * </p>
     */
    public boolean getRemoveStandardFonts()
    {
        boolean retVal = getRemoveStandardFontsNative(getHandle());
        if (!retVal)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }

     /**
     * <h1>Whether to remove standard fonts (Setter)</h1>
     * <p>
     * Enable or disable un-embedding of the font programs of all embedded
     * standard fonts, such as Arial, Courier, CourierNew, Helvetica, Symbol,
     * Times, TimesNewRoman and ZapfDingbats.
     * This decreases the file size.</p>
     * <p>
     * The fonts are replaced with one of the 14 PDF Standard Fonts, all of
     * which have no associated font program.
     * A PDF viewer must be able to display these 14 PDF Standard Fonts correctly.
     * Therefore, enabling this property usually does not visually alter the PDF
     * when it is displayed.</p>
     * <p>
     * Un-embedding the font works based on the font's Unicode information,
     * i.e. the un-embedded font's characters are mapped to those of the
     * original font with the same Unicode.
     * Therefore, only fonts with Unicode information are un-embedded.</p>
     * <p>
     * If a font's Unicode information is incorrect, un-embedding may lead
     * to visual differences.
     * The correctness of a Unicode information can be verified by extracting
     * text that uses the font.</p>
     * <p>
     * If the extracted text is meaningful, the font's Unicode information is
     * correct, and un-embedding of the font does not cause visual differences.</p>
     * <p>
     * Default: {@code false} (disabled) except in the profile {@link com.pdftools.optimization.profiles.MinimalFileSize profiles.MinimalFileSize}.
     * </p>
     */
    public void setRemoveStandardFonts(boolean value)
    {
        boolean retVal = setRemoveStandardFontsNative(getHandle(), value);
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
    }




    private native boolean getMergeNative(long handle);
    private native boolean setMergeNative(long handle, boolean value);
    private native boolean getRemoveStandardFontsNative(long handle);
    private native boolean setRemoveStandardFontsNative(long handle, boolean value);

}

