/****************************************************************************
 *
 * File:            ImageRecompressionOptions.java
 *
 * Description:     PDFTOOLS ImageRecompressionOptions Class
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.optimization;

import com.pdftools.sys.*;
import com.pdftools.internal.*;
import java.util.EnumSet;
import java.time.OffsetDateTime;
/**
 * <h1>The parameters for image recompression</h1>
 */
public class ImageRecompressionOptions extends NativeObject 
{
    protected ImageRecompressionOptions(long handle) 
    {
        super(handle);
    }

    /**
     * @hidden
     */
    public static ImageRecompressionOptions createDynamicObject(long handle)
    {
        return new ImageRecompressionOptions(handle);
    }




     /**
     * <h1>The strategy for image recompression (Getter)</h1>
     * <p>
     * For each image to be recompressed, a specific choice of compression
     * algorithms are tried. The selection of algorithms depends on this strategy, the
     * type of the optimizer profile (e.g. {@link com.pdftools.optimization.profiles.Web profiles.Web}),
     * the color space of the image, and {@link ImageRecompressionOptions#getCompressionQuality }.
     * The image is recompressed using the algorithm resulting in the
     * smallest output file.</p>
     * <p>
     * Refer to {@link CompressionAlgorithmSelection } for
     * more information on strategies.</p>
     * <p>
     * Default:
     * <ul>
     * <li>
     * {@link com.pdftools.optimization.profiles.Web profiles.Web} profile: {@link CompressionAlgorithmSelection#BALANCED }</li>
     * <li>
     * {@link com.pdftools.optimization.profiles.Print profiles.Print} profile: {@link CompressionAlgorithmSelection#PRESERVE_QUALITY }</li>
     * <li>
     * {@link com.pdftools.optimization.profiles.Archive profiles.Archive} profile: {@link CompressionAlgorithmSelection#PRESERVE_QUALITY }</li>
     * <li>
     * {@link com.pdftools.optimization.profiles.MinimalFileSize profiles.MinimalFileSize} profile: {@link CompressionAlgorithmSelection#BALANCED }</li>
     * <li>
     * {@link com.pdftools.optimization.profiles.Mrc profiles.Mrc} profile: {@link CompressionAlgorithmSelection#BALANCED }</li>
     * </ul></p>
     */
    public com.pdftools.optimization.CompressionAlgorithmSelection getAlgorithmSelection()
    {
        int retVal = getAlgorithmSelectionNative(getHandle());
        if (retVal == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return com.pdftools.optimization.CompressionAlgorithmSelection.fromValue(retVal);
    }

     /**
     * <h1>The strategy for image recompression (Setter)</h1>
     * <p>
     * For each image to be recompressed, a specific choice of compression
     * algorithms are tried. The selection of algorithms depends on this strategy, the
     * type of the optimizer profile (e.g. {@link com.pdftools.optimization.profiles.Web profiles.Web}),
     * the color space of the image, and {@link ImageRecompressionOptions#getCompressionQuality }.
     * The image is recompressed using the algorithm resulting in the
     * smallest output file.</p>
     * <p>
     * Refer to {@link CompressionAlgorithmSelection } for
     * more information on strategies.</p>
     * <p>
     * Default:
     * <ul>
     * <li>
     * {@link com.pdftools.optimization.profiles.Web profiles.Web} profile: {@link CompressionAlgorithmSelection#BALANCED }</li>
     * <li>
     * {@link com.pdftools.optimization.profiles.Print profiles.Print} profile: {@link CompressionAlgorithmSelection#PRESERVE_QUALITY }</li>
     * <li>
     * {@link com.pdftools.optimization.profiles.Archive profiles.Archive} profile: {@link CompressionAlgorithmSelection#PRESERVE_QUALITY }</li>
     * <li>
     * {@link com.pdftools.optimization.profiles.MinimalFileSize profiles.MinimalFileSize} profile: {@link CompressionAlgorithmSelection#BALANCED }</li>
     * <li>
     * {@link com.pdftools.optimization.profiles.Mrc profiles.Mrc} profile: {@link CompressionAlgorithmSelection#BALANCED }</li>
     * </ul></p>
     * @throws IllegalArgumentException if {@code value} is {@code null}
     */
    public void setAlgorithmSelection(com.pdftools.optimization.CompressionAlgorithmSelection value)
    {
        if (value == null)
            throw new IllegalArgumentException("Argument 'value' must not be null.", new NullPointerException("'value'"));
        boolean retVal = setAlgorithmSelectionNative(getHandle(), value.getValue());
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
    }


     /**
     * <h1>The compression quality for lossy image compression algorithms (Getter)</h1>
     * <p>
     * This property specifies the compression quality for the JPEG and JPEG2000 image compression algorithms.
     * Valid values are between 0 (lowest quality) and 1 (highest quality).</p>
     * <p>
     * Although the JBIG2 algorithm for bi-tonal images also allows lossy compression, it is not influenced by this property.
     * The JBIG2 compression quality is fixed at 1 (lossless).</p>
     * <p>
     * Default:
     * <ul>
     * <li>
     * {@link com.pdftools.optimization.profiles.Web profiles.Web} profile: 0.8</li>
     * <li>
     * {@link com.pdftools.optimization.profiles.Print profiles.Print} profile: 0.9</li>
     * <li>
     * {@link com.pdftools.optimization.profiles.Archive profiles.Archive} profile: 0.9</li>
     * <li>
     * {@link com.pdftools.optimization.profiles.MinimalFileSize profiles.MinimalFileSize} profile: 0.75</li>
     * <li>
     * {@link com.pdftools.optimization.profiles.Mrc profiles.Mrc} profile: 0.75</li>
     * </ul></p>
     */
    public double getCompressionQuality()
    {
        double retVal = getCompressionQualityNative(getHandle());
        if (retVal == -1.0)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }

     /**
     * <h1>The compression quality for lossy image compression algorithms (Setter)</h1>
     * <p>
     * This property specifies the compression quality for the JPEG and JPEG2000 image compression algorithms.
     * Valid values are between 0 (lowest quality) and 1 (highest quality).</p>
     * <p>
     * Although the JBIG2 algorithm for bi-tonal images also allows lossy compression, it is not influenced by this property.
     * The JBIG2 compression quality is fixed at 1 (lossless).</p>
     * <p>
     * Default:
     * <ul>
     * <li>
     * {@link com.pdftools.optimization.profiles.Web profiles.Web} profile: 0.8</li>
     * <li>
     * {@link com.pdftools.optimization.profiles.Print profiles.Print} profile: 0.9</li>
     * <li>
     * {@link com.pdftools.optimization.profiles.Archive profiles.Archive} profile: 0.9</li>
     * <li>
     * {@link com.pdftools.optimization.profiles.MinimalFileSize profiles.MinimalFileSize} profile: 0.75</li>
     * <li>
     * {@link com.pdftools.optimization.profiles.Mrc profiles.Mrc} profile: 0.75</li>
     * </ul></p>
     *
     * @throws IllegalArgumentException If the given value is outside of the range 0 - 1
     */
    public void setCompressionQuality(double value)
    {
        boolean retVal = setCompressionQualityNative(getHandle(), value);
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                case 3: throw new IllegalArgumentException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }
    }


     /**
     * <h1>Enable color complexity reduction. (Getter)</h1>
     * When enabled, the software analyzes images that utilize device color spaces (DeviceRGB, DeviceCMYK, or DeviceGray) as well as indexed images with a device color space as their base. If applicable, the images are converted according to the following criteria:
     * <ul>
     * <li>
     * Images in DeviceRGB or DeviceCMYK color space where all pixels are gray will be converted to grayscale using the DeviceGray color space.</li>
     * <li>
     * Images containing only black and white pixels will be converted to bitonal images.</li>
     * <li>
     * Images where all pixels are of the same color will be downsampled to a single pixel.</li>
     * </ul>
     * Additionally, image masks and soft masks are optimized in the following ways:
     * <ul>
     * <li>
     * Soft masks consisting only of black and white pixels will be converted into a standard mask.</li>
     * <li>
     * Any (soft) mask that is completely opaque will be removed.</li>
     * </ul>
     * Default:
     * <ul>
     * <li>
     * {@link com.pdftools.optimization.profiles.Web profiles.Web} profile: {@code true}</li>
     * <li>
     * {@link com.pdftools.optimization.profiles.Print profiles.Print} profile: {@code true}</li>
     * <li>
     * {@link com.pdftools.optimization.profiles.Archive profiles.Archive} profile: {@code true}</li>
     * <li>
     * {@link com.pdftools.optimization.profiles.MinimalFileSize profiles.MinimalFileSize} profile: {@code true}</li>
     * <li>
     * {@link com.pdftools.optimization.profiles.Mrc profiles.Mrc} profile: {@code false}</li>
     * </ul>
     */
    public boolean getReduceColorComplexity()
    {
        boolean retVal = getReduceColorComplexityNative(getHandle());
        if (!retVal)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }

     /**
     * <h1>Enable color complexity reduction. (Setter)</h1>
     * When enabled, the software analyzes images that utilize device color spaces (DeviceRGB, DeviceCMYK, or DeviceGray) as well as indexed images with a device color space as their base. If applicable, the images are converted according to the following criteria:
     * <ul>
     * <li>
     * Images in DeviceRGB or DeviceCMYK color space where all pixels are gray will be converted to grayscale using the DeviceGray color space.</li>
     * <li>
     * Images containing only black and white pixels will be converted to bitonal images.</li>
     * <li>
     * Images where all pixels are of the same color will be downsampled to a single pixel.</li>
     * </ul>
     * Additionally, image masks and soft masks are optimized in the following ways:
     * <ul>
     * <li>
     * Soft masks consisting only of black and white pixels will be converted into a standard mask.</li>
     * <li>
     * Any (soft) mask that is completely opaque will be removed.</li>
     * </ul>
     * Default:
     * <ul>
     * <li>
     * {@link com.pdftools.optimization.profiles.Web profiles.Web} profile: {@code true}</li>
     * <li>
     * {@link com.pdftools.optimization.profiles.Print profiles.Print} profile: {@code true}</li>
     * <li>
     * {@link com.pdftools.optimization.profiles.Archive profiles.Archive} profile: {@code true}</li>
     * <li>
     * {@link com.pdftools.optimization.profiles.MinimalFileSize profiles.MinimalFileSize} profile: {@code true}</li>
     * <li>
     * {@link com.pdftools.optimization.profiles.Mrc profiles.Mrc} profile: {@code false}</li>
     * </ul>
     */
    public void setReduceColorComplexity(boolean value)
    {
        boolean retVal = setReduceColorComplexityNative(getHandle(), value);
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
    }




    private native int getAlgorithmSelectionNative(long handle);
    private native boolean setAlgorithmSelectionNative(long handle, int value);
    private native double getCompressionQualityNative(long handle);
    private native boolean setCompressionQualityNative(long handle, double value);
    private native boolean getReduceColorComplexityNative(long handle);
    private native boolean setReduceColorComplexityNative(long handle, boolean value);

}

