/****************************************************************************
 *
 * File:            Optimizer.java
 *
 * Description:     PDFTOOLS Optimizer Class
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.optimization;

import com.pdftools.sys.*;
import com.pdftools.internal.*;
import java.util.EnumSet;
import java.time.OffsetDateTime;
/**
 * <h1>The class to optimize PDF documents</h1>
 */
public class Optimizer extends NativeObject 
{
    protected Optimizer(long handle) 
    {
        super(handle);
    }

    /**
     * @hidden
     */
    public static Optimizer createDynamicObject(long handle)
    {
        return new Optimizer(handle);
    }


    /**
     * 
     */
    public Optimizer()
    {
        this(newHelper());
    }

    private static long newHelper()
    {
        long handle = newNative();
        if (handle == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }

        return handle;
    }


    /**
     * <h1>Optimize the PDF document</h1>
     * @param inDoc 
     * The input PDF document
     * @param outStream 
     * The stream to which the output PDF is written
     * @param profile 
     * The profile defining the optimization parameters.
     * @return 
     * <p>
     * The optimized result PDF, which can be used
     * as a new input for further processing.</p>
     * <p>
     * Note that, this object must be disposed before the output stream
     * object (method argument {@link com.pdftools.optimization.Optimizer#optimizeDocument outStream}).
     * </p>
     *
     * @throws com.pdftools.LicenseException The license check has failed.
     * @throws IllegalArgumentException An invalid encryption was specified in {@link com.pdftools.optimization.Optimizer#optimizeDocument outOptions}.
     * @throws com.pdftools.ProcessingException The processing has failed.
     * @throws java.io.IOException Writing to the output PDF has failed.
     * @throws com.pdftools.GenericException A generic error occurred.
     * @throws IllegalArgumentException if {@code inDoc} is {@code null}
     * @throws IllegalArgumentException if {@code outStream} is {@code null}
     * @throws IllegalArgumentException if {@code profile} is {@code null}
     */
    public com.pdftools.pdf.Document optimizeDocument(com.pdftools.pdf.Document inDoc, com.pdftools.sys.Stream outStream, com.pdftools.optimization.profiles.Profile profile) 
        throws 
            java.io.IOException,
            com.pdftools.GenericException,
            com.pdftools.LicenseException,

            com.pdftools.ProcessingException
    {
        if (inDoc == null)
            throw new IllegalArgumentException("Argument 'inDoc' must not be null.", new NullPointerException("'inDoc'"));
        if (outStream == null)
            throw new IllegalArgumentException("Argument 'outStream' must not be null.", new NullPointerException("'outStream'"));
        if (profile == null)
            throw new IllegalArgumentException("Argument 'profile' must not be null.", new NullPointerException("'profile'"));

        return optimizeDocument(inDoc, outStream, profile, null);
    }

    /**
     * <h1>Optimize the PDF document</h1>
     * @param inDoc 
     * The input PDF document
     * @param outStream 
     * The stream to which the output PDF is written
     * @param profile 
     * The profile defining the optimization parameters.
     * @param outOptions 
     * The PDF output options, e.g. to encrypt the output document.
     * @return 
     * <p>
     * The optimized result PDF, which can be used
     * as a new input for further processing.</p>
     * <p>
     * Note that, this object must be disposed before the output stream
     * object (method argument {@link com.pdftools.optimization.Optimizer#optimizeDocument outStream}).
     * </p>
     *
     * @throws com.pdftools.LicenseException The license check has failed.
     * @throws IllegalArgumentException An invalid encryption was specified in {@link com.pdftools.optimization.Optimizer#optimizeDocument outOptions}.
     * @throws com.pdftools.ProcessingException The processing has failed.
     * @throws java.io.IOException Writing to the output PDF has failed.
     * @throws com.pdftools.GenericException A generic error occurred.
     * @throws IllegalArgumentException if {@code inDoc} is {@code null}
     * @throws IllegalArgumentException if {@code outStream} is {@code null}
     * @throws IllegalArgumentException if {@code profile} is {@code null}
     */
    public com.pdftools.pdf.Document optimizeDocument(com.pdftools.pdf.Document inDoc, com.pdftools.sys.Stream outStream, com.pdftools.optimization.profiles.Profile profile, com.pdftools.pdf.OutputOptions outOptions) 
        throws 
            java.io.IOException,
            com.pdftools.GenericException,
            com.pdftools.LicenseException,

            com.pdftools.ProcessingException
    {
        if (inDoc == null)
            throw new IllegalArgumentException("Argument 'inDoc' must not be null.", new NullPointerException("'inDoc'"));
        if (outStream == null)
            throw new IllegalArgumentException("Argument 'outStream' must not be null.", new NullPointerException("'outStream'"));
        if (profile == null)
            throw new IllegalArgumentException("Argument 'profile' must not be null.", new NullPointerException("'profile'"));

        long retHandle = optimizeDocumentNative(getHandle(), getHandle(inDoc), inDoc, outStream, getHandle(profile), profile, getHandle(outOptions), outOptions);

        if (retHandle == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                case 3: throw new IllegalArgumentException(getLastErrorMessage());
                case 4: throw new java.io.IOException(getLastErrorMessage());
                case 10: throw new com.pdftools.GenericException(getLastErrorMessage());
                case 12: throw new com.pdftools.LicenseException(getLastErrorMessage());
                case 21: throw new com.pdftools.ProcessingException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }

        return com.pdftools.pdf.Document.createDynamicObject(retHandle);
    }




    private static native long newNative();
    private native long optimizeDocumentNative(long handle, long inDoc, com.pdftools.pdf.Document inDocObj, com.pdftools.sys.Stream outStream, long profile, com.pdftools.optimization.profiles.Profile profileObj, long outOptions, com.pdftools.pdf.OutputOptions outOptionsObj);

}

