/****************************************************************************
 *
 * File:            RemovalOptions.java
 *
 * Description:     PDFTOOLS RemovalOptions Class
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.optimization;

import com.pdftools.sys.*;
import com.pdftools.internal.*;
import java.util.EnumSet;
import java.time.OffsetDateTime;
/**
 * <h1>The parameters defining the optional data to remove or flatten</h1>
 * <p>
 * Removal options specify the PDF data structures to copy or remove,
 * e.g. article threads, metadata, or alternate images.</p>
 * <p>
 * In addition, the visual appearances of signatures, annotations, form fields,
 * and links can be flattened.</p>
 * <p>
 * Flattening means, that the appearance of such a data structure is drawn as
 * non-editable graphic onto the page; for visual appearances of signatures,
 * flattening has a slightly different meaning
 * (see property {@link RemovalOptions#getRemoveSignatureAppearances }).
 * </p>
 */
public class RemovalOptions extends NativeObject 
{
    protected RemovalOptions(long handle) 
    {
        super(handle);
    }

    /**
     * @hidden
     */
    public static RemovalOptions createDynamicObject(long handle)
    {
        return new RemovalOptions(handle);
    }




     /**
     * <h1>Whether to remove additional or alternative versions of images (Getter)</h1>
     * Default: {@code false} except in the profile {@link com.pdftools.optimization.profiles.Print profiles.Print}.
     */
    public boolean getRemoveAlternateImages()
    {
        boolean retVal = getRemoveAlternateImagesNative(getHandle());
        if (!retVal)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }

     /**
     * <h1>Whether to remove additional or alternative versions of images (Setter)</h1>
     * Default: {@code false} except in the profile {@link com.pdftools.optimization.profiles.Print profiles.Print}.
     */
    public void setRemoveAlternateImages(boolean value)
    {
        boolean retVal = setRemoveAlternateImagesNative(getHandle(), value);
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
    }


     /**
     * <h1>Whether to remove the sequential flows (threads) of articles (Getter)</h1>
     * Default: {@code true} except in the profile {@link com.pdftools.optimization.profiles.Archive profiles.Archive}.
     */
    public boolean getRemoveArticleThreads()
    {
        boolean retVal = getRemoveArticleThreadsNative(getHandle());
        if (!retVal)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }

     /**
     * <h1>Whether to remove the sequential flows (threads) of articles (Setter)</h1>
     * Default: {@code true} except in the profile {@link com.pdftools.optimization.profiles.Archive profiles.Archive}.
     */
    public void setRemoveArticleThreads(boolean value)
    {
        boolean retVal = setRemoveArticleThreadsNative(getHandle(), value);
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
    }


     /**
     * <h1>Whether to remove document's XMP metadata (Getter)</h1>
     * Default:
     * <ul>
     * <li>
     * {@link com.pdftools.optimization.profiles.Web profiles.Web} profile: {@code true}</li>
     * <li>
     * {@link com.pdftools.optimization.profiles.Print profiles.Print} profile: {@code false}</li>
     * <li>
     * {@link com.pdftools.optimization.profiles.Archive profiles.Archive} profile: {@code false}</li>
     * <li>
     * {@link com.pdftools.optimization.profiles.MinimalFileSize profiles.MinimalFileSize} profile: {@code true}</li>
     * <li>
     * {@link com.pdftools.optimization.profiles.Mrc profiles.Mrc} profile: {@code false}</li>
     * </ul>
     */
    public boolean getRemoveMetadata()
    {
        boolean retVal = getRemoveMetadataNative(getHandle());
        if (!retVal)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }

     /**
     * <h1>Whether to remove document's XMP metadata (Setter)</h1>
     * Default:
     * <ul>
     * <li>
     * {@link com.pdftools.optimization.profiles.Web profiles.Web} profile: {@code true}</li>
     * <li>
     * {@link com.pdftools.optimization.profiles.Print profiles.Print} profile: {@code false}</li>
     * <li>
     * {@link com.pdftools.optimization.profiles.Archive profiles.Archive} profile: {@code false}</li>
     * <li>
     * {@link com.pdftools.optimization.profiles.MinimalFileSize profiles.MinimalFileSize} profile: {@code true}</li>
     * <li>
     * {@link com.pdftools.optimization.profiles.Mrc profiles.Mrc} profile: {@code false}</li>
     * </ul>
     */
    public void setRemoveMetadata(boolean value)
    {
        boolean retVal = setRemoveMetadataNative(getHandle(), value);
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
    }


     /**
     * <h1>Whether to remove all output intents (Getter)</h1>
     * <p>
     * Output intents provide a means for matching the color characteristics
     * of PDF page content with those of a target output device or
     * production environment in which the document will be printed.</p>
     * <p>
     * Default: {@code false} except in the profile {@link com.pdftools.optimization.profiles.MinimalFileSize profiles.MinimalFileSize}.
     * </p>
     */
    public boolean getRemoveOutputIntents()
    {
        boolean retVal = getRemoveOutputIntentsNative(getHandle());
        if (!retVal)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }

     /**
     * <h1>Whether to remove all output intents (Setter)</h1>
     * <p>
     * Output intents provide a means for matching the color characteristics
     * of PDF page content with those of a target output device or
     * production environment in which the document will be printed.</p>
     * <p>
     * Default: {@code false} except in the profile {@link com.pdftools.optimization.profiles.MinimalFileSize profiles.MinimalFileSize}.
     * </p>
     */
    public void setRemoveOutputIntents(boolean value)
    {
        boolean retVal = setRemoveOutputIntentsNative(getHandle(), value);
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
    }


     /**
     * <h1>Whether to remove the piece-info dictionary (private PDF processor data) (Getter)</h1>
     * <p>
     * The removal of this proprietary application data has no effect on the document's
     * visual appearance.</p>
     * <p>
     * Default: {@code true} except in the profile {@link com.pdftools.optimization.profiles.Archive profiles.Archive}.
     * </p>
     */
    public boolean getRemovePieceInfo()
    {
        boolean retVal = getRemovePieceInfoNative(getHandle());
        if (!retVal)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }

     /**
     * <h1>Whether to remove the piece-info dictionary (private PDF processor data) (Setter)</h1>
     * <p>
     * The removal of this proprietary application data has no effect on the document's
     * visual appearance.</p>
     * <p>
     * Default: {@code true} except in the profile {@link com.pdftools.optimization.profiles.Archive profiles.Archive}.
     * </p>
     */
    public void setRemovePieceInfo(boolean value)
    {
        boolean retVal = setRemovePieceInfoNative(getHandle(), value);
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
    }


     /**
     * <h1>Whether to remove the data describing the logical structure of a PDF (Getter)</h1>
     * <p>
     * The logical structure of the document is a description of the content of its pages.
     * It consists of a fine granular hierarchical tagging that distinguishes between the actual content and artifacts (such as page numbers, layout artifacts, etc.).
     * The tagging provides a meaningful description, for example "This is a header", "This color image shows a small sailing boat at sunset", etc.
     * This information can be used e.g. to read the document to the visually impaired.</p>
     * <p>
     * Default: {@code true} except in the profile {@link com.pdftools.optimization.profiles.Archive profiles.Archive}.
     * </p>
     */
    public boolean getRemoveStructureTree()
    {
        boolean retVal = getRemoveStructureTreeNative(getHandle());
        if (!retVal)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }

     /**
     * <h1>Whether to remove the data describing the logical structure of a PDF (Setter)</h1>
     * <p>
     * The logical structure of the document is a description of the content of its pages.
     * It consists of a fine granular hierarchical tagging that distinguishes between the actual content and artifacts (such as page numbers, layout artifacts, etc.).
     * The tagging provides a meaningful description, for example "This is a header", "This color image shows a small sailing boat at sunset", etc.
     * This information can be used e.g. to read the document to the visually impaired.</p>
     * <p>
     * Default: {@code true} except in the profile {@link com.pdftools.optimization.profiles.Archive profiles.Archive}.
     * </p>
     */
    public void setRemoveStructureTree(boolean value)
    {
        boolean retVal = setRemoveStructureTreeNative(getHandle(), value);
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
    }


     /**
     * <h1>Whether to remove thumbnail images which represent the PDF pages in miniature form (Getter)</h1>
     * Default: {@code true} in all profiles.
     */
    public boolean getRemoveThumbnails()
    {
        boolean retVal = getRemoveThumbnailsNative(getHandle());
        if (!retVal)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }

     /**
     * <h1>Whether to remove thumbnail images which represent the PDF pages in miniature form (Setter)</h1>
     * Default: {@code true} in all profiles.
     */
    public void setRemoveThumbnails(boolean value)
    {
        boolean retVal = setRemoveThumbnailsNative(getHandle(), value);
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
    }


     /**
     * <h1>Whether to remove or flatten signature appearances (Getter)</h1>
     * <p>
     * A signature in a PDF consist of two parts:</p>
     * <p>
     * <ul>
     * <li>
     * <em>(a)</em> The invisible digital signature in the PDF.</li>
     * <li>
     * <em>(b)</em> The visual appearance that was attributed to the signature.</li>
     * </ul></p>
     * <p>
     * Part (a) can be used by a viewing application, to verify that the PDF
     * has not changed since it has been signed and report this to the user.</p>
     * <p>
     * During optimizing, the PDF is altered and hence its digital signature
     * (a) is broken and must be removed.</p>
     * <p>
     * <ul>
     * <li>
     * {@link RemovalStrategy#FLATTEN }: (a) is removed and (b) is drawn as non-editable graphic onto the page.
     * Within the context of signatures this is called "flattening".</li>
     * <li>
     * {@link RemovalStrategy#REMOVE }: (a) and (b) are removed.</li>
     * </ul></p>
     * <p>
     * Default: {@link RemovalStrategy#FLATTEN } in all profiles.
     * </p>
     */
    public com.pdftools.optimization.RemovalStrategy getRemoveSignatureAppearances()
    {
        int retVal = getRemoveSignatureAppearancesNative(getHandle());
        if (retVal == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return com.pdftools.optimization.RemovalStrategy.fromValue(retVal);
    }

     /**
     * <h1>Whether to remove or flatten signature appearances (Setter)</h1>
     * <p>
     * A signature in a PDF consist of two parts:</p>
     * <p>
     * <ul>
     * <li>
     * <em>(a)</em> The invisible digital signature in the PDF.</li>
     * <li>
     * <em>(b)</em> The visual appearance that was attributed to the signature.</li>
     * </ul></p>
     * <p>
     * Part (a) can be used by a viewing application, to verify that the PDF
     * has not changed since it has been signed and report this to the user.</p>
     * <p>
     * During optimizing, the PDF is altered and hence its digital signature
     * (a) is broken and must be removed.</p>
     * <p>
     * <ul>
     * <li>
     * {@link RemovalStrategy#FLATTEN }: (a) is removed and (b) is drawn as non-editable graphic onto the page.
     * Within the context of signatures this is called "flattening".</li>
     * <li>
     * {@link RemovalStrategy#REMOVE }: (a) and (b) are removed.</li>
     * </ul></p>
     * <p>
     * Default: {@link RemovalStrategy#FLATTEN } in all profiles.
     * </p>
     * @throws IllegalArgumentException if {@code value} is {@code null}
     */
    public void setRemoveSignatureAppearances(com.pdftools.optimization.RemovalStrategy value)
    {
        if (value == null)
            throw new IllegalArgumentException("Argument 'value' must not be null.", new NullPointerException("'value'"));
        boolean retVal = setRemoveSignatureAppearancesNative(getHandle(), value.getValue());
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
    }


     /**
     * <h1>The conversion strategy for annotations (Getter)</h1>
     * <p>
     * The conversion strategy for annotations.</p>
     * <p>
     * Annotations in PDF are interactive elements on the pages, such as:
     * <ul>
     * <li>
     * Sticky notes</li>
     * <li>
     * Free text annotations</li>
     * <li>
     * Line, square, circle, and polygon annotations</li>
     * <li>
     * Highlight, underline and strikeout annotations</li>
     * <li>
     * Stamp annotations</li>
     * <li>
     * Ink annotations</li>
     * <li>
     * File attachment annotation</li>
     * <li>
     * Sound and movie annotations</li>
     * <li>
     * 3D annotations</li>
     * </ul></p>
     * <p>
     * Note that this does not include form fields (see {@link RemovalOptions#getFormFields }) and links (see {@link RemovalOptions#getLinks }).</p>
     * <p>
     * Default: {@link ConversionStrategy#COPY } in all profiles.
     * </p>
     */
    public com.pdftools.optimization.ConversionStrategy getAnnotations()
    {
        int retVal = getAnnotationsNative(getHandle());
        if (retVal == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return com.pdftools.optimization.ConversionStrategy.fromValue(retVal);
    }

     /**
     * <h1>The conversion strategy for annotations (Setter)</h1>
     * <p>
     * The conversion strategy for annotations.</p>
     * <p>
     * Annotations in PDF are interactive elements on the pages, such as:
     * <ul>
     * <li>
     * Sticky notes</li>
     * <li>
     * Free text annotations</li>
     * <li>
     * Line, square, circle, and polygon annotations</li>
     * <li>
     * Highlight, underline and strikeout annotations</li>
     * <li>
     * Stamp annotations</li>
     * <li>
     * Ink annotations</li>
     * <li>
     * File attachment annotation</li>
     * <li>
     * Sound and movie annotations</li>
     * <li>
     * 3D annotations</li>
     * </ul></p>
     * <p>
     * Note that this does not include form fields (see {@link RemovalOptions#getFormFields }) and links (see {@link RemovalOptions#getLinks }).</p>
     * <p>
     * Default: {@link ConversionStrategy#COPY } in all profiles.
     * </p>
     * @throws IllegalArgumentException if {@code value} is {@code null}
     */
    public void setAnnotations(com.pdftools.optimization.ConversionStrategy value)
    {
        if (value == null)
            throw new IllegalArgumentException("Argument 'value' must not be null.", new NullPointerException("'value'"));
        boolean retVal = setAnnotationsNative(getHandle(), value.getValue());
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
    }


     /**
     * <h1>The conversion strategy for interactive forms (Getter)</h1>
     * Default: {@link ConversionStrategy#COPY } in all profiles.
     */
    public com.pdftools.optimization.ConversionStrategy getFormFields()
    {
        int retVal = getFormFieldsNative(getHandle());
        if (retVal == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return com.pdftools.optimization.ConversionStrategy.fromValue(retVal);
    }

     /**
     * <h1>The conversion strategy for interactive forms (Setter)</h1>
     * Default: {@link ConversionStrategy#COPY } in all profiles.
     * @throws IllegalArgumentException if {@code value} is {@code null}
     */
    public void setFormFields(com.pdftools.optimization.ConversionStrategy value)
    {
        if (value == null)
            throw new IllegalArgumentException("Argument 'value' must not be null.", new NullPointerException("'value'"));
        boolean retVal = setFormFieldsNative(getHandle(), value.getValue());
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
    }


     /**
     * <h1>The conversion strategy for links (Getter)</h1>
     * Default: {@link ConversionStrategy#COPY } in all profiles.
     */
    public com.pdftools.optimization.ConversionStrategy getLinks()
    {
        int retVal = getLinksNative(getHandle());
        if (retVal == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return com.pdftools.optimization.ConversionStrategy.fromValue(retVal);
    }

     /**
     * <h1>The conversion strategy for links (Setter)</h1>
     * Default: {@link ConversionStrategy#COPY } in all profiles.
     * @throws IllegalArgumentException if {@code value} is {@code null}
     */
    public void setLinks(com.pdftools.optimization.ConversionStrategy value)
    {
        if (value == null)
            throw new IllegalArgumentException("Argument 'value' must not be null.", new NullPointerException("'value'"));
        boolean retVal = setLinksNative(getHandle(), value.getValue());
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
    }




    private native boolean getRemoveAlternateImagesNative(long handle);
    private native boolean setRemoveAlternateImagesNative(long handle, boolean value);
    private native boolean getRemoveArticleThreadsNative(long handle);
    private native boolean setRemoveArticleThreadsNative(long handle, boolean value);
    private native boolean getRemoveMetadataNative(long handle);
    private native boolean setRemoveMetadataNative(long handle, boolean value);
    private native boolean getRemoveOutputIntentsNative(long handle);
    private native boolean setRemoveOutputIntentsNative(long handle, boolean value);
    private native boolean getRemovePieceInfoNative(long handle);
    private native boolean setRemovePieceInfoNative(long handle, boolean value);
    private native boolean getRemoveStructureTreeNative(long handle);
    private native boolean setRemoveStructureTreeNative(long handle, boolean value);
    private native boolean getRemoveThumbnailsNative(long handle);
    private native boolean setRemoveThumbnailsNative(long handle, boolean value);
    private native int getRemoveSignatureAppearancesNative(long handle);
    private native boolean setRemoveSignatureAppearancesNative(long handle, int value);
    private native int getAnnotationsNative(long handle);
    private native boolean setAnnotationsNative(long handle, int value);
    private native int getFormFieldsNative(long handle);
    private native boolean setFormFieldsNative(long handle, int value);
    private native int getLinksNative(long handle);
    private native boolean setLinksNative(long handle, int value);

}

