/****************************************************************************
 *
 * File:            MinimalFileSize.java
 *
 * Description:     PDFTOOLS MinimalFileSize Class
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.optimization.profiles;

import com.pdftools.sys.*;
import com.pdftools.internal.*;
import java.util.EnumSet;
import java.time.OffsetDateTime;
/**
 * <h1>The optimization profile producing a minimal file size</h1>
 * <p>
 * This profile optimizes the output PDF for minimal file size.
 * This is achieved by using a varied palette of image compression
 * algorithms, appropriate resolution setting and higher
 * compression rates at the price of slightly lower image quality.</p>
 * <p>
 * The output file size is further reduced by converting Embedded
 * Type1 (PostScript) fonts to Type1C (Compact Font Format) and
 * removing metadata and output intents 
 * (see {@link Profile#getRemovalOptions }).
 * Also Spider (web capture) information is removed.</p>
 * <p>
 * Images above 182 DPI are down-sampled and recompressed to 130 DPI.
 * This leads to smaller output files. The property
 * {@link MinimalFileSize#setResolutionDPI } has influence on both values.</p>
 * <p>
 * When an image is recompressed, the
 * {@link com.pdftools.optimization.CompressionAlgorithmSelection#BALANCED pdftools.optimization.CompressionAlgorithmSelection.BALANCED}
 * strategy is used; this can be overridden through the
 * property {@link Profile#getImageRecompressionOptions }.</p>
 * <p>
 * With this profile, the output PDF version is updated to PDF 1.7 or higher and
 * PDF/A conformance removed.
 * </p>
 */
public class MinimalFileSize extends com.pdftools.optimization.profiles.Profile 
{
    protected MinimalFileSize(long handle) 
    {
        super(handle);
    }

    /**
     * @hidden
     */
    public static MinimalFileSize createDynamicObject(long handle)
    {
        return new MinimalFileSize(handle);
    }


    /**
     * 
     */
    public MinimalFileSize()
    {
        this(newHelper());
    }

    private static long newHelper()
    {
        long handle = newNative();
        if (handle == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }

        return handle;
    }



     /**
     * <h1>The target resolution of images in DPI (Getter)</h1>
     * <p>
     * The target resolution in DPI (dots per inch) for color and grayscale images.</p>
     * <p>
     * Images with a resolution above {@link MinimalFileSize#getThresholdDPI } are down-sampled.</p>
     * <p>
     * Valid values are in the range 1.0 to 10000.</p>
     * <p>
     * If {@code null}, then resolution setting is disabled.</p>
     * <p>
     * Default: {@code 130}.
     * </p>
     */
    public Double getResolutionDPI()
    {
        Double retVal = getResolutionDPINative(getHandle());
        if (retVal == null)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }

     /**
     * <h1>The target resolution of images in DPI (Setter)</h1>
     * <p>
     * The target resolution in DPI (dots per inch) for color and grayscale images.</p>
     * <p>
     * Images with a resolution above {@link MinimalFileSize#getThresholdDPI } are down-sampled.</p>
     * <p>
     * Valid values are in the range 1.0 to 10000.</p>
     * <p>
     * If {@code null}, then resolution setting is disabled.</p>
     * <p>
     * Default: {@code 130}.
     * </p>
     *
     * @throws IllegalArgumentException The given value is outside of range 1.0 to 10000.0.
     */
    public void setResolutionDPI(Double value)
    {
        boolean retVal = setResolutionDPINative(getHandle(), value);
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                case 3: throw new IllegalArgumentException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }
    }


     /**
     * <h1>The threshold resolution of images in DPI. (Getter)</h1>
     * <p>
     * The threshold resolution in DPI (dots per inch) to selectively activate downsampling for color and grayscale images.</p>
     * <p>
     * Valid values are in the range 1.0 to 10000.
     * To deactivate down-sampling of images set {@link MinimalFileSize#getResolutionDPI } to {@code null}.</p>
     * <p>
     * Default: {@code 1.4} times {@link MinimalFileSize#getResolutionDPI }.
     * </p>
     */
    public double getThresholdDPI()
    {
        double retVal = getThresholdDPINative(getHandle());
        if (retVal == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }

     /**
     * <h1>The threshold resolution of images in DPI. (Setter)</h1>
     * <p>
     * The threshold resolution in DPI (dots per inch) to selectively activate downsampling for color and grayscale images.</p>
     * <p>
     * Valid values are in the range 1.0 to 10000.
     * To deactivate down-sampling of images set {@link MinimalFileSize#getResolutionDPI } to {@code null}.</p>
     * <p>
     * Default: {@code 1.4} times {@link MinimalFileSize#getResolutionDPI }.
     * </p>
     *
     * @throws IllegalArgumentException The given value is outside of range 1.0 to 10000.0.
     */
    public void setThresholdDPI(double value)
    {
        boolean retVal = setThresholdDPINative(getHandle(), value);
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                case 3: throw new IllegalArgumentException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }
    }




    private static native long newNative();
    private native Double getResolutionDPINative(long handle);
    private native boolean setResolutionDPINative(long handle, Double value);
    private native double getThresholdDPINative(long handle);
    private native boolean setThresholdDPINative(long handle, double value);

}

