/****************************************************************************
 *
 * File:            Metadata.java
 *
 * Description:     PDFTOOLS Metadata Class
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.pdf;

import com.pdftools.sys.*;
import com.pdftools.internal.*;
import java.util.EnumSet;
import java.time.OffsetDateTime;
/**
 * 
 * <p>
 * Represents the metadata of a document or an object in a document.</p>
 * <p>
 * For document level metadata,
 * all changes are reflected in both,
 * XMP metadata and document info dictionary depending on the conformance
 * of the document.
 * </p>
 */
public class Metadata extends NativeObject 
{
    protected Metadata(long handle) 
    {
        super(handle);
    }

    /**
     * @hidden
     */
    public static Metadata createDynamicObject(long handle)
    {
        return new Metadata(handle);
    }




     /**
     * <h1>The title of the document or resource. (Getter)</h1>
     * This property corresponds to the "dc:title" entry
     * in the XMP metadata and to the "Title" entry in
     * the document information dictionary.
     *
     * @throws IllegalStateException if the metadata have already been closed
     */
    public String getTitle()
    {
        String retVal = getTitleNative(getHandle());
        if (retVal == null)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                case 2: throw new IllegalStateException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }


     /**
     * <h1>The name of the person who created the document or resource. (Getter)</h1>
     * This property corresponds to the "dc:creator" entry
     * in the XMP metadata and to the "Author" entry in
     * the document information dictionary.
     *
     * @throws IllegalStateException if the metadata have already been closed
     */
    public String getAuthor()
    {
        String retVal = getAuthorNative(getHandle());
        if (retVal == null)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                case 2: throw new IllegalStateException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }


     /**
     * <h1>The subject of the document or resource. (Getter)</h1>
     * This property corresponds to the "dc:description" entry
     * in the XMP metadata and to the "Subject" entry in
     * the document information dictionary.
     *
     * @throws IllegalStateException if the metadata have already been closed
     */
    public String getSubject()
    {
        String retVal = getSubjectNative(getHandle());
        if (retVal == null)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                case 2: throw new IllegalStateException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }


     /**
     * <h1>Keywords associated with the document or resource. (Getter)</h1>
     * <p>
     * Keywords can be separated by:
     * <ul>
     * <li>
     * carriage return / line feed</li>
     * <li>
     * comma</li>
     * <li>
     * semicolon</li>
     * <li>
     * tab</li>
     * <li>
     * double space</li>
     * </ul></p>
     * <p>
     * This property corresponds to the "pdf:Keywords" entry
     * in the XMP metadata and to the "Keywords" entry in
     * the document information dictionary.
     * </p>
     *
     * @throws IllegalStateException if the metadata have already been closed
     */
    public String getKeywords()
    {
        String retVal = getKeywordsNative(getHandle());
        if (retVal == null)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                case 2: throw new IllegalStateException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }


     /**
     * <h1>The original application that created the document. (Getter)</h1>
     * <p>
     * The name of the first known tool used to create the document or resource.</p>
     * <p>
     * This property corresponds to the "xmp:CreatorTool" entry
     * in the XMP metadata and to the "Creator" entry in
     * the document information dictionary.
     * </p>
     *
     * @throws IllegalStateException if the metadata have already been closed
     */
    public String getCreator()
    {
        String retVal = getCreatorNative(getHandle());
        if (retVal == null)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                case 2: throw new IllegalStateException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }


     /**
     * <h1>The application that created the PDF (Getter)</h1>
     * <p>
     * If the document was converted to PDF from another format,
     * the name of the PDF processor that converted it to PDF.</p>
     * <p>
     * This property corresponds to the "pdf:Producer" entry
     * in the XMP metadata and to the "Producer" entry in
     * the document information dictionary.
     * </p>
     *
     * @throws IllegalStateException if the metadata have already been closed
     */
    public String getProducer()
    {
        String retVal = getProducerNative(getHandle());
        if (retVal == null)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                case 2: throw new IllegalStateException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }


     /**
     * <h1>The date and time the document or resource was originally created. (Getter)</h1>
     * This property corresponds to the "xmp:CreateDate" entry
     * in the XMP metadata and to the "CreationDate" entry in
     * the document information dictionary.
     *
     * @throws IllegalStateException if the metadata have already been closed
     */
    public OffsetDateTime getCreationDate()
    {
        OffsetDateTime retVal = getCreationDateNative(getHandle());
        if (retVal == null)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                case 2: throw new IllegalStateException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }


     /**
     * <h1>The date and time the document or resource was most recently modified. (Getter)</h1>
     * This property corresponds to the "xmp:ModifyDate" entry
     * in the XMP metadata and to the "ModDate" entry in
     * the document information dictionary.
     *
     * @throws com.pdftools.CorruptException The date is corrupt.
     * @throws IllegalStateException if the metadata have already been closed
     */
    public OffsetDateTime getModificationDate()
        throws 
            com.pdftools.CorruptException
    {
        OffsetDateTime retVal = getModificationDateNative(getHandle());
        if (retVal == null)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                case 2: throw new IllegalStateException(getLastErrorMessage());
                case 16: throw new com.pdftools.CorruptException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }




    private native String getTitleNative(long handle);
    private native String getAuthorNative(long handle);
    private native String getSubjectNative(long handle);
    private native String getKeywordsNative(long handle);
    private native String getCreatorNative(long handle);
    private native String getProducerNative(long handle);
    private native OffsetDateTime getCreationDateNative(long handle);
    private native OffsetDateTime getModificationDateNative(long handle);

}

