/****************************************************************************
 *
 * File:            SignatureField.java
 *
 * Description:     PDFTOOLS SignatureField Class
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.pdf;

import com.pdftools.sys.*;
import com.pdftools.internal.*;
import java.util.EnumSet;
import java.time.OffsetDateTime;
/**
 * <h1>A digital signature field</h1>
 */
public abstract class SignatureField extends NativeObject 
{
    protected SignatureField(long handle) 
    {
        super(handle);
    }

    /**
     * @hidden
     */
    public static SignatureField createDynamicObject(long handle)
    {
        int type = getType(handle);
        switch (type)
        {
            case 1:
                return com.pdftools.pdf.UnsignedSignatureField.createDynamicObject(handle);
            case 4:
                return com.pdftools.pdf.DocumentSignature.createDynamicObject(handle);
            case 5:
                return com.pdftools.pdf.CertificationSignature.createDynamicObject(handle);
            case 6:
                return com.pdftools.pdf.DocumentTimestamp.createDynamicObject(handle);
            default:
                return null;
        }
    }




     /**
     * <h1>The name of the signature field (Getter)</h1>
     * The field name uniquely identifies the signature field within the document.
     *
     * @throws IllegalStateException If the object has already been closed
     */
    public String getFieldName()
    {
        String retVal = getFieldNameNative(getHandle());
        if (retVal == null)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                case 2: throw new IllegalStateException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }


     /**
     * <h1>The number of the page where this signature is located (Getter)</h1>
     * Whether the signature field has a visual appearance on that page is indicated by the {@link SignatureField#getBoundingBox }.
     *
     * @throws IllegalStateException If the object has already been closed.
     * @throws com.pdftools.NotFoundException If the field is not properly linked to a page.
     */
    public int getPageNumber()
        throws 
            com.pdftools.NotFoundException
    {
        int retVal = getPageNumberNative(getHandle());
        if (retVal == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                case 2: throw new IllegalStateException(getLastErrorMessage());
                case 5: throw new com.pdftools.NotFoundException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }


     /**
     * <h1>The location on the page (Getter)</h1>
     * The location of the signature field on the page.
     * Or {@code null} if the signature field has no visual appearance.
     *
     * @throws IllegalStateException If the object has already been closed
     */
    public com.pdftools.geometry.units.Rectangle getBoundingBox()
    {
        com.pdftools.geometry.units.Rectangle retVal = getBoundingBoxNative(getHandle());
        if (retVal == null)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                case 2: throw new IllegalStateException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }




    private static native int getType(long handle);
    private native String getFieldNameNative(long handle);
    private native int getPageNumberNative(long handle);
    private native com.pdftools.geometry.units.Rectangle getBoundingBoxNative(long handle);

}

