/****************************************************************************
 *
 * File:            SignedSignatureField.java
 *
 * Description:     PDFTOOLS SignedSignatureField Class
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.pdf;

import com.pdftools.sys.*;
import com.pdftools.internal.*;
import java.util.EnumSet;
import java.time.OffsetDateTime;
/**
 * <h1>A base class for signature fields that have been signed</h1>
 * The existence of a signed signature field does not imply that the signature is valid.
 * The signature is not validated at all.
 */
public abstract class SignedSignatureField extends com.pdftools.pdf.SignatureField 
{
    protected SignedSignatureField(long handle) 
    {
        super(handle);
    }

    /**
     * @hidden
     */
    public static SignedSignatureField createDynamicObject(long handle)
    {
        int type = getType(handle);
        switch (type)
        {
            case 2:
                return com.pdftools.pdf.DocumentSignature.createDynamicObject(handle);
            case 3:
                return com.pdftools.pdf.CertificationSignature.createDynamicObject(handle);
            case 4:
                return com.pdftools.pdf.DocumentTimestamp.createDynamicObject(handle);
            default:
                return null;
        }
    }




     /**
     * <h1>The name of the person or authority that signed the document (Getter)</h1>
     * <p>
     * This is the name of the signing certificate.</p>
     * <p>
     * <em>Note:</em> The name of the signing certificate can only be extracted for signatures conforming to the PAdES or PDF standard
     * and not for proprietary/non-standard signature formats.
     * For non-standard signature formats, the name as stored in the PDF is returned.
     * </p>
     *
     * @throws IllegalStateException If the object has already been closed
     */
    public String getName()
    {
        String retVal = getNameNative(getHandle());
        if (retVal == null)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                case 2: throw new IllegalStateException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }


     /**
     * <h1>The date and time of signing (Getter)</h1>
     * <p>
     * This represents the date and time of signing as specified in the signature.
     * For signatures that contain a time-stamp, the trusted time-stamp time is returned.</p>
     * <p>
     * <em>Note:</em> The value can only be extracted for signatures conforming to the PAdES or PDF standard
     * and not for proprietary/non-standard signature formats.
     * For non-standard signature formats, the date as stored in the PDF is returned.
     * </p>
     *
     * @throws IllegalStateException If the object has already been closed
     */
    public OffsetDateTime getDate()
    {
        OffsetDateTime retVal = getDateNative(getHandle());
        if (retVal == null)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                case 2: throw new IllegalStateException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }


     /**
     * <h1>The document revision (Getter)</h1>
     * The revision (version) of the document that the signature signs.
     *
     * @throws IllegalStateException If the object has already been closed
     * @throws com.pdftools.CorruptException If the signature specifies an invalid document revision
     */
    public com.pdftools.pdf.Revision getRevision()
        throws 
            com.pdftools.CorruptException
    {
        long retHandle = getRevisionNative(getHandle());
        if (retHandle == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                case 2: throw new IllegalStateException(getLastErrorMessage());
                case 16: throw new com.pdftools.CorruptException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }
        return com.pdftools.pdf.Revision.createDynamicObject(retHandle);

    }




    private static native int getType(long handle);
    private native String getNameNative(long handle);
    private native OffsetDateTime getDateNative(long handle);
    private native long getRevisionNative(long handle);

}

