/****************************************************************************
 *
 * File:            JpegImageOptions.java
 *
 * Description:     PDFTOOLS JpegImageOptions Class
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.pdf2image;

import com.pdftools.sys.*;
import com.pdftools.internal.*;
import java.util.EnumSet;
import java.time.OffsetDateTime;
/**
 * <h1>The settings for JPEG output images</h1>
 * <p>
 * JPEG images use a lossy compression algorithm that provides a high compression ratio.
 * It is best suited for photographs and content with little or no text.</p>
 * <p>
 * For the output file name, it is recommended to use the file extension ".jpg".
 * </p>
 */
public class JpegImageOptions extends com.pdftools.pdf2image.ImageOptions 
{
    protected JpegImageOptions(long handle) 
    {
        super(handle);
    }

    /**
     * @hidden
     */
    public static JpegImageOptions createDynamicObject(long handle)
    {
        return new JpegImageOptions(handle);
    }


    /**
     * 
     */
    public JpegImageOptions()
    {
        this(newHelper());
    }

    private static long newHelper()
    {
        long handle = newNative();
        if (handle == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }

        return handle;
    }



     /**
     * <h1>The color space of the output image (Getter)</h1>
     * <p>
     * Get or set the color space of the image.
     * If {@code null}, the blending color space of the page is used.</p>
     * <p>
     * Default: {@link JpegColorSpace#RGB }</p>
     */
    public com.pdftools.pdf2image.JpegColorSpace getColorSpace()
    {
        Integer retVal = getColorSpaceNative(getHandle());
        if (retVal == null)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
            return null;
        }

        return com.pdftools.pdf2image.JpegColorSpace.fromValue(retVal);
    }

     /**
     * <h1>The color space of the output image (Setter)</h1>
     * <p>
     * Get or set the color space of the image.
     * If {@code null}, the blending color space of the page is used.</p>
     * <p>
     * Default: {@link JpegColorSpace#RGB }</p>
     */
    public void setColorSpace(com.pdftools.pdf2image.JpegColorSpace value)
    {
        boolean retVal = setColorSpaceNative(getHandle(), value == null ? 0 : value.getValue(), value == null);
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
    }


     /**
     * <h1>The JPEG quality factor (Getter)</h1>
     * <p>
     * Get or set the JPEG compression quality.
     * Valid values are 1, or 100, or in between.</p>
     * <p>
     * Default: {@code 85}</p>
     */
    public int getJpegQuality()
    {
        int retVal = getJpegQualityNative(getHandle());
        if (retVal == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }

     /**
     * <h1>The JPEG quality factor (Setter)</h1>
     * <p>
     * Get or set the JPEG compression quality.
     * Valid values are 1, or 100, or in between.</p>
     * <p>
     * Default: {@code 85}</p>
     *
     * @throws IllegalArgumentException The given value is smaller than 1 or greater than 100.
     */
    public void setJpegQuality(int value)
    {
        boolean retVal = setJpegQualityNative(getHandle(), value);
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                case 3: throw new IllegalArgumentException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }
    }




    private static native long newNative();
    private native Integer getColorSpaceNative(long handle);
    private native boolean setColorSpaceNative(long handle, int value, boolean isNull);
    private native int getJpegQualityNative(long handle);
    private native boolean setJpegQualityNative(long handle, int value);

}

