/****************************************************************************
 *
 * File:            RenderPageToMaxImageSize.java
 *
 * Description:     PDFTOOLS RenderPageToMaxImageSize Class
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.pdf2image;

import com.pdftools.sys.*;
import com.pdftools.internal.*;
import java.util.EnumSet;
import java.time.OffsetDateTime;
/**
 * <h1>The image section mapping to render entire pages using a specific image pixel size</h1>
 * <p>
 * Render a PDF page and scale it, thereby preserving the aspect
 * ratio, to fit best on the target image size. The image size is
 * specified in number of pixels.</p>
 * <p>
 * For example, this mapping is suitable to create thumbnail images.
 * </p>
 */
public class RenderPageToMaxImageSize extends com.pdftools.pdf2image.ImageSectionMapping 
{
    protected RenderPageToMaxImageSize(long handle) 
    {
        super(handle);
    }

    /**
     * @hidden
     */
    public static RenderPageToMaxImageSize createDynamicObject(long handle)
    {
        return new RenderPageToMaxImageSize(handle);
    }


    /**
     * 
     * @param size 
     * The maximum size of the image in pixels.
     *
     * @throws IllegalArgumentException The dimensions of {@code size} are smaller than 1.
     * @throws IllegalArgumentException if {@code size} is {@code null}
     */
    public RenderPageToMaxImageSize(com.pdftools.geometry.integer.Size size)
    {
        this(newHelper(size));
    }

    private static long newHelper(com.pdftools.geometry.integer.Size size)
    {
        if (size == null)
            throw new IllegalArgumentException("Argument 'size' must not be null.", new NullPointerException("'size'"));

        long handle = newNative(size.getWidth(), size.getHeight());
        if (handle == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                case 3: throw new IllegalArgumentException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }

        return handle;
    }



     /**
     * <h1>The maximum size of the image in pixels (Getter)</h1>
     * <p>
     * Set or get the image size.</p>
     * <p>
     * The dimensions of {@code size} must be 1 or greater.
     * </p>
     */
    public com.pdftools.geometry.integer.Size getSize()
    {
        com.pdftools.geometry.integer.Size retVal = getSizeNative(getHandle());
        if (retVal == null)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }

     /**
     * <h1>The maximum size of the image in pixels (Setter)</h1>
     * <p>
     * Set or get the image size.</p>
     * <p>
     * The dimensions of {@code size} must be 1 or greater.
     * </p>
     *
     * @throws IllegalArgumentException The dimensions of {@code size} are smaller than 1.
     * @throws IllegalArgumentException if {@code value} is {@code null}
     */
    public void setSize(com.pdftools.geometry.integer.Size value)
    {
        if (value == null)
            throw new IllegalArgumentException("Argument 'value' must not be null.", new NullPointerException("'value'"));
        boolean retVal = setSizeNative(getHandle(), value.getWidth(), value.getHeight());
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                case 3: throw new IllegalArgumentException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }
    }




    private static native long newNative(int sizeWidth, int sizeHeight);
    private native com.pdftools.geometry.integer.Size getSizeNative(long handle);
    private native boolean setSizeNative(long handle, int valueWidth, int valueHeight);

}

