/****************************************************************************
 *
 * File:            TiffLzwImageOptions.java
 *
 * Description:     PDFTOOLS TiffLzwImageOptions Class
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.pdf2image;

import com.pdftools.sys.*;
import com.pdftools.internal.*;
import java.util.EnumSet;
import java.time.OffsetDateTime;
/**
 * <h1>The settings for TIFF output images using the LZW compression algorithm</h1>
 * <p>
 * LZW (Lempel-Ziv-Welch) is a lossless compression algorithm for images. Please
 * consult the copyright laws of your country prior to using this
 * compression algorithm.</p>
 * <p>
 * For the output file name, it is recommended to use the file extension ".tif".
 * </p>
 */
public class TiffLzwImageOptions extends com.pdftools.pdf2image.ImageOptions 
{
    protected TiffLzwImageOptions(long handle) 
    {
        super(handle);
    }

    /**
     * @hidden
     */
    public static TiffLzwImageOptions createDynamicObject(long handle)
    {
        return new TiffLzwImageOptions(handle);
    }


    /**
     * 
     */
    public TiffLzwImageOptions()
    {
        this(newHelper());
    }

    private static long newHelper()
    {
        long handle = newNative();
        if (handle == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }

        return handle;
    }



     /**
     * <h1>Combine a background with the image (Getter)</h1>
     * <p>
     * This property allows a choice of which background
     * to combine with the image.</p>
     * <p>
     * Default: {@link BackgroundType#WHITE }</p>
     */
    public com.pdftools.pdf2image.BackgroundType getBackground()
    {
        int retVal = getBackgroundNative(getHandle());
        if (retVal == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return com.pdftools.pdf2image.BackgroundType.fromValue(retVal);
    }

     /**
     * <h1>Combine a background with the image (Setter)</h1>
     * <p>
     * This property allows a choice of which background
     * to combine with the image.</p>
     * <p>
     * Default: {@link BackgroundType#WHITE }</p>
     * @throws IllegalArgumentException if {@code value} is {@code null}
     */
    public void setBackground(com.pdftools.pdf2image.BackgroundType value)
    {
        if (value == null)
            throw new IllegalArgumentException("Argument 'value' must not be null.", new NullPointerException("'value'"));
        boolean retVal = setBackgroundNative(getHandle(), value.getValue());
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
    }


     /**
     * <h1>The color space of the output image (Getter)</h1>
     * <p>
     * Get or set the color space. If null, the blending color space of the page is used.</p>
     * <p>
     * Default: {@link ColorSpace#RGB }</p>
     */
    public com.pdftools.pdf2image.ColorSpace getColorSpace()
    {
        Integer retVal = getColorSpaceNative(getHandle());
        if (retVal == null)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
            return null;
        }

        return com.pdftools.pdf2image.ColorSpace.fromValue(retVal);
    }

     /**
     * <h1>The color space of the output image (Setter)</h1>
     * <p>
     * Get or set the color space. If null, the blending color space of the page is used.</p>
     * <p>
     * Default: {@link ColorSpace#RGB }</p>
     */
    public void setColorSpace(com.pdftools.pdf2image.ColorSpace value)
    {
        boolean retVal = setColorSpaceNative(getHandle(), value == null ? 0 : value.getValue(), value == null);
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
    }




    private static native long newNative();
    private native int getBackgroundNative(long handle);
    private native boolean setBackgroundNative(long handle, int value);
    private native Integer getColorSpaceNative(long handle);
    private native boolean setColorSpaceNative(long handle, int value, boolean isNull);

}

