/****************************************************************************
 *
 * File:            Archive.java
 *
 * Description:     PDFTOOLS Archive Class
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.pdf2image.profiles;

import com.pdftools.sys.*;
import com.pdftools.internal.*;
import java.util.EnumSet;
import java.time.OffsetDateTime;
/**
 * <h1>The profile to convert PDF documents to TIFF images for archiving</h1>
 * <p>
 * This profile is suitable for archiving PDF documents as rasterized images.</p>
 * <p>
 * The output format is TIFF and cannot be changed.
 * Several compression types are configurable through
 * {@link Archive#getImageOptions }.</p>
 * <p>
 * By default,
 * <ul>
 * <li>
 * {@link Archive#getImageOptions } is set to
 * {@link com.pdftools.pdf2image.TiffLzwImageOptions pdftools.pdf2image.TiffLzwImageOptions}</li>
 * <li>
 * the color space of each image corresponds to the color
 * space of the PDF page
 * </li>
 * </ul></p>
 */
public class Archive extends com.pdftools.pdf2image.profiles.Profile 
{
    protected Archive(long handle) 
    {
        super(handle);
    }

    /**
     * @hidden
     */
    public static Archive createDynamicObject(long handle)
    {
        return new Archive(handle);
    }


    /**
     * 
     */
    public Archive()
    {
        this(newHelper());
    }

    private static long newHelper()
    {
        long handle = newNative();
        if (handle == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }

        return handle;
    }



     /**
     * <h1>The settings for the output TIFF (Getter)</h1>
     * <p>
     * Defines the compression algorithm of the TIFF output image.</p>
     * <p>
     * Supported types are:
     * <ul>
     * <li>
     * {@link com.pdftools.pdf2image.TiffJpegImageOptions pdftools.pdf2image.TiffJpegImageOptions}</li>
     * <li>
     * {@link com.pdftools.pdf2image.TiffLzwImageOptions pdftools.pdf2image.TiffLzwImageOptions}</li>
     * <li>
     * {@link com.pdftools.pdf2image.TiffFlateImageOptions pdftools.pdf2image.TiffFlateImageOptions}</li>
     * </ul></p>
     * <p>
     * Default: {@link com.pdftools.pdf2image.TiffLzwImageOptions pdftools.pdf2image.TiffLzwImageOptions}</p>
     */
    public com.pdftools.pdf2image.ImageOptions getImageOptions()
    {
        long retHandle = getImageOptionsNative(getHandle());
        if (retHandle == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
        return com.pdftools.pdf2image.ImageOptions.createDynamicObject(retHandle);

    }

     /**
     * <h1>The settings for the output TIFF (Setter)</h1>
     * <p>
     * Defines the compression algorithm of the TIFF output image.</p>
     * <p>
     * Supported types are:
     * <ul>
     * <li>
     * {@link com.pdftools.pdf2image.TiffJpegImageOptions pdftools.pdf2image.TiffJpegImageOptions}</li>
     * <li>
     * {@link com.pdftools.pdf2image.TiffLzwImageOptions pdftools.pdf2image.TiffLzwImageOptions}</li>
     * <li>
     * {@link com.pdftools.pdf2image.TiffFlateImageOptions pdftools.pdf2image.TiffFlateImageOptions}</li>
     * </ul></p>
     * <p>
     * Default: {@link com.pdftools.pdf2image.TiffLzwImageOptions pdftools.pdf2image.TiffLzwImageOptions}</p>
     *
     * @throws IllegalArgumentException The given object has the wrong type.
     * @throws IllegalArgumentException if {@code value} is {@code null}
     */
    public void setImageOptions(com.pdftools.pdf2image.ImageOptions value)
    {
        if (value == null)
            throw new IllegalArgumentException("Argument 'value' must not be null.", new NullPointerException("'value'"));
        boolean retVal = setImageOptionsNative(getHandle(), getHandle(value), value);
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                case 3: throw new IllegalArgumentException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }
    }


     /**
     * <h1>The image section mapping (Getter)</h1>
     * <p>
     * This property defines the resolution of the output images.</p>
     * <p>
     * Default: 300 DPI.
     * </p>
     */
    public com.pdftools.pdf2image.RenderPageAtResolution getImageSectionMapping()
    {
        long retHandle = getImageSectionMappingNative(getHandle());
        if (retHandle == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
        return com.pdftools.pdf2image.RenderPageAtResolution.createDynamicObject(retHandle);

    }




    private static native long newNative();
    private native long getImageOptionsNative(long handle);
    private native boolean setImageOptionsNative(long handle, long value, com.pdftools.pdf2image.ImageOptions valueObj);
    private native long getImageSectionMappingNative(long handle);

}

