/****************************************************************************
 *
 * File:            Fax.java
 *
 * Description:     PDFTOOLS Fax Class
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.pdf2image.profiles;

import com.pdftools.sys.*;
import com.pdftools.internal.*;
import java.util.EnumSet;
import java.time.OffsetDateTime;
/**
 * <h1>The profile to convert PDF documents to TIFF Fax images</h1>
 * <p>
 * This profile is suitable for converting PDFs into
 * TIFF-F conforming rasterized images for Facsimile transmission.</p>
 * <p>
 * The output format is a multi-page TIFF file containing all
 * rasterized PDF pages.</p>
 * <p>
 * By default,
 * <ul>
 * <li>
 * the output images are Group 3 - compressed</li>
 * <li>
 * scaled to a width of 1728 pixels, a horizontal
 * resolution of 204 DPI, and a vertical resolution
 * of 98 DPI
 * </li>
 * <li>
 * all colors and gray scale tones are converted
 * to bitonal by using dithering
 * </li>
 * </ul></p>
 * <p>
 * The compression type and the vertical resolution can be set
 * through {@link Fax#getImageOptions }.
 * </p>
 */
public class Fax extends com.pdftools.pdf2image.profiles.Profile 
{
    protected Fax(long handle) 
    {
        super(handle);
    }

    /**
     * @hidden
     */
    public static Fax createDynamicObject(long handle)
    {
        return new Fax(handle);
    }


    /**
     * 
     */
    public Fax()
    {
        this(newHelper());
    }

    private static long newHelper()
    {
        long handle = newNative();
        if (handle == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }

        return handle;
    }



     /**
     * <h1>The settings for the output image (Getter)</h1>
     * 
     */
    public com.pdftools.pdf2image.FaxImageOptions getImageOptions()
    {
        long retHandle = getImageOptionsNative(getHandle());
        if (retHandle == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
        return com.pdftools.pdf2image.FaxImageOptions.createDynamicObject(retHandle);

    }


     /**
     * <h1>The image section mapping (Getter)</h1>
     * This property specifies how a PDF page is placed onto the target image.
     */
    public com.pdftools.pdf2image.RenderPageAsFax getImageSectionMapping()
    {
        long retHandle = getImageSectionMappingNative(getHandle());
        if (retHandle == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
        return com.pdftools.pdf2image.RenderPageAsFax.createDynamicObject(retHandle);

    }




    private static native long newNative();
    private native long getImageOptionsNative(long handle);
    private native long getImageSectionMappingNative(long handle);

}

