/****************************************************************************
 *
 * File:            Viewing.java
 *
 * Description:     PDFTOOLS Viewing Class
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.pdf2image.profiles;

import com.pdftools.sys.*;
import com.pdftools.internal.*;
import java.util.EnumSet;
import java.time.OffsetDateTime;
/**
 * <h1>The profile to convert PDF documents to JPEG or PNG images for viewing</h1>
 * <p>
 * This profile is suitable for converting PDFs to
 * rasterized images for using in web and desktop viewing
 * applications or as thumbnails.</p>
 * <p>
 * By default, {@link Viewing#getImageOptions } is set to
 * {@link com.pdftools.pdf2image.PngImageOptions pdftools.pdf2image.PngImageOptions} which uses the output format
 * PNG and lossless compression.
 * If set to {@link com.pdftools.pdf2image.JpegImageOptions pdftools.pdf2image.JpegImageOptions}, the output format
 * is JPEG.
 * </p>
 */
public class Viewing extends com.pdftools.pdf2image.profiles.Profile 
{
    protected Viewing(long handle) 
    {
        super(handle);
    }

    /**
     * @hidden
     */
    public static Viewing createDynamicObject(long handle)
    {
        return new Viewing(handle);
    }


    /**
     * 
     */
    public Viewing()
    {
        this(newHelper());
    }

    private static long newHelper()
    {
        long handle = newNative();
        if (handle == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }

        return handle;
    }



     /**
     * <h1>The settings for the output image (Getter)</h1>
     * <p>
     * Supported types are:
     * <ul>
     * <li>
     * {@link com.pdftools.pdf2image.PngImageOptions pdftools.pdf2image.PngImageOptions} to create PNG images</li>
     * <li>
     * {@link com.pdftools.pdf2image.JpegImageOptions pdftools.pdf2image.JpegImageOptions} to create JPEG images</li>
     * </ul></p>
     * <p>
     * Default: {@link com.pdftools.pdf2image.PngImageOptions pdftools.pdf2image.PngImageOptions}</p>
     */
    public com.pdftools.pdf2image.ImageOptions getImageOptions()
    {
        long retHandle = getImageOptionsNative(getHandle());
        if (retHandle == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
        return com.pdftools.pdf2image.ImageOptions.createDynamicObject(retHandle);

    }

     /**
     * <h1>The settings for the output image (Setter)</h1>
     * <p>
     * Supported types are:
     * <ul>
     * <li>
     * {@link com.pdftools.pdf2image.PngImageOptions pdftools.pdf2image.PngImageOptions} to create PNG images</li>
     * <li>
     * {@link com.pdftools.pdf2image.JpegImageOptions pdftools.pdf2image.JpegImageOptions} to create JPEG images</li>
     * </ul></p>
     * <p>
     * Default: {@link com.pdftools.pdf2image.PngImageOptions pdftools.pdf2image.PngImageOptions}</p>
     *
     * @throws IllegalArgumentException The given object has the wrong type.
     * @throws IllegalArgumentException if {@code value} is {@code null}
     */
    public void setImageOptions(com.pdftools.pdf2image.ImageOptions value)
    {
        if (value == null)
            throw new IllegalArgumentException("Argument 'value' must not be null.", new NullPointerException("'value'"));
        boolean retVal = setImageOptionsNative(getHandle(), getHandle(value), value);
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                case 3: throw new IllegalArgumentException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }
    }


     /**
     * <h1>The image section mapping (Getter)</h1>
     * <p>
     * This property specifies how a PDF page is placed onto the target image.</p>
     * <p>
     * Supported types are:
     * <ul>
     * <li>
     * {@link com.pdftools.pdf2image.RenderPageAtResolution pdftools.pdf2image.RenderPageAtResolution} to define the resolution of the output images.</li>
     * <li>
     * {@link com.pdftools.pdf2image.RenderPageToMaxImageSize pdftools.pdf2image.RenderPageToMaxImageSize} to define the maximum image size of the output images.</li>
     * </ul></p>
     * <p>
     * Default: {@link com.pdftools.pdf2image.RenderPageAtResolution pdftools.pdf2image.RenderPageAtResolution} with 150 DPI.
     * </p>
     */
    public com.pdftools.pdf2image.ImageSectionMapping getImageSectionMapping()
    {
        long retHandle = getImageSectionMappingNative(getHandle());
        if (retHandle == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
        return com.pdftools.pdf2image.ImageSectionMapping.createDynamicObject(retHandle);

    }

     /**
     * <h1>The image section mapping (Setter)</h1>
     * <p>
     * This property specifies how a PDF page is placed onto the target image.</p>
     * <p>
     * Supported types are:
     * <ul>
     * <li>
     * {@link com.pdftools.pdf2image.RenderPageAtResolution pdftools.pdf2image.RenderPageAtResolution} to define the resolution of the output images.</li>
     * <li>
     * {@link com.pdftools.pdf2image.RenderPageToMaxImageSize pdftools.pdf2image.RenderPageToMaxImageSize} to define the maximum image size of the output images.</li>
     * </ul></p>
     * <p>
     * Default: {@link com.pdftools.pdf2image.RenderPageAtResolution pdftools.pdf2image.RenderPageAtResolution} with 150 DPI.
     * </p>
     *
     * @throws IllegalArgumentException The given object has the wrong type.
     * @throws IllegalArgumentException if {@code value} is {@code null}
     */
    public void setImageSectionMapping(com.pdftools.pdf2image.ImageSectionMapping value)
    {
        if (value == null)
            throw new IllegalArgumentException("Argument 'value' must not be null.", new NullPointerException("'value'"));
        boolean retVal = setImageSectionMappingNative(getHandle(), getHandle(value), value);
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                case 3: throw new IllegalArgumentException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }
    }




    private static native long newNative();
    private native long getImageOptionsNative(long handle);
    private native boolean setImageOptionsNative(long handle, long value, com.pdftools.pdf2image.ImageOptions valueObj);
    private native long getImageSectionMappingNative(long handle);
    private native boolean setImageSectionMappingNative(long handle, long value, com.pdftools.pdf2image.ImageSectionMapping valueObj);

}

