/****************************************************************************
 *
 * File:            EventCategory.java
 *
 * Description:     EventCategory Enumeration
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.pdfa.conversion;

/**
 * <h1>The category of conversion events</h1>
 * See {@link Converter.ConversionEventListener } for more information on conversion events.
 */
public enum EventCategory  
{
    /**
     * <h1>The conversion resulted in visual differences to the document.</h1>
     * <p>
     * The conversion is optimized to preserve the visual appearance of documents.
     * However, under some circumstances visual differences cannot be avoided.
     * This is typically the case for low quality and erroneous input documents.</p>
     * <p>
     * Examples:
     * <ul>
     * <li>
     * The visual appearance of a proprietary annotation type could not be generated.</li>
     * <li>
     * Numbers that exceed the allowed value range have been clipped.</li>
     * <li>
     * Text of an invalid font is unclear because its mapping to glyphs is ambiguous.</li>
     * </ul></p>
     * <p>
     * Suggested severity: {@link EventSeverity#WARNING }</p>
     * <p>
     * It is not possible for the SDK to gauge the effect of the visual differences on the document's content.
     * Therefore, it is recommended to let a user assess, whether or not the conversion result is acceptable.
     * If a manual review is not feasible, events of this category should be classified as an {@link EventSeverity#ERROR }.
     * </p>
     */
    VISUAL_DIFFERENCES(0x00000001),
    /**
     * <h1>Repaired a corrupt document.</h1>
     * <p>
     * Corrupt documents are repaired automatically.
     * Since the specification does not define how corrupt documents should be repaired, each viewer has its own heuristics for doing so.
     * Therefore, the repaired document might have visual differences to the input document in your viewer.
     * For that reason, an event is generated such that the repaired document can be reviewed, similarly to {@link EventCategory#VISUAL_DIFFERENCES }.</p>
     * <p>
     * Examples for documents that must be repaired:
     * <ul>
     * <li>
     * The document has been damaged, e.g. during an incomplete file upload.</li>
     * <li>
     * The document has been created by an erroneous application.</li>
     * </ul></p>
     * <p>
     * Suggested severity: {@link EventSeverity#WARNING }</p>
     */
    REPAIRED_CORRUPTION(0x00000002),
    /**
     * <h1>Managed colors of input document.</h1>
     * <p>
     * Purely informational messages related to color management.</p>
     * <p>
     * Examples:
     * <ul>
     * <li>
     * Copied PDF/A output intent from input file.</li>
     * <li>
     * Embedded ICC color profile.</li>
     * </ul></p>
     * <p>
     * Suggested severity: {@link EventSeverity#INFORMATION }</p>
     */
    MANAGED_COLORS(0x00000004),
    /**
     * <h1>Resolved ambiguous or conflicting descriptions of colorants (spot colors).</h1>
     * <p>
     * Colorants are special inks used in addition to the process colors (e.g. red, green, and blue in the RGB color space or cyan, magenta, yellow and black in the CMYK color space).
     * Popular colorants are PANTONE colors typically used in printing; or also metallic or fluorescent inks.</p>
     * <p>
     * Colorants in PDF documents contain a description that is required to paint a good approximation of the intended color in case the colorant is unavailable.
     * Within the same document all descriptions for the same colorant should be equal.
     * This warning is generated if conflicting descriptions must be harmonized, for example during PDF/A conversion.</p>
     * <p>
     * This has no effect on output devices where the colorant is available, e.g. on certain printers.
     * For other output devices this warning may indicate visual differences.
     * However, for well-formed documents (i.e. not maliciously created documents), the visual differences are not noticeable.</p>
     * <p>
     * Suggested severity: {@link EventSeverity#INFORMATION }</p>
     */
    CHANGED_COLORANT(0x00000008),
    /**
     * <h1>Removed references to external content.</h1>
     * <p>
     * Examples:
     * <ul>
     * <li>
     * Removed references to external files containing stream data used in the document.</li>
     * </ul></p>
     * <p>
     * Suggested severity: {@link EventSeverity#INFORMATION }</p>
     */
    REMOVED_EXTERNAL_CONTENT(0x00000010),
    /**
     * <h1>Converted fonts of input document.</h1>
     * <p>
     * Purely informational messages related to font management.</p>
     * <p>
     * Examples:
     * <ul>
     * <li>
     * Embedded a font.</li>
     * </ul></p>
     * <p>
     * Suggested severity: {@link EventSeverity#INFORMATION }</p>
     */
    CONVERTED_FONT(0x00000020),
    /**
     * <h1>Substituted a font for a similar one.</h1>
     * <p>
     * If a required font is not embedded and not available in the installed fonts, a similar font must be chosen and used.
     * This is a commonly performed when viewing or printing a PDF document.
     * While this may lead to minor visual differences, all text is preserved.</p>
     * <p>
     * It is important that the installed fonts contain all fonts that are not embedded in the input documents.
     * See the product's installation documentation for a list of fonts that are recommended to install.</p>
     * <p>
     * Examples:
     * <ul>
     * <li>
     * Substituted font 'GothicBBB-Medium' with 'MS-Gothic'.</li>
     * </ul></p>
     * <p>
     * Suggested severity: {@link EventSeverity#INFORMATION }</p>
     */
    SUBSTITUTED_FONT(0x00000040),
    /**
     * <h1>Converted transparent object to opaque.</h1>
     * <p>
     * Because transparency is not allowed in PDF/A-1, transparent objects have to be converted to opaque when converting a document to PDF/A-1.
     * This can lead to visual differences.
     * Even though the conversion has been optimized to reduce visual differences, they might be noticeable.
     * Therefore, it is highly recommended to convert documents to PDF/A-2 or higher.
     * These versions of the standard allow transparency, which results in a higher conversion quality.</p>
     * <p>
     * This conversion event should be handled similarly to {@link EventCategory#VISUAL_DIFFERENCES }.</p>
     * <p>
     * Suggested severity: {@link EventSeverity#WARNING }</p>
     */
    REMOVED_TRANSPARENCY(0x00000080),
    /**
     * <h1>Removed prohibited annotation type.</h1>
     * <p>
     * Removing annotations does not lead to visual differences, but merely removes the interactivity of the elements.</p>
     * <p>
     * Examples:
     * <ul>
     * <li>
     * Removed proprietary annotation types.</li>
     * <li>
     * Removed forbidden annotation types, e.g. 3D.</li>
     * </ul></p>
     * <p>
     * Suggested severity: {@link EventSeverity#INFORMATION }</p>
     */
    REMOVED_ANNOTATION(0x00000100),
    /**
     * <h1>Removed multimedia content (sound, movie).</h1>
     * <p></p>
     * <p>
     * Suggested severity: {@link EventSeverity#INFORMATION }</p>
     */
    REMOVED_MULTIMEDIA(0x00000200),
    /**
     * <h1>Removed prohibited action type.</h1>
     * <p>
     * Removing actions does not lead to visual differences.</p>
     * <p>
     * Examples:
     * <ul>
     * <li>
     * Removed JavaScript actions in interactive form fields.</li>
     * </ul></p>
     * <p>
     * Suggested severity: {@link EventSeverity#INFORMATION }</p>
     */
    REMOVED_ACTION(0x00000400),
    /**
     * <h1>Removed parts of the XMP metadata that could not be repaired.</h1>
     * <p>
     * This event indicates that metadata properties have been removed during conversion.
     * This includes any kind of metadata like e.g. the XMP metadata of a PDF document.</p>
     * <p>
     * Examples:
     * <ul>
     * <li>
     * Parts of the XMP metadata of a PDF did not conform to the PDF/A standard and had to be removed.</li>
     * </ul></p>
     * <p>
     * Suggested severity: {@link EventSeverity#INFORMATION }</p>
     */
    REMOVED_METADATA(0x00000800),
    /**
     * <h1>Removed logical structure (tagging) information.</h1>
     * <p>
     * The logical structure of the document is a description of the content of its pages.
     * This description has to be provided by the creator of the document.
     * It consists of a fine granular hierarchical tagging that distinguishes between the actual content and artifacts (such as page numbers, layout artifacts, etc.).
     * The tagging provides a meaningful description, for example "This is a header", "This color image shows a small sailing boat at sunset", etc.
     * This information can be used e.g. to read the document to the visually impaired.</p>
     * <p>
     * The SDK has been optimized to preserve tagging information.
     * Typically, tagging information only has to be removed if it is invalid or corrupt.</p>
     * <p>
     * Suggested severity: {@link EventSeverity#INFORMATION }</p>
     */
    REMOVED_STRUCTURE(0x00001000),
    /**
     * <h1>Removed optional content groups (layers).</h1>
     * <p>
     * Because optional content is not allowed in PDF/A-1, it has to be removed when converting a document to PDF/A-1.
     * Removing layers does not change the initial appearance of pages.
     * However, the visibility of content cannot be changed anymore.
     * Therefore, it is highly recommended to convert documents to PDF/A-2 or higher.
     * These versions of the standard allow optional content, which results in a higher conversion quality.</p>
     * <p>
     * Suggested severity: {@link EventSeverity#WARNING }</p>
     */
    REMOVED_OPTIONAL_CONTENT(0x00002000),
    /**
     * <h1>Converted embedded file.</h1>
     * <p>
     * Purely informational messages related to the conversion of embedded files.</p>
     * <p>
     * Examples:
     * <ul>
     * <li>
     * Copied an embedded file.</li>
     * <li>
     * Embedded a file that has successfully been converted to PDF/A.</li>
     * </ul></p>
     * <p>
     * Suggested severity: {@link EventSeverity#INFORMATION }</p>
     */
    CONVERTED_EMBEDDED_FILE(0x00004000),
    /**
     * <h1>Removed embedded files.</h1>
     * <p>
     * Whether embedded files have to be removed depends on the conformance:
     * <ul>
     * <li>
     * <p>
     * <em>PDF/A-1:</em>
     * Embedded files are not allowed.</p>
     * <p>
     * All embedded files have to be removed.
     * </p></li>
     * <li>
     * <p>
     * <em>PDF/A-2:</em>
     * Only embedded files are allowed, that conform to PDF/A.</p>
     * <p>
     * All embedded PDF documents are converted to PDF/A.
     * All other files have to be removed.</p>
     * <p>
     * The Conversion Service can be used to convert PDF documents with other types of embedded files, e.g. Microsoft Office documents, images, and mails, to PDF/A-2.
     * </p></li>
     * <li>
     * <p>
     * <em>PDF/A-3:</em>
     * All types of embedded files are allowed and copied as-is.</p>
     * <p>
     * The Conversion Service can be used, if a more fine-grained control over the conversion and copying of embedded files is required.
     * </p></li>
     * </ul></p>
     * <p>
     * Suggested severity: {@link EventSeverity#WARNING }</p>
     */
    REMOVED_EMBEDDED_FILE(0x00008000),
    /**
     * <h1>Removed signatures of signed input file.</h1>
     * <p>
     * Converting a signed document invalidates its signatures.
     * For that reason, the cryptographic parts of the signatures are removed while their visual appearances are preserved.</p>
     * <p>
     * Note that we generally recommend to sign PDF/A documents only for two reasons.
     * First, this ensures that the file is not corrupt and its visual appearance is well defined, such than it can be reproduced flawlessly and authentically in any environment.
     * Second, PDF/A conformance is typically required if the file is to be archived, e.g. by the recipient.
     * Because signed files cannot be converted to PDF/A without breaking the signature, the signature must be removed before the file can be archived.
     * By converting files to PDF/A before applying the signature, this dilemma can be avoided.</p>
     * <p>
     * Suggested severity: {@link EventSeverity#INFORMATION }</p>
     */
    REMOVED_SIGNATURE(0x00010000);

    EventCategory(int value) 
    {
        this.value = value;
    }

    /**
     * @hidden
     */
    public static EventCategory fromValue(int value) 
    {
        switch (value) 
        {
            case 0x00000001: return VISUAL_DIFFERENCES;
            case 0x00000002: return REPAIRED_CORRUPTION;
            case 0x00000004: return MANAGED_COLORS;
            case 0x00000008: return CHANGED_COLORANT;
            case 0x00000010: return REMOVED_EXTERNAL_CONTENT;
            case 0x00000020: return CONVERTED_FONT;
            case 0x00000040: return SUBSTITUTED_FONT;
            case 0x00000080: return REMOVED_TRANSPARENCY;
            case 0x00000100: return REMOVED_ANNOTATION;
            case 0x00000200: return REMOVED_MULTIMEDIA;
            case 0x00000400: return REMOVED_ACTION;
            case 0x00000800: return REMOVED_METADATA;
            case 0x00001000: return REMOVED_STRUCTURE;
            case 0x00002000: return REMOVED_OPTIONAL_CONTENT;
            case 0x00004000: return CONVERTED_EMBEDDED_FILE;
            case 0x00008000: return REMOVED_EMBEDDED_FILE;
            case 0x00010000: return REMOVED_SIGNATURE;
        }
        throw new IllegalArgumentException("Unknown value for EventCategory: " + value);
    }

    /**
     * @hidden
     */
    public int getValue() 
    {
        return value;
    }

    private int value;
}
