/****************************************************************************
 *
 * File:            EventSeverity.java
 *
 * Description:     EventSeverity Enumeration
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.pdfa.conversion;

/**
 * <h1>The severity of conversion events</h1>
 * See {@link Converter.ConversionEventListener } for more information on conversion events.
 */
public enum EventSeverity  
{
    /**
     * <h1>A conversion event which is of an informational nature</h1>
     * <p>
     * An informational event requires no further action.</p>
     * <p>
     * By default events of the following {@link EventCategory } are classified as {@link EventSeverity#INFORMATION }:
     * <ul>
     * <li>
     * {@link EventCategory#MANAGED_COLORS }</li>
     * <li>
     * {@link EventCategory#CHANGED_COLORANT }</li>
     * <li>
     * {@link EventCategory#REMOVED_EXTERNAL_CONTENT }</li>
     * <li>
     * {@link EventCategory#CONVERTED_FONT }</li>
     * <li>
     * {@link EventCategory#SUBSTITUTED_FONT }</li>
     * <li>
     * {@link EventCategory#REMOVED_ANNOTATION }</li>
     * <li>
     * {@link EventCategory#REMOVED_MULTIMEDIA }</li>
     * <li>
     * {@link EventCategory#REMOVED_ACTION }</li>
     * <li>
     * {@link EventCategory#REMOVED_METADATA }</li>
     * <li>
     * {@link EventCategory#REMOVED_STRUCTURE }</li>
     * <li>
     * {@link EventCategory#CONVERTED_EMBEDDED_FILE }</li>
     * <li>
     * {@link EventCategory#REMOVED_SIGNATURE }</li>
     * </ul></p>
     */
    INFORMATION(1),
    /**
     * <h1>A conversion event which is generally considered a non-critical issue</h1>
     * <p>
     * An warning that might require further actions.</p>
     * <p>
     * By default events of the following {@link EventCategory } are classified as {@link EventSeverity#WARNING }:
     * <ul>
     * <li>
     * {@link EventCategory#VISUAL_DIFFERENCES }</li>
     * <li>
     * {@link EventCategory#REPAIRED_CORRUPTION }</li>
     * <li>
     * {@link EventCategory#REMOVED_TRANSPARENCY } (PDF/A-1 only)</li>
     * <li>
     * {@link EventCategory#REMOVED_EMBEDDED_FILE }  (PDF/A-1 and PDF/A-2 only)</li>
     * <li>
     * {@link EventCategory#REMOVED_OPTIONAL_CONTENT } (PDF/A-1 only)</li>
     * </ul></p>
     */
    WARNING(2),
    /**
     * <h1>A conversion event which is generally considered a critical issue</h1>
     * <p>
     * A critical issue for which the conversion must be considered as failed.</p>
     * <p>
     * By default no event uses this severity.
     * </p>
     */
    ERROR(3);

    EventSeverity(int value) 
    {
        this.value = value;
    }

    /**
     * @hidden
     */
    public static EventSeverity fromValue(int value) 
    {
        switch (value) 
        {
            case 1: return INFORMATION;
            case 2: return WARNING;
            case 3: return ERROR;
        }
        throw new IllegalArgumentException("Unknown value for EventSeverity: " + value);
    }

    /**
     * @hidden
     */
    public int getValue() 
    {
        return value;
    }

    private int value;
}
