/****************************************************************************
 *
 * File:            ValidationOptions.java
 *
 * Description:     PDFTOOLS ValidationOptions Class
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.pdfa.validation;

import com.pdftools.sys.*;
import com.pdftools.internal.*;
import java.util.EnumSet;
import java.time.OffsetDateTime;
/**
 * <h1>The PDF validation options</h1>
 * Options to check the quality and standard conformance of documents using the validator's method {@link Validator#validate }.
 */
public class ValidationOptions extends NativeObject 
{
    protected ValidationOptions(long handle) 
    {
        super(handle);
    }

    /**
     * @hidden
     */
    public static ValidationOptions createDynamicObject(long handle)
    {
        return new ValidationOptions(handle);
    }


    /**
     * 
     */
    public ValidationOptions()
    {
        this(newHelper());
    }

    private static long newHelper()
    {
        long handle = newNative();
        if (handle == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }

        return handle;
    }



     /**
     * <h1>The conformance to be validated (Getter)</h1>
     * <p>
     * The required conformance or {@code null} to validate the document's claimed conformance, i.e. {@link com.pdftools.pdf.Document#getConformance pdftools.pdf.Document.getConformance}.</p>
     * <p>
     * The PDF validation verifies if the input document conforms to all standards associated with this conformance.</p>
     * <p>
     * Note that it is generally only meaningful to validate the claimed conformance of a document.</p>
     * <p>
     * Default value: {@code null}, i.e. validate the document's claimed conformance.
     * </p>
     */
    public com.pdftools.pdf.Conformance getConformance()
    {
        Integer retVal = getConformanceNative(getHandle());
        if (retVal == null)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
            return null;
        }
        return new com.pdftools.pdf.Conformance(retVal);

    }

     /**
     * <h1>The conformance to be validated (Setter)</h1>
     * <p>
     * The required conformance or {@code null} to validate the document's claimed conformance, i.e. {@link com.pdftools.pdf.Document#getConformance pdftools.pdf.Document.getConformance}.</p>
     * <p>
     * The PDF validation verifies if the input document conforms to all standards associated with this conformance.</p>
     * <p>
     * Note that it is generally only meaningful to validate the claimed conformance of a document.</p>
     * <p>
     * Default value: {@code null}, i.e. validate the document's claimed conformance.
     * </p>
     */
    public void setConformance(com.pdftools.pdf.Conformance value)
    {
        boolean retVal = setConformanceNative(getHandle(), value == null ? 0 : value.getValue(), value == null);
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
    }




    private static native long newNative();
    private native Integer getConformanceNative(long handle);
    private native boolean setConformanceNative(long handle, int value, boolean isNull);

}

