/****************************************************************************
 *
 * File:            Validator.java
 *
 * Description:     PDFTOOLS Validator Class
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.pdfa.validation;

import com.pdftools.sys.*;
import com.pdftools.internal.*;
import java.util.EnumSet;
import java.time.OffsetDateTime;
import java.util.ArrayList;
import java.util.*;
/**
 * <h1>The class to validate the standard conformance of documents</h1>
 */
public class Validator extends NativeObject 
{
    protected Validator(long handle) 
    {
        super(handle);
    }

    /**
     * @hidden
     */
    public static Validator createDynamicObject(long handle)
    {
        return new Validator(handle);
    }

    /***
     * Listener interface for the {@link Error} event.
     *
     */
    public interface ErrorListener extends EventListener
    {
        void error(Error event);
    }

    /**
     * 
     * Report a validation issue found in {@link Validator#analyze } or {@link Validator#validate }.
     */
    public class Error extends EventObject
    {
        private static final long serialVersionUID = 522L;

        private String dataPart;
        private String message;
        private com.pdftools.pdfa.validation.ErrorCategory category;
        private String context;
        private int pageNo;
        private int objectNo;

        private Error(Object source, String dataPart, String message, com.pdftools.pdfa.validation.ErrorCategory category, String context, int pageNo, int objectNo) {
            super(source);
            this.dataPart = dataPart;
            this.message = message;
            this.category = category;
            this.context = context;
            this.pageNo = pageNo;
            this.objectNo = objectNo;
        }

        /**
         * <p>
         * The data part is {@code null} for the main file and a data part specification for embedded files.</p>
         * <p>
         * Examples:
         * <ul>
         * <li>
         * {@code embedded-file:file.pdf}: For a file {@code file.pdf} that is embedded in the main file.</li>
         * <li>
         * {@code embedded-file:file1.pdf/embedded-file:file2.pdf}: For a file {@code file2.pdf} that is embedded in an embedded file {@code file1.pdf}.</li>
         * </ul></p>
         */
        public String getDataPart() {
            return dataPart;
        }

        /**
         * The validation message
         */
        public String getMessage() {
            return message;
        }

        /**
         * The category of the validation error
         */
        public com.pdftools.pdfa.validation.ErrorCategory getCategory() {
            return category;
        }

        /**
         * A description of the context where the error occurred
         */
        public String getContext() {
            return context;
        }

        /**
         * The page number this error is associated to or {@code 0}
         */
        public int getPageNo() {
            return pageNo;
        }

        /**
         * The number of the PDF object this error is associated to
         */
        public int getObjectNo() {
            return objectNo;
        }


    }

    private Hashtable<ErrorListener, ErrorNativeClass> errorDic = new Hashtable<ErrorListener, ErrorNativeClass>();

    private class ErrorNativeClass
    {
        private ErrorListener listener;
        private long context;
        public ErrorNativeClass(ErrorListener listener)
        {
            this.listener = listener;
        }

        public void errorHandler(String dataPart, String message, com.pdftools.pdfa.validation.ErrorCategory category, String context, int pageNo, int objectNo)
        {
            Error event = new Error(this, dataPart, message, category, context, pageNo, objectNo);
            try {
                this.listener.error(event);
            }
            catch (Exception ex) { }
        }
    }

    /**
     * Add a listener for the {@link Error} event.
     * @param listener Listener for the {@link Error} event. 
     *                 If a listener is added that is already registered, it is ignored.
     */
    public void addErrorListener(ErrorListener listener)
    {
        if(!errorDic.containsKey(listener))
        {
            ErrorNativeClass eventNativeClass = new ErrorNativeClass(listener);
            long context = addErrorHandlerNative(getHandle(), eventNativeClass);
            if (context == 0)
                throwLastRuntimeException();

            eventNativeClass.context = context;
            errorDic.put(listener, eventNativeClass);
        }
    }

    /**
     * Remove registered listener for the {@link Error} event.
     * @param listener Listener for the {@link Error} event that should be removed.
     *                 If the listener is not registered, it is ignored.
     */
    public void removeErrorListener(ErrorListener listener)
    {
        if(errorDic.containsKey(listener))
        {
            if (!removeErrorHandlerNative(getHandle(), errorDic.get(listener).context))
            {
                if (getLastErrorCode() != 5)
                    throwLastRuntimeException();
            }
            errorDic.remove(listener);
        }
    }



    /**
     * 
     */
    public Validator()
    {
        this(newHelper());
    }

    private static long newHelper()
    {
        long handle = newNative();
        if (handle == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }

        return handle;
    }


    /**
     * <h1>Validate the standards conformance of a PDF document.</h1>
     * @param document 
     * The document to check the quality of
     * @return 
     * The result of the validation
     *
     * @throws com.pdftools.LicenseException The license check has failed.
     * @throws com.pdftools.ProcessingException The processing has failed.
     * @throws IllegalArgumentException if {@code document} is {@code null}
     */
    public com.pdftools.pdfa.validation.ValidationResult validate(com.pdftools.pdf.Document document) 
        throws 
            com.pdftools.LicenseException,

            com.pdftools.ProcessingException
    {
        if (document == null)
            throw new IllegalArgumentException("Argument 'document' must not be null.", new NullPointerException("'document'"));

        return validate(document, null);
    }

    /**
     * <h1>Validate the standards conformance of a PDF document.</h1>
     * @param document 
     * The document to check the quality of
     * @param options 
     * The options or {@code null} for default validation options
     * @return 
     * The result of the validation
     *
     * @throws com.pdftools.LicenseException The license check has failed.
     * @throws com.pdftools.ProcessingException The processing has failed.
     * @throws IllegalArgumentException if {@code document} is {@code null}
     */
    public com.pdftools.pdfa.validation.ValidationResult validate(com.pdftools.pdf.Document document, com.pdftools.pdfa.validation.ValidationOptions options) 
        throws 
            com.pdftools.LicenseException,

            com.pdftools.ProcessingException
    {
        if (document == null)
            throw new IllegalArgumentException("Argument 'document' must not be null.", new NullPointerException("'document'"));

        long retHandle = validateNative(getHandle(), getHandle(document), document, getHandle(options), options);

        if (retHandle == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                case 12: throw new com.pdftools.LicenseException(getLastErrorMessage());
                case 21: throw new com.pdftools.ProcessingException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }

        return com.pdftools.pdfa.validation.ValidationResult.createDynamicObject(retHandle);
    }

    /**
     * <h1>Analyze a PDF document in preparation for its conversion to PDF/A.</h1>
     * This method validates the document's standards conformance like {@link Validator#validate }.
     * In addition to that, certain additional checks can be performed.
     * However, the main difference is that the analysis result can be used in {@link com.pdftools.pdfa.conversion.Converter#convert pdftools.pdfa.conversion.Converter.convert} to convert the PDF document to PDF/A.
     * @param document 
     * The document to analyze
     * @return 
     * The result of the analysis
     *
     * @throws com.pdftools.LicenseException The license check has failed.
     * @throws IllegalArgumentException The conformance of the {@link com.pdftools.pdfa.validation.Validator#analyze options} argument is not PDF/A.
     * @throws IllegalArgumentException if {@code document} is {@code null}
     */
    public com.pdftools.pdfa.validation.AnalysisResult analyze(com.pdftools.pdf.Document document) 
        throws 
            com.pdftools.LicenseException
    {
        if (document == null)
            throw new IllegalArgumentException("Argument 'document' must not be null.", new NullPointerException("'document'"));

        return analyze(document, null);
    }

    /**
     * <h1>Analyze a PDF document in preparation for its conversion to PDF/A.</h1>
     * This method validates the document's standards conformance like {@link Validator#validate }.
     * In addition to that, certain additional checks can be performed.
     * However, the main difference is that the analysis result can be used in {@link com.pdftools.pdfa.conversion.Converter#convert pdftools.pdfa.conversion.Converter.convert} to convert the PDF document to PDF/A.
     * @param document 
     * The document to analyze
     * @param options 
     * The options or {@code null} for default analysis options
     * @return 
     * The result of the analysis
     *
     * @throws com.pdftools.LicenseException The license check has failed.
     * @throws IllegalArgumentException The conformance of the {@link com.pdftools.pdfa.validation.Validator#analyze options} argument is not PDF/A.
     * @throws IllegalArgumentException if {@code document} is {@code null}
     */
    public com.pdftools.pdfa.validation.AnalysisResult analyze(com.pdftools.pdf.Document document, com.pdftools.pdfa.validation.AnalysisOptions options) 
        throws 
            com.pdftools.LicenseException
    {
        if (document == null)
            throw new IllegalArgumentException("Argument 'document' must not be null.", new NullPointerException("'document'"));

        long retHandle = analyzeNative(getHandle(), getHandle(document), document, getHandle(options), options);

        if (retHandle == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                case 3: throw new IllegalArgumentException(getLastErrorMessage());
                case 12: throw new com.pdftools.LicenseException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }

        return com.pdftools.pdfa.validation.AnalysisResult.createDynamicObject(retHandle);
    }




    private native long addErrorHandlerNative(long handle, ErrorNativeClass eventClass);
    private native boolean removeErrorHandlerNative(long handle, long context);

    private static native long newNative();
    private native long validateNative(long handle, long document, com.pdftools.pdf.Document documentObj, long options, com.pdftools.pdfa.validation.ValidationOptions optionsObj);
    private native long analyzeNative(long handle, long document, com.pdftools.pdf.Document documentObj, long options, com.pdftools.pdfa.validation.AnalysisOptions optionsObj);

}

