/****************************************************************************
 *
 * File:            CustomTextVariableMap.java
 *
 * Description:     PDFTOOLS CustomTextVariableMap Class
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.sign;

import com.pdftools.sys.*;
import com.pdftools.internal.*;
import java.util.EnumSet;
import java.time.OffsetDateTime;
import java.util.AbstractSet;
import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
/**
 * 
 * A map that maps custom text variable names to its value.
 */
public class CustomTextVariableMap extends NativeObject implements Map<String, String> 
{
    protected CustomTextVariableMap(long handle) 
    {
        super(handle);
    }

    /**
     * @hidden
     */
    public static CustomTextVariableMap createDynamicObject(long handle)
    {
        return new CustomTextVariableMap(handle);
    }





    /////////////////////////////////////////////////////
    // Map implementation
    /////////////////////////////////////////////////////

    @Override
    public void putAll(Map<? extends String, ? extends String> arg0)
    {
        if (arg0 == null)
            throw new NullPointerException("'arg0'");
        for (Entry<? extends String, ? extends String> e : arg0.entrySet())
            put(e.getKey(), e.getValue());
    }

    @SuppressWarnings("unchecked")
    @Override
    public boolean containsKey(Object key)
    {
        if (key == null)
            throw new NullPointerException("'key'");
        return getIt((String)key) >= 0;
    }

    @Override
    public boolean containsValue(Object value)
    {
        if (value == null)
            throw new NullPointerException("'value'");
        for (String val : values())
            if (value.equals(val))
                return true;
        return false;
    }

    @SuppressWarnings("unchecked")
    @Override
    public String get(Object key)
    {
        if (key == null)
            throw new NullPointerException("'key'");
        int it = getIt((String)key);
        return getValue(it);
    }

    @Override
    public Set<Entry<String, String>> entrySet()
    {
        return new CustomTextVariableMapEntrySet(this);
    }

    @Override
    public boolean isEmpty()
    {
        return getBegin() == getEnd();
    }

    @Override
    public Set<String> keySet()
    {
        return new KeySet<String,String>(this);
    }

    @SuppressWarnings("unchecked")
    @Override
    public String remove(Object key)
    {
        if (key == null)
            throw new NullPointerException("'key'");
        int it = getIt((String)key);
        if (it == -1)
            return null;
        String value = getValue(it);
        remove(it);
        return value;
    }

    @Override
    public Collection<String> values()
    {
        return new ValueCollection<String,String>(this);
    }

    @Override
    public int size()
    {
        int size = size(getHandle());
        if (size == -1)
            throwLastRuntimeException(false);
        return size;
    }

    @Override
    public void clear()
    {
        if(!clear(getHandle()))
            throwLastRuntimeException(false);
    }

    int getIt(String key)
    {
        if (key == null)
            throw new NullPointerException("'key'");
        int it = getIt(getHandle(), key);
        if (it == -1 && getLastErrorCode() != 5)
            throwLastRuntimeException(false);

        return it;
    }

    int getBegin()
    {
        int retVal = getBegin(getHandle());
        if (retVal == -1)
            throwLastRuntimeException(false);

        return retVal;
    }

    int getEnd()
    {
        int retVal = getEnd(getHandle());
        if (retVal == -1)
            throwLastRuntimeException(false);

        return retVal;
    }

    int getNext(int it)
    {
        int retVal = getNext(getHandle(), it);
        if (retVal == -1)
            throwLastRuntimeException(false);

        return retVal;
    }

    String getKey(int it)
    {
        String key = getKey(getHandle(), it);
        if(key == null)
            throwLastRuntimeException(false);

        return key;
    }

    void remove(int it)
    {
        if(!remove(getHandle(), it))
            throwLastRuntimeException(false);
    }

    @Override
    public String put(String key, String value)
    {
        if (key == null)
            throw new NullPointerException("'key'");
        if (value == null)
            throw new NullPointerException("'value'");

        long handle = getHandle();
        int it = getIt(key);
        String prev = it >= 0 ? getValue(handle, it) : null;
        if(!put(handle, key, value))
            throwLastRuntimeException(false);
        return prev;
    }

    String getValue(int it)
    {
        String retValue = getValue(getHandle(), it);
        if (retValue == null)
            throwLastRuntimeException(false);
        return retValue;
    }

    void setValue(int it, String value)
    {
        if (value == null)
            throw new NullPointerException("'value'");

        if(!setValue(getHandle(), it, value))
            throwLastRuntimeException(false);
    }

    private native int size(long handle);
    private native boolean clear(long handle);
    private native int getIt(long handle, String key);
    private native int getBegin(long handle);
    private native int getEnd(long handle);
    private native String getKey(long handle, int it);
    private native int getNext(long handle, int it);
    private native boolean remove(long handle, int it);
    private native boolean put(long handle, String key, String value);
    private native String getValue(long handle, int it);
    private native boolean setValue(long handle, int it, String value);

}

class CustomTextVariableMapEntry implements Entry<String,String>
{
    public CustomTextVariableMapEntry(CustomTextVariableMap map, int it)
    {
        this.map = map;
        this.it = it;
    }

    @Override
    public String getKey()
    {
        return map.getKey(it);
    }

    @Override
    public String getValue()
    {
        return map.getValue(it);
    }

    @Override
    public String setValue(String value)
    {
        String prev = map.getValue(it);
        map.setValue(it, value);
        return prev;
    }

    CustomTextVariableMap map;
    int it;
}

class CustomTextVariableMapEntryIterator implements Iterator<Entry<String,String>>
{
    public CustomTextVariableMapEntryIterator(CustomTextVariableMap map)
    {
        this.map = map;
        this.it = map.getBegin();
    }

    @Override
    public boolean hasNext()
    {
        return it != map.getEnd();
    }

    @Override
    public Entry<String, String> next()
    {
        Entry<String,String> entry = new CustomTextVariableMapEntry(map, it);
        it = map.getNext(it);
        return entry;
    }

    @Override
    public void remove()
    {
        map.remove(it);
    }

    CustomTextVariableMap map;
    int it;
}

class CustomTextVariableMapEntrySet extends AbstractSet<Entry<String,String>>
{
    public CustomTextVariableMapEntrySet(CustomTextVariableMap map)
    {
        this.map = map;
    }

    /**
     * Not supported.
     */
    @Override
    public boolean remove(Object o)
    {
        throw new UnsupportedOperationException();
    }

    @Override
    public void clear()
    {
        map.clear();
    }

    /**
     * Not supported.
     */
    @Override
    public boolean contains(Object object)
    {
        throw new UnsupportedOperationException();
    }

    @Override
    public Iterator<Entry<String,String>> iterator()
    {
        return new CustomTextVariableMapEntryIterator(map);
    }

    @Override
    public int size()
    {
        return map.size();
    }

    private CustomTextVariableMap map;
}
