/****************************************************************************
 *
 * File:            OutputOptions.java
 *
 * Description:     PDFTOOLS OutputOptions Class
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.sign;

import com.pdftools.sys.*;
import com.pdftools.internal.*;
import java.util.EnumSet;
import java.time.OffsetDateTime;
/**
 * <h1>Additional document level options</h1>
 * <p>
 * These options are available for all signature operations of the {@link Signer }.
 * They can also be used without a signature operation with the method {@link Signer#process }.</p>
 * <p>
 * Notes on document encryption when processing files with the {@link Signer }:
 * <ul>
 * <li>
 * PDF/A conformance is removed from input files.
 * In this case, a {@link Signer.WarningListener } with an {@link WarningCategory#PDF_A_REMOVED } is generated.
 * </li>
 * <li>
 * Signed documents cannot be encrypted or decrypted.
 * In this case, a {@link Signer.WarningListener } with an {@link WarningCategory#SIGNED_DOC_ENCRYPTION_UNCHANGED } is generated.
 * </li>
 * </ul></p>
 */
public class OutputOptions extends com.pdftools.pdf.OutputOptions 
{
    protected OutputOptions(long handle) 
    {
        super(handle);
    }

    /**
     * @hidden
     */
    public static OutputOptions createDynamicObject(long handle)
    {
        return new OutputOptions(handle);
    }


    /**
     * 
     */
    public OutputOptions()
    {
        this(newHelper());
    }

    private static long newHelper()
    {
        long handle = newNative();
        if (handle == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }

        return handle;
    }



     /**
     * <h1>Whether to remove any signatures (Getter)</h1>
     * <p>
     * By default, all signatures of the input document are preserved.
     * Optionally, some or all of them can be removed.</p>
     * <p>
     * Default: {@link SignatureRemoval#NONE }</p>
     */
    public com.pdftools.sign.SignatureRemoval getRemoveSignatures()
    {
        int retVal = getRemoveSignaturesNative(getHandle());
        if (retVal == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return com.pdftools.sign.SignatureRemoval.fromValue(retVal);
    }

     /**
     * <h1>Whether to remove any signatures (Setter)</h1>
     * <p>
     * By default, all signatures of the input document are preserved.
     * Optionally, some or all of them can be removed.</p>
     * <p>
     * Default: {@link SignatureRemoval#NONE }</p>
     * @throws IllegalArgumentException if {@code value} is {@code null}
     */
    public void setRemoveSignatures(com.pdftools.sign.SignatureRemoval value)
    {
        if (value == null)
            throw new IllegalArgumentException("Argument 'value' must not be null.", new NullPointerException("'value'"));
        boolean retVal = setRemoveSignaturesNative(getHandle(), value.getValue());
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
    }


     /**
     * <h1>Add validation information to existing signatures of input document (Getter)</h1>
     * <p>
     * Add signature validation information to the document security store (DSS).
     * This information includes:
     * <ul>
     * <li>
     * All certificates of the signing certificate’s trust chain, unless they are already embedded into the signature.</li>
     * <li>
     * Revocation data (OCSP or CRL) for all certificates that support revocation information.</li>
     * </ul>
     * This method can be used to create signatures with long-term validation material or to enlarge the longevity of existing signatures.
     * For more details on validation information, see also {@link com.pdftools.crypto.ValidationInformation pdftools.crypto.ValidationInformation}.</p>
     * <p>
     * Validation information for embedded time-stamp tokens is added as well.</p>
     * <p>
     * If adding validation information fails, an {@link Signer.WarningListener } with an
     * {@link WarningCategory#ADD_VALIDATION_INFORMATION_FAILED } is generated.</p>
     * <p>
     * All types of cryptographic providers support this method.
     * However, this method fails when using a provider whose certificate store is missing a required certificate.</p>
     * <p>
     * <em>Note:</em>
     * This property has no effect on any new signatures or time-stamp that may also be added.
     * The validation information of signatures and time-stamps is controlled by the respective property in the signature or time-stamp configuration object.</p>
     * <p>
     * <em>Note:</em>
     * This method does not validate the signatures, but only downloads the information required.</p>
     * <p>
     * <em>Note:</em>
     * Adding validation information for expired certificates is not possible.
     * Therefore, it is crucial to enlarge the longevity of signatures before they expire.</p>
     * <p>
     * <em>Note:</em>
     * Adding validation information to document certification (MDP) signatures is not possible,
     * because it would break the signature.
     * Validation information must be added to certification signatures when creating them.</p>
     * <p>
     * Default: {@link AddValidationInformation#NONE }</p>
     */
    public com.pdftools.sign.AddValidationInformation getAddValidationInformation()
    {
        int retVal = getAddValidationInformationNative(getHandle());
        if (retVal == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return com.pdftools.sign.AddValidationInformation.fromValue(retVal);
    }

     /**
     * <h1>Add validation information to existing signatures of input document (Setter)</h1>
     * <p>
     * Add signature validation information to the document security store (DSS).
     * This information includes:
     * <ul>
     * <li>
     * All certificates of the signing certificate’s trust chain, unless they are already embedded into the signature.</li>
     * <li>
     * Revocation data (OCSP or CRL) for all certificates that support revocation information.</li>
     * </ul>
     * This method can be used to create signatures with long-term validation material or to enlarge the longevity of existing signatures.
     * For more details on validation information, see also {@link com.pdftools.crypto.ValidationInformation pdftools.crypto.ValidationInformation}.</p>
     * <p>
     * Validation information for embedded time-stamp tokens is added as well.</p>
     * <p>
     * If adding validation information fails, an {@link Signer.WarningListener } with an
     * {@link WarningCategory#ADD_VALIDATION_INFORMATION_FAILED } is generated.</p>
     * <p>
     * All types of cryptographic providers support this method.
     * However, this method fails when using a provider whose certificate store is missing a required certificate.</p>
     * <p>
     * <em>Note:</em>
     * This property has no effect on any new signatures or time-stamp that may also be added.
     * The validation information of signatures and time-stamps is controlled by the respective property in the signature or time-stamp configuration object.</p>
     * <p>
     * <em>Note:</em>
     * This method does not validate the signatures, but only downloads the information required.</p>
     * <p>
     * <em>Note:</em>
     * Adding validation information for expired certificates is not possible.
     * Therefore, it is crucial to enlarge the longevity of signatures before they expire.</p>
     * <p>
     * <em>Note:</em>
     * Adding validation information to document certification (MDP) signatures is not possible,
     * because it would break the signature.
     * Validation information must be added to certification signatures when creating them.</p>
     * <p>
     * Default: {@link AddValidationInformation#NONE }</p>
     * @throws IllegalArgumentException if {@code value} is {@code null}
     */
    public void setAddValidationInformation(com.pdftools.sign.AddValidationInformation value)
    {
        if (value == null)
            throw new IllegalArgumentException("Argument 'value' must not be null.", new NullPointerException("'value'"));
        boolean retVal = setAddValidationInformationNative(getHandle(), value.getValue());
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
    }




    private static native long newNative();
    private native int getRemoveSignaturesNative(long handle);
    private native boolean setRemoveSignaturesNative(long handle, int value);
    private native int getAddValidationInformationNative(long handle);
    private native boolean setAddValidationInformationNative(long handle, int value);

}

