/****************************************************************************
 *
 * File:            SignatureFieldOptions.java
 *
 * Description:     PDFTOOLS SignatureFieldOptions Class
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.sign;

import com.pdftools.sys.*;
import com.pdftools.internal.*;
import java.util.EnumSet;
import java.time.OffsetDateTime;
/**
 * <h1>Options for adding unsigned signature fields</h1>
 * These options control the creation of unsigned signature fields in {@link Signer#addSignatureField }.
 */
public class SignatureFieldOptions extends NativeObject 
{
    protected SignatureFieldOptions(long handle) 
    {
        super(handle);
    }

    /**
     * @hidden
     */
    public static SignatureFieldOptions createDynamicObject(long handle)
    {
        return new SignatureFieldOptions(handle);
    }


    /**
     * 
     * @param boundingBox 
     * The bounding box of the signature field
     *
     * @throws IllegalArgumentException If the {@link com.pdftools.sign.SignatureFieldOptions# boundingBox} argument is {@code null} or not a valid bounding box
     * @throws IllegalArgumentException if {@code boundingBox} is {@code null}
     */
    public SignatureFieldOptions(com.pdftools.sign.Appearance boundingBox)
    {
        this(newHelper(boundingBox));
    }

    private static long newHelper(com.pdftools.sign.Appearance boundingBox)
    {
        if (boundingBox == null)
            throw new IllegalArgumentException("Argument 'boundingBox' must not be null.", new NullPointerException("'boundingBox'"));

        long handle = newNative(getHandle(boundingBox), boundingBox);
        if (handle == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                case 3: throw new IllegalArgumentException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }

        return handle;
    }



     /**
     * <h1>The bounding box of the signature field (Getter)</h1>
     * <p>
     * The bounding box is the area where the visual appearance of the signature is inserted, when the signature field is signed.</p>
     * <p>
     * Use {@link Appearance#createFieldBoundingBox } to create the bounding box object.
     * </p>
     */
    public com.pdftools.sign.Appearance getBoundingBox()
    {
        long retHandle = getBoundingBoxNative(getHandle());
        if (retHandle == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
        return com.pdftools.sign.Appearance.createDynamicObject(retHandle);

    }


     /**
     * <h1>The name of the new signature field (Getter)</h1>
     * <p>
     * If {@code null}, a new signature field is created using a unique field name.</p>
     * <p>
     * Default: {@code null}</p>
     */
    public String getFieldName()
    {
        String retVal = getFieldNameNative(getHandle());
        if (retVal == null)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }

     /**
     * <h1>The name of the new signature field (Setter)</h1>
     * <p>
     * If {@code null}, a new signature field is created using a unique field name.</p>
     * <p>
     * Default: {@code null}</p>
     */
    public void setFieldName(String value)
    {
        boolean retVal = setFieldNameNative(getHandle(), value);
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
    }




    private static native long newNative(long boundingBox, com.pdftools.sign.Appearance boundingBoxObj);
    private native long getBoundingBoxNative(long handle);
    private native String getFieldNameNative(long handle);
    private native boolean setFieldNameNative(long handle, String value);

}

