/****************************************************************************
 *
 * File:            Signer.java
 *
 * Description:     PDFTOOLS Signer Class
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.sign;

import com.pdftools.sys.*;
import com.pdftools.internal.*;
import java.util.EnumSet;
import java.time.OffsetDateTime;
import java.util.ArrayList;
import java.util.*;
/**
 * <h1>Process signatures and signature fields</h1>
 */
public class Signer extends NativeObject 
{
    protected Signer(long handle) 
    {
        super(handle);
    }

    /**
     * @hidden
     */
    public static Signer createDynamicObject(long handle)
    {
        return new Signer(handle);
    }

    /***
     * Listener interface for the {@link Warning} event.
     *
     */
    public interface WarningListener extends EventListener
    {
        void warning(Warning event);
    }

    /**
     * <h1>Event for non-critical errors occurring during signature processing</h1>
     */
    public class Warning extends EventObject
    {
        private static final long serialVersionUID = 726L;

        private String message;
        private com.pdftools.sign.WarningCategory category;
        private String context;

        private Warning(Object source, String message, com.pdftools.sign.WarningCategory category, String context) {
            super(source);
            this.message = message;
            this.category = category;
            this.context = context;
        }

        /**
         * The message describing the warning
         */
        public String getMessage() {
            return message;
        }

        /**
         * The category of the warning
         */
        public com.pdftools.sign.WarningCategory getCategory() {
            return category;
        }

        /**
         * A description of the context where the warning occurred
         */
        public String getContext() {
            return context;
        }


    }

    private Hashtable<WarningListener, WarningNativeClass> warningDic = new Hashtable<WarningListener, WarningNativeClass>();

    private class WarningNativeClass
    {
        private WarningListener listener;
        private long context;
        public WarningNativeClass(WarningListener listener)
        {
            this.listener = listener;
        }

        public void warningHandler(String message, com.pdftools.sign.WarningCategory category, String context)
        {
            Warning event = new Warning(this, message, category, context);
            try {
                this.listener.warning(event);
            }
            catch (Exception ex) { }
        }
    }

    /**
     * Add a listener for the {@link Warning} event.
     * @param listener Listener for the {@link Warning} event. 
     *                 If a listener is added that is already registered, it is ignored.
     */
    public void addWarningListener(WarningListener listener)
    {
        if(!warningDic.containsKey(listener))
        {
            WarningNativeClass eventNativeClass = new WarningNativeClass(listener);
            long context = addWarningHandlerNative(getHandle(), eventNativeClass);
            if (context == 0)
                throwLastRuntimeException();

            eventNativeClass.context = context;
            warningDic.put(listener, eventNativeClass);
        }
    }

    /**
     * Remove registered listener for the {@link Warning} event.
     * @param listener Listener for the {@link Warning} event that should be removed.
     *                 If the listener is not registered, it is ignored.
     */
    public void removeWarningListener(WarningListener listener)
    {
        if(warningDic.containsKey(listener))
        {
            if (!removeWarningHandlerNative(getHandle(), warningDic.get(listener).context))
            {
                if (getLastErrorCode() != 5)
                    throwLastRuntimeException();
            }
            warningDic.remove(listener);
        }
    }



    /**
     * 
     */
    public Signer()
    {
        this(newHelper());
    }

    private static long newHelper()
    {
        long handle = newNative();
        if (handle == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }

        return handle;
    }


    /**
     * <h1>Add a document signature</h1>
     * <p>
     * Document signatures are sometimes also called approval signatures.
     * This type of signature lets you verify the integrity of the signed part of the document and authenticate the signer’s identity.</p>
     * <p>
     * The features and format of the signature are defined by the {@link com.pdftools.crypto.providers.Provider pdftools.crypto.providers.Provider} and the {@link com.pdftools.sign.Signer#sign configuration}.</p>
     * <p>
     * Non-critical processing errors raise a {@link Signer.WarningListener }.
     * It is recommended to review the {@link WarningCategory } and handle them if necessary for the application.
     * </p>
     * @param document 
     * The input document to sign
     * @param configuration 
     * The signature configuration
     * @param stream 
     * The stream where the signed document is written
     * @return 
     * The signed document
     *
     * @throws com.pdftools.LicenseException The license check has failed.
     * @throws java.io.IOException Writing to the {@link com.pdftools.sign.Signer#sign stream} failed.
     * @throws com.pdftools.UnsupportedFeatureException The input PDF contains unrendered XFA form fields.
     *         See {@link com.pdftools.pdf.Document#getXfa pdftools.pdf.Document.getXfa} for more information on how to detect and handle XFA documents.
     * @throws IllegalArgumentException If the {@link com.pdftools.sign.Signer#sign configuration} is invalid, e.g. because the creating provider has been closed.
     * @throws IllegalArgumentException If the {@link com.pdftools.sign.Signer#sign configuration} is invalid, e.g. because it has been revoked.
     * @throws com.pdftools.NotFoundException If the {@link SignatureConfiguration#getFieldName } does not exist in {@link com.pdftools.sign.Signer#sign document}.
     * @throws com.pdftools.NotFoundException If an image, a PDF or a font file for the visual appearance could not be found.
     * @throws com.pdftools.RetryException If an unexpected error occurs that can be resolved by retrying the operation.
     *         For example, if a signature service returns an unexpectedly large signature.
     * @throws com.pdftools.RetryException If a resource required by the cryptographic provider is temporarily unavailable.
     * @throws com.pdftools.HttpException If a network error occurs, e.g. downloading revocation information (OCSP, CRL) or a time-stamp.
     * @throws com.pdftools.UnsupportedFeatureException If the cryptographic provider does not support the requested signing algorithm.
     * @throws com.pdftools.PermissionException If the cryptographic provider does not allow the signing operation.
     * @throws IllegalArgumentException if {@code document} is {@code null}
     * @throws IllegalArgumentException if {@code configuration} is {@code null}
     * @throws IllegalArgumentException if {@code stream} is {@code null}
     */
    public com.pdftools.pdf.Document sign(com.pdftools.pdf.Document document, com.pdftools.sign.SignatureConfiguration configuration, com.pdftools.sys.Stream stream) 
        throws 
            java.io.IOException,
            com.pdftools.NotFoundException,
            com.pdftools.LicenseException,
            com.pdftools.UnsupportedFeatureException,
            com.pdftools.PermissionException,
            com.pdftools.HttpException,

            com.pdftools.RetryException
    {
        if (document == null)
            throw new IllegalArgumentException("Argument 'document' must not be null.", new NullPointerException("'document'"));
        if (configuration == null)
            throw new IllegalArgumentException("Argument 'configuration' must not be null.", new NullPointerException("'configuration'"));
        if (stream == null)
            throw new IllegalArgumentException("Argument 'stream' must not be null.", new NullPointerException("'stream'"));

        return sign(document, configuration, stream, null);
    }

    /**
     * <h1>Add a document signature</h1>
     * <p>
     * Document signatures are sometimes also called approval signatures.
     * This type of signature lets you verify the integrity of the signed part of the document and authenticate the signer’s identity.</p>
     * <p>
     * The features and format of the signature are defined by the {@link com.pdftools.crypto.providers.Provider pdftools.crypto.providers.Provider} and the {@link com.pdftools.sign.Signer#sign configuration}.</p>
     * <p>
     * Non-critical processing errors raise a {@link Signer.WarningListener }.
     * It is recommended to review the {@link WarningCategory } and handle them if necessary for the application.
     * </p>
     * @param document 
     * The input document to sign
     * @param configuration 
     * The signature configuration
     * @param stream 
     * The stream where the signed document is written
     * @param outputOptions 
     * Document-level output options not directly related to the signature
     * @return 
     * The signed document
     *
     * @throws com.pdftools.LicenseException The license check has failed.
     * @throws java.io.IOException Writing to the {@link com.pdftools.sign.Signer#sign stream} failed.
     * @throws com.pdftools.UnsupportedFeatureException The input PDF contains unrendered XFA form fields.
     *         See {@link com.pdftools.pdf.Document#getXfa pdftools.pdf.Document.getXfa} for more information on how to detect and handle XFA documents.
     * @throws IllegalArgumentException If the {@link com.pdftools.sign.Signer#sign configuration} is invalid, e.g. because the creating provider has been closed.
     * @throws IllegalArgumentException If the {@link com.pdftools.sign.Signer#sign configuration} is invalid, e.g. because it has been revoked.
     * @throws com.pdftools.NotFoundException If the {@link SignatureConfiguration#getFieldName } does not exist in {@link com.pdftools.sign.Signer#sign document}.
     * @throws com.pdftools.NotFoundException If an image, a PDF or a font file for the visual appearance could not be found.
     * @throws com.pdftools.RetryException If an unexpected error occurs that can be resolved by retrying the operation.
     *         For example, if a signature service returns an unexpectedly large signature.
     * @throws com.pdftools.RetryException If a resource required by the cryptographic provider is temporarily unavailable.
     * @throws com.pdftools.HttpException If a network error occurs, e.g. downloading revocation information (OCSP, CRL) or a time-stamp.
     * @throws com.pdftools.UnsupportedFeatureException If the cryptographic provider does not support the requested signing algorithm.
     * @throws com.pdftools.PermissionException If the cryptographic provider does not allow the signing operation.
     * @throws IllegalArgumentException if {@code document} is {@code null}
     * @throws IllegalArgumentException if {@code configuration} is {@code null}
     * @throws IllegalArgumentException if {@code stream} is {@code null}
     */
    public com.pdftools.pdf.Document sign(com.pdftools.pdf.Document document, com.pdftools.sign.SignatureConfiguration configuration, com.pdftools.sys.Stream stream, com.pdftools.sign.OutputOptions outputOptions) 
        throws 
            java.io.IOException,
            com.pdftools.NotFoundException,
            com.pdftools.LicenseException,
            com.pdftools.UnsupportedFeatureException,
            com.pdftools.PermissionException,
            com.pdftools.HttpException,

            com.pdftools.RetryException
    {
        if (document == null)
            throw new IllegalArgumentException("Argument 'document' must not be null.", new NullPointerException("'document'"));
        if (configuration == null)
            throw new IllegalArgumentException("Argument 'configuration' must not be null.", new NullPointerException("'configuration'"));
        if (stream == null)
            throw new IllegalArgumentException("Argument 'stream' must not be null.", new NullPointerException("'stream'"));

        long retHandle = signNative(getHandle(), getHandle(document), document, getHandle(configuration), configuration, stream, getHandle(outputOptions), outputOptions);

        if (retHandle == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                case 3: throw new IllegalArgumentException(getLastErrorMessage());
                case 4: throw new java.io.IOException(getLastErrorMessage());
                case 5: throw new com.pdftools.NotFoundException(getLastErrorMessage());
                case 12: throw new com.pdftools.LicenseException(getLastErrorMessage());
                case 19: throw new com.pdftools.UnsupportedFeatureException(getLastErrorMessage());
                case 23: throw new com.pdftools.PermissionException(getLastErrorMessage());
                case 24: throw new com.pdftools.HttpException(getLastErrorMessage());
                case 25: throw new com.pdftools.RetryException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }

        return com.pdftools.pdf.Document.createDynamicObject(retHandle);
    }

    /**
     * <h1>Add a document certification signature</h1>
     * <p>
     * This type of signature lets you detect rejected changes specified by the author.
     * These signatures are also called Modification Detection and Prevention (MDP) signatures.
     * The allowed permissions are defined by {@link com.pdftools.sign.Signer#certify permissions}.</p>
     * <p>
     * The features and format of the signature are defined by the {@link com.pdftools.crypto.providers.Provider pdftools.crypto.providers.Provider} and the
     * {@link com.pdftools.sign.Signer#certify configuration}.</p>
     * <p>
     * Non-critical processing errors raise a {@link Signer.WarningListener }.
     * It is recommended to review the {@link WarningCategory } and handle them if necessary for the application.
     * </p>
     * @param document 
     * The input document to certify
     * @param configuration 
     * The signature configuration
     * @param stream 
     * The stream where the certified document is written
     * @return 
     *
     * @throws com.pdftools.LicenseException The license check has failed.
     * @throws java.io.IOException Writing to the {@link com.pdftools.sign.Signer#certify stream} failed.
     * @throws com.pdftools.UnsupportedFeatureException The input PDF contains unrendered XFA form fields.
     *         See {@link com.pdftools.pdf.Document#getXfa pdftools.pdf.Document.getXfa} for more information on how to detect and handle XFA documents.
     * @throws IllegalArgumentException If the {@link com.pdftools.sign.Signer#certify configuration} is invalid, e.g. because the creating provider has been closed.
     * @throws com.pdftools.NotFoundException If the {@link SignatureConfiguration#getFieldName } does not exist in {@link com.pdftools.sign.Signer#certify document}.
     * @throws com.pdftools.RetryException If an unexpected error occurs that can be resolved by retrying the operation.
     *         For example, if a signature service returns an unexpectedly large signature.
     * @throws com.pdftools.RetryException If a resource required by the cryptographic provider is temporarily unavailable.
     * @throws com.pdftools.HttpException If a network error occurs, e.g. downloading revocation information (OCSP, CRL) or a time-stamp.
     * @throws com.pdftools.UnsupportedFeatureException If the cryptographic provider does not support the requested signing algorithm.
     * @throws com.pdftools.PermissionException If the cryptographic provider does not allow the signing operation.
     * @throws IllegalArgumentException if {@code document} is {@code null}
     * @throws IllegalArgumentException if {@code configuration} is {@code null}
     * @throws IllegalArgumentException if {@code stream} is {@code null}
     */
    public com.pdftools.pdf.Document certify(com.pdftools.pdf.Document document, com.pdftools.sign.SignatureConfiguration configuration, com.pdftools.sys.Stream stream) 
        throws 
            java.io.IOException,
            com.pdftools.NotFoundException,
            com.pdftools.LicenseException,
            com.pdftools.UnsupportedFeatureException,
            com.pdftools.PermissionException,
            com.pdftools.HttpException,

            com.pdftools.RetryException
    {
        if (document == null)
            throw new IllegalArgumentException("Argument 'document' must not be null.", new NullPointerException("'document'"));
        if (configuration == null)
            throw new IllegalArgumentException("Argument 'configuration' must not be null.", new NullPointerException("'configuration'"));
        if (stream == null)
            throw new IllegalArgumentException("Argument 'stream' must not be null.", new NullPointerException("'stream'"));

        return certify(document, configuration, stream, null, null);
    }

    /**
     * <h1>Add a document certification signature</h1>
     * <p>
     * This type of signature lets you detect rejected changes specified by the author.
     * These signatures are also called Modification Detection and Prevention (MDP) signatures.
     * The allowed permissions are defined by {@link com.pdftools.sign.Signer#certify permissions}.</p>
     * <p>
     * The features and format of the signature are defined by the {@link com.pdftools.crypto.providers.Provider pdftools.crypto.providers.Provider} and the
     * {@link com.pdftools.sign.Signer#certify configuration}.</p>
     * <p>
     * Non-critical processing errors raise a {@link Signer.WarningListener }.
     * It is recommended to review the {@link WarningCategory } and handle them if necessary for the application.
     * </p>
     * @param document 
     * The input document to certify
     * @param configuration 
     * The signature configuration
     * @param stream 
     * The stream where the certified document is written
     * @param permissions 
     * The permissions allowed. The default is {@link com.pdftools.pdf.MdpPermissions#NO_CHANGES pdftools.pdf.MdpPermissions.NO_CHANGES}.
     * @return 
     *
     * @throws com.pdftools.LicenseException The license check has failed.
     * @throws java.io.IOException Writing to the {@link com.pdftools.sign.Signer#certify stream} failed.
     * @throws com.pdftools.UnsupportedFeatureException The input PDF contains unrendered XFA form fields.
     *         See {@link com.pdftools.pdf.Document#getXfa pdftools.pdf.Document.getXfa} for more information on how to detect and handle XFA documents.
     * @throws IllegalArgumentException If the {@link com.pdftools.sign.Signer#certify configuration} is invalid, e.g. because the creating provider has been closed.
     * @throws com.pdftools.NotFoundException If the {@link SignatureConfiguration#getFieldName } does not exist in {@link com.pdftools.sign.Signer#certify document}.
     * @throws com.pdftools.RetryException If an unexpected error occurs that can be resolved by retrying the operation.
     *         For example, if a signature service returns an unexpectedly large signature.
     * @throws com.pdftools.RetryException If a resource required by the cryptographic provider is temporarily unavailable.
     * @throws com.pdftools.HttpException If a network error occurs, e.g. downloading revocation information (OCSP, CRL) or a time-stamp.
     * @throws com.pdftools.UnsupportedFeatureException If the cryptographic provider does not support the requested signing algorithm.
     * @throws com.pdftools.PermissionException If the cryptographic provider does not allow the signing operation.
     * @throws IllegalArgumentException if {@code document} is {@code null}
     * @throws IllegalArgumentException if {@code configuration} is {@code null}
     * @throws IllegalArgumentException if {@code stream} is {@code null}
     */
    public com.pdftools.pdf.Document certify(com.pdftools.pdf.Document document, com.pdftools.sign.SignatureConfiguration configuration, com.pdftools.sys.Stream stream, com.pdftools.sign.MdpPermissionOptions permissions) 
        throws 
            java.io.IOException,
            com.pdftools.NotFoundException,
            com.pdftools.LicenseException,
            com.pdftools.UnsupportedFeatureException,
            com.pdftools.PermissionException,
            com.pdftools.HttpException,

            com.pdftools.RetryException
    {
        if (document == null)
            throw new IllegalArgumentException("Argument 'document' must not be null.", new NullPointerException("'document'"));
        if (configuration == null)
            throw new IllegalArgumentException("Argument 'configuration' must not be null.", new NullPointerException("'configuration'"));
        if (stream == null)
            throw new IllegalArgumentException("Argument 'stream' must not be null.", new NullPointerException("'stream'"));

        return certify(document, configuration, stream, permissions, null);
    }

    /**
     * <h1>Add a document certification signature</h1>
     * <p>
     * This type of signature lets you detect rejected changes specified by the author.
     * These signatures are also called Modification Detection and Prevention (MDP) signatures.
     * The allowed permissions are defined by {@link com.pdftools.sign.Signer#certify permissions}.</p>
     * <p>
     * The features and format of the signature are defined by the {@link com.pdftools.crypto.providers.Provider pdftools.crypto.providers.Provider} and the
     * {@link com.pdftools.sign.Signer#certify configuration}.</p>
     * <p>
     * Non-critical processing errors raise a {@link Signer.WarningListener }.
     * It is recommended to review the {@link WarningCategory } and handle them if necessary for the application.
     * </p>
     * @param document 
     * The input document to certify
     * @param configuration 
     * The signature configuration
     * @param stream 
     * The stream where the certified document is written
     * @param permissions 
     * The permissions allowed. The default is {@link com.pdftools.pdf.MdpPermissions#NO_CHANGES pdftools.pdf.MdpPermissions.NO_CHANGES}.
     * @param outputOptions 
     * Document-level output options not directly related to the document certification
     * @return 
     *
     * @throws com.pdftools.LicenseException The license check has failed.
     * @throws java.io.IOException Writing to the {@link com.pdftools.sign.Signer#certify stream} failed.
     * @throws com.pdftools.UnsupportedFeatureException The input PDF contains unrendered XFA form fields.
     *         See {@link com.pdftools.pdf.Document#getXfa pdftools.pdf.Document.getXfa} for more information on how to detect and handle XFA documents.
     * @throws IllegalArgumentException If the {@link com.pdftools.sign.Signer#certify configuration} is invalid, e.g. because the creating provider has been closed.
     * @throws com.pdftools.NotFoundException If the {@link SignatureConfiguration#getFieldName } does not exist in {@link com.pdftools.sign.Signer#certify document}.
     * @throws com.pdftools.RetryException If an unexpected error occurs that can be resolved by retrying the operation.
     *         For example, if a signature service returns an unexpectedly large signature.
     * @throws com.pdftools.RetryException If a resource required by the cryptographic provider is temporarily unavailable.
     * @throws com.pdftools.HttpException If a network error occurs, e.g. downloading revocation information (OCSP, CRL) or a time-stamp.
     * @throws com.pdftools.UnsupportedFeatureException If the cryptographic provider does not support the requested signing algorithm.
     * @throws com.pdftools.PermissionException If the cryptographic provider does not allow the signing operation.
     * @throws IllegalArgumentException if {@code document} is {@code null}
     * @throws IllegalArgumentException if {@code configuration} is {@code null}
     * @throws IllegalArgumentException if {@code stream} is {@code null}
     */
    public com.pdftools.pdf.Document certify(com.pdftools.pdf.Document document, com.pdftools.sign.SignatureConfiguration configuration, com.pdftools.sys.Stream stream, com.pdftools.sign.MdpPermissionOptions permissions, com.pdftools.sign.OutputOptions outputOptions) 
        throws 
            java.io.IOException,
            com.pdftools.NotFoundException,
            com.pdftools.LicenseException,
            com.pdftools.UnsupportedFeatureException,
            com.pdftools.PermissionException,
            com.pdftools.HttpException,

            com.pdftools.RetryException
    {
        if (document == null)
            throw new IllegalArgumentException("Argument 'document' must not be null.", new NullPointerException("'document'"));
        if (configuration == null)
            throw new IllegalArgumentException("Argument 'configuration' must not be null.", new NullPointerException("'configuration'"));
        if (stream == null)
            throw new IllegalArgumentException("Argument 'stream' must not be null.", new NullPointerException("'stream'"));

        long retHandle = certifyNative(getHandle(), getHandle(document), document, getHandle(configuration), configuration, stream, getHandle(permissions), permissions, getHandle(outputOptions), outputOptions);

        if (retHandle == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                case 3: throw new IllegalArgumentException(getLastErrorMessage());
                case 4: throw new java.io.IOException(getLastErrorMessage());
                case 5: throw new com.pdftools.NotFoundException(getLastErrorMessage());
                case 12: throw new com.pdftools.LicenseException(getLastErrorMessage());
                case 19: throw new com.pdftools.UnsupportedFeatureException(getLastErrorMessage());
                case 23: throw new com.pdftools.PermissionException(getLastErrorMessage());
                case 24: throw new com.pdftools.HttpException(getLastErrorMessage());
                case 25: throw new com.pdftools.RetryException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }

        return com.pdftools.pdf.Document.createDynamicObject(retHandle);
    }

    /**
     * <h1>Add a document time-stamp</h1>
     * <p>
     * This type of signature provides evidence that the document existed at a specific time and protects the document’s integrity.</p>
     * <p>
     * The features and format of the signature are defined by the {@link com.pdftools.crypto.providers.Provider pdftools.crypto.providers.Provider} and the
     * {@link com.pdftools.sign.Signer#addTimestamp configuration}.</p>
     * <p>
     * Non-critical processing errors raise a {@link Signer.WarningListener }.
     * It is recommended to review the {@link WarningCategory } and handle them if necessary for the application.
     * </p>
     * @param document 
     * The input document to add a time-stamp to
     * @param configuration 
     * The time-stamp configuration
     * @param stream 
     * The stream where the output document is written
     * @return 
     *
     * @throws com.pdftools.LicenseException The license check has failed.
     * @throws java.io.IOException Writing to the {@link com.pdftools.sign.Signer#addTimestamp stream} failed.
     * @throws com.pdftools.UnsupportedFeatureException The input PDF contains unrendered XFA form fields.
     *         See {@link com.pdftools.pdf.Document#getXfa pdftools.pdf.Document.getXfa} for more information on how to detect and handle XFA documents.
     * @throws IllegalArgumentException If the {@link com.pdftools.sign.Signer#addTimestamp configuration} is invalid, e.g. because the creating provider has been closed.
     * @throws com.pdftools.NotFoundException If the {@link SignatureConfiguration#getFieldName } does not exist in {@link com.pdftools.sign.Signer#addTimestamp document}.
     * @throws com.pdftools.RetryException If an unexpected error occurs that can be resolved by retrying the operation.
     *         For example, if a signature service returns an unexpectedly large signature.
     * @throws com.pdftools.RetryException If a resource required by the cryptographic provider is temporarily unavailable.
     * @throws com.pdftools.HttpException If a network error occurs, e.g. downloading revocation information (OCSP, CRL) or a time-stamp.
     * @throws com.pdftools.UnsupportedFeatureException If the cryptographic provider does not support the requested signing algorithm.
     * @throws com.pdftools.PermissionException If the cryptographic provider does not allow the signing operation.
     * @throws IllegalArgumentException if {@code document} is {@code null}
     * @throws IllegalArgumentException if {@code configuration} is {@code null}
     * @throws IllegalArgumentException if {@code stream} is {@code null}
     */
    public com.pdftools.pdf.Document addTimestamp(com.pdftools.pdf.Document document, com.pdftools.sign.TimestampConfiguration configuration, com.pdftools.sys.Stream stream) 
        throws 
            java.io.IOException,
            com.pdftools.NotFoundException,
            com.pdftools.LicenseException,
            com.pdftools.UnsupportedFeatureException,
            com.pdftools.PermissionException,
            com.pdftools.HttpException,

            com.pdftools.RetryException
    {
        if (document == null)
            throw new IllegalArgumentException("Argument 'document' must not be null.", new NullPointerException("'document'"));
        if (configuration == null)
            throw new IllegalArgumentException("Argument 'configuration' must not be null.", new NullPointerException("'configuration'"));
        if (stream == null)
            throw new IllegalArgumentException("Argument 'stream' must not be null.", new NullPointerException("'stream'"));

        return addTimestamp(document, configuration, stream, null);
    }

    /**
     * <h1>Add a document time-stamp</h1>
     * <p>
     * This type of signature provides evidence that the document existed at a specific time and protects the document’s integrity.</p>
     * <p>
     * The features and format of the signature are defined by the {@link com.pdftools.crypto.providers.Provider pdftools.crypto.providers.Provider} and the
     * {@link com.pdftools.sign.Signer#addTimestamp configuration}.</p>
     * <p>
     * Non-critical processing errors raise a {@link Signer.WarningListener }.
     * It is recommended to review the {@link WarningCategory } and handle them if necessary for the application.
     * </p>
     * @param document 
     * The input document to add a time-stamp to
     * @param configuration 
     * The time-stamp configuration
     * @param stream 
     * The stream where the output document is written
     * @param outputOptions 
     * Document-level output options not directly related to the document time-stamp
     * @return 
     *
     * @throws com.pdftools.LicenseException The license check has failed.
     * @throws java.io.IOException Writing to the {@link com.pdftools.sign.Signer#addTimestamp stream} failed.
     * @throws com.pdftools.UnsupportedFeatureException The input PDF contains unrendered XFA form fields.
     *         See {@link com.pdftools.pdf.Document#getXfa pdftools.pdf.Document.getXfa} for more information on how to detect and handle XFA documents.
     * @throws IllegalArgumentException If the {@link com.pdftools.sign.Signer#addTimestamp configuration} is invalid, e.g. because the creating provider has been closed.
     * @throws com.pdftools.NotFoundException If the {@link SignatureConfiguration#getFieldName } does not exist in {@link com.pdftools.sign.Signer#addTimestamp document}.
     * @throws com.pdftools.RetryException If an unexpected error occurs that can be resolved by retrying the operation.
     *         For example, if a signature service returns an unexpectedly large signature.
     * @throws com.pdftools.RetryException If a resource required by the cryptographic provider is temporarily unavailable.
     * @throws com.pdftools.HttpException If a network error occurs, e.g. downloading revocation information (OCSP, CRL) or a time-stamp.
     * @throws com.pdftools.UnsupportedFeatureException If the cryptographic provider does not support the requested signing algorithm.
     * @throws com.pdftools.PermissionException If the cryptographic provider does not allow the signing operation.
     * @throws IllegalArgumentException if {@code document} is {@code null}
     * @throws IllegalArgumentException if {@code configuration} is {@code null}
     * @throws IllegalArgumentException if {@code stream} is {@code null}
     */
    public com.pdftools.pdf.Document addTimestamp(com.pdftools.pdf.Document document, com.pdftools.sign.TimestampConfiguration configuration, com.pdftools.sys.Stream stream, com.pdftools.sign.OutputOptions outputOptions) 
        throws 
            java.io.IOException,
            com.pdftools.NotFoundException,
            com.pdftools.LicenseException,
            com.pdftools.UnsupportedFeatureException,
            com.pdftools.PermissionException,
            com.pdftools.HttpException,

            com.pdftools.RetryException
    {
        if (document == null)
            throw new IllegalArgumentException("Argument 'document' must not be null.", new NullPointerException("'document'"));
        if (configuration == null)
            throw new IllegalArgumentException("Argument 'configuration' must not be null.", new NullPointerException("'configuration'"));
        if (stream == null)
            throw new IllegalArgumentException("Argument 'stream' must not be null.", new NullPointerException("'stream'"));

        long retHandle = addTimestampNative(getHandle(), getHandle(document), document, getHandle(configuration), configuration, stream, getHandle(outputOptions), outputOptions);

        if (retHandle == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                case 3: throw new IllegalArgumentException(getLastErrorMessage());
                case 4: throw new java.io.IOException(getLastErrorMessage());
                case 5: throw new com.pdftools.NotFoundException(getLastErrorMessage());
                case 12: throw new com.pdftools.LicenseException(getLastErrorMessage());
                case 19: throw new com.pdftools.UnsupportedFeatureException(getLastErrorMessage());
                case 23: throw new com.pdftools.PermissionException(getLastErrorMessage());
                case 24: throw new com.pdftools.HttpException(getLastErrorMessage());
                case 25: throw new com.pdftools.RetryException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }

        return com.pdftools.pdf.Document.createDynamicObject(retHandle);
    }

    /**
     * <h1>Add an unsigned signature field</h1>
     * <p>
     * Add an unsigned signature field that can later be signed (see {@link com.pdftools.pdf.UnsignedSignatureField pdftools.pdf.UnsignedSignatureField}).</p>
     * <p>
     * Non-critical processing errors raise a {@link Signer.WarningListener }.
     * It is recommended to review the {@link WarningCategory } and handle them if necessary for the application.
     * </p>
     * @param document 
     * The input document to add the signature field to
     * @param options 
     * The options for the unsigned signature field
     * @param stream 
     * The stream where the output document is written
     * @return 
     *
     * @throws com.pdftools.LicenseException The license check has failed.
     * @throws java.io.IOException Writing to the {@link com.pdftools.sign.Signer#addSignatureField stream} failed.
     * @throws com.pdftools.UnsupportedFeatureException The input PDF contains unrendered XFA form fields.
     *         See {@link com.pdftools.pdf.Document#getXfa pdftools.pdf.Document.getXfa} for more information on how to detect and handle XFA documents.
     * @throws com.pdftools.ExistsException The {@link SignatureFieldOptions#getFieldName } exists already in the input document.
     * @throws IllegalArgumentException if {@code document} is {@code null}
     * @throws IllegalArgumentException if {@code options} is {@code null}
     * @throws IllegalArgumentException if {@code stream} is {@code null}
     */
    public com.pdftools.pdf.Document addSignatureField(com.pdftools.pdf.Document document, com.pdftools.sign.SignatureFieldOptions options, com.pdftools.sys.Stream stream) 
        throws 
            java.io.IOException,
            com.pdftools.LicenseException,
            com.pdftools.UnsupportedFeatureException,

            com.pdftools.ExistsException
    {
        if (document == null)
            throw new IllegalArgumentException("Argument 'document' must not be null.", new NullPointerException("'document'"));
        if (options == null)
            throw new IllegalArgumentException("Argument 'options' must not be null.", new NullPointerException("'options'"));
        if (stream == null)
            throw new IllegalArgumentException("Argument 'stream' must not be null.", new NullPointerException("'stream'"));

        return addSignatureField(document, options, stream, null);
    }

    /**
     * <h1>Add an unsigned signature field</h1>
     * <p>
     * Add an unsigned signature field that can later be signed (see {@link com.pdftools.pdf.UnsignedSignatureField pdftools.pdf.UnsignedSignatureField}).</p>
     * <p>
     * Non-critical processing errors raise a {@link Signer.WarningListener }.
     * It is recommended to review the {@link WarningCategory } and handle them if necessary for the application.
     * </p>
     * @param document 
     * The input document to add the signature field to
     * @param options 
     * The options for the unsigned signature field
     * @param stream 
     * The stream where the output document is written
     * @param outputOptions 
     * Document-level output options not directly related to the signature field
     * @return 
     *
     * @throws com.pdftools.LicenseException The license check has failed.
     * @throws java.io.IOException Writing to the {@link com.pdftools.sign.Signer#addSignatureField stream} failed.
     * @throws com.pdftools.UnsupportedFeatureException The input PDF contains unrendered XFA form fields.
     *         See {@link com.pdftools.pdf.Document#getXfa pdftools.pdf.Document.getXfa} for more information on how to detect and handle XFA documents.
     * @throws com.pdftools.ExistsException The {@link SignatureFieldOptions#getFieldName } exists already in the input document.
     * @throws IllegalArgumentException if {@code document} is {@code null}
     * @throws IllegalArgumentException if {@code options} is {@code null}
     * @throws IllegalArgumentException if {@code stream} is {@code null}
     */
    public com.pdftools.pdf.Document addSignatureField(com.pdftools.pdf.Document document, com.pdftools.sign.SignatureFieldOptions options, com.pdftools.sys.Stream stream, com.pdftools.sign.OutputOptions outputOptions) 
        throws 
            java.io.IOException,
            com.pdftools.LicenseException,
            com.pdftools.UnsupportedFeatureException,

            com.pdftools.ExistsException
    {
        if (document == null)
            throw new IllegalArgumentException("Argument 'document' must not be null.", new NullPointerException("'document'"));
        if (options == null)
            throw new IllegalArgumentException("Argument 'options' must not be null.", new NullPointerException("'options'"));
        if (stream == null)
            throw new IllegalArgumentException("Argument 'stream' must not be null.", new NullPointerException("'stream'"));

        long retHandle = addSignatureFieldNative(getHandle(), getHandle(document), document, getHandle(options), options, stream, getHandle(outputOptions), outputOptions);

        if (retHandle == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                case 4: throw new java.io.IOException(getLastErrorMessage());
                case 12: throw new com.pdftools.LicenseException(getLastErrorMessage());
                case 19: throw new com.pdftools.UnsupportedFeatureException(getLastErrorMessage());
                case 22: throw new com.pdftools.ExistsException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }

        return com.pdftools.pdf.Document.createDynamicObject(retHandle);
    }

    /**
     * <h1>Add a prepared signature</h1>
     * <p>
     * Adding a prepared signature is only required in very particular or specialized use cases.
     * This method is the same as {@link Signer#sign }, but without actually creating the cryptographic signature.
     * The cryptographic signature can be inserted later using {@link Signer#signPreparedSignature }.</p>
     * <p>
     * While the {@link com.pdftools.sign.Signer#addPreparedSignature configuration} can be created by any {@link com.pdftools.crypto.providers.Provider pdftools.crypto.providers.Provider},
     * it is typically created by {@link com.pdftools.crypto.providers.builtin.Provider#createPreparedSignature pdftools.crypto.providers.builtin.Provider.createPreparedSignature}.
     * </p>
     * @param document 
     * The input document to add the prepared signature
     * @param configuration 
     * The signature configuration
     * @param stream 
     * The stream where the output document is written
     * @return 
     *
     * @throws com.pdftools.LicenseException The license check has failed.
     * @throws java.io.IOException Writing to the {@link com.pdftools.sign.Signer#addPreparedSignature stream} failed.
     * @throws com.pdftools.UnsupportedFeatureException The input PDF contains unrendered XFA form fields.
     *         See {@link com.pdftools.pdf.Document#getXfa pdftools.pdf.Document.getXfa} for more information on how to detect and handle XFA documents.
     * @throws IllegalArgumentException If the {@link com.pdftools.sign.Signer#addPreparedSignature configuration} is invalid, e.g. because the creating provider has been closed
     * @throws com.pdftools.NotFoundException If the {@link SignatureConfiguration#getFieldName } does not exist in {@link com.pdftools.sign.Signer#addPreparedSignature document}.
     * @throws com.pdftools.UnsupportedFeatureException If the cryptographic provider does not support the requested signing algorithm.
     * @throws com.pdftools.HttpException If a network error occurs, e.g. downloading revocation information (OCSP, CRL).
     * @throws IllegalArgumentException if {@code document} is {@code null}
     * @throws IllegalArgumentException if {@code configuration} is {@code null}
     * @throws IllegalArgumentException if {@code stream} is {@code null}
     */
    public com.pdftools.sign.PreparedDocument addPreparedSignature(com.pdftools.pdf.Document document, com.pdftools.sign.SignatureConfiguration configuration, com.pdftools.sys.Stream stream) 
        throws 
            java.io.IOException,
            com.pdftools.NotFoundException,
            com.pdftools.LicenseException,
            com.pdftools.UnsupportedFeatureException,

            com.pdftools.HttpException
    {
        if (document == null)
            throw new IllegalArgumentException("Argument 'document' must not be null.", new NullPointerException("'document'"));
        if (configuration == null)
            throw new IllegalArgumentException("Argument 'configuration' must not be null.", new NullPointerException("'configuration'"));
        if (stream == null)
            throw new IllegalArgumentException("Argument 'stream' must not be null.", new NullPointerException("'stream'"));

        return addPreparedSignature(document, configuration, stream, null);
    }

    /**
     * <h1>Add a prepared signature</h1>
     * <p>
     * Adding a prepared signature is only required in very particular or specialized use cases.
     * This method is the same as {@link Signer#sign }, but without actually creating the cryptographic signature.
     * The cryptographic signature can be inserted later using {@link Signer#signPreparedSignature }.</p>
     * <p>
     * While the {@link com.pdftools.sign.Signer#addPreparedSignature configuration} can be created by any {@link com.pdftools.crypto.providers.Provider pdftools.crypto.providers.Provider},
     * it is typically created by {@link com.pdftools.crypto.providers.builtin.Provider#createPreparedSignature pdftools.crypto.providers.builtin.Provider.createPreparedSignature}.
     * </p>
     * @param document 
     * The input document to add the prepared signature
     * @param configuration 
     * The signature configuration
     * @param stream 
     * The stream where the output document is written
     * @param outputOptions 
     * Document-level output options not directly related to preparing the signature
     * @return 
     *
     * @throws com.pdftools.LicenseException The license check has failed.
     * @throws java.io.IOException Writing to the {@link com.pdftools.sign.Signer#addPreparedSignature stream} failed.
     * @throws com.pdftools.UnsupportedFeatureException The input PDF contains unrendered XFA form fields.
     *         See {@link com.pdftools.pdf.Document#getXfa pdftools.pdf.Document.getXfa} for more information on how to detect and handle XFA documents.
     * @throws IllegalArgumentException If the {@link com.pdftools.sign.Signer#addPreparedSignature configuration} is invalid, e.g. because the creating provider has been closed
     * @throws com.pdftools.NotFoundException If the {@link SignatureConfiguration#getFieldName } does not exist in {@link com.pdftools.sign.Signer#addPreparedSignature document}.
     * @throws com.pdftools.UnsupportedFeatureException If the cryptographic provider does not support the requested signing algorithm.
     * @throws com.pdftools.HttpException If a network error occurs, e.g. downloading revocation information (OCSP, CRL).
     * @throws IllegalArgumentException if {@code document} is {@code null}
     * @throws IllegalArgumentException if {@code configuration} is {@code null}
     * @throws IllegalArgumentException if {@code stream} is {@code null}
     */
    public com.pdftools.sign.PreparedDocument addPreparedSignature(com.pdftools.pdf.Document document, com.pdftools.sign.SignatureConfiguration configuration, com.pdftools.sys.Stream stream, com.pdftools.sign.OutputOptions outputOptions) 
        throws 
            java.io.IOException,
            com.pdftools.NotFoundException,
            com.pdftools.LicenseException,
            com.pdftools.UnsupportedFeatureException,

            com.pdftools.HttpException
    {
        if (document == null)
            throw new IllegalArgumentException("Argument 'document' must not be null.", new NullPointerException("'document'"));
        if (configuration == null)
            throw new IllegalArgumentException("Argument 'configuration' must not be null.", new NullPointerException("'configuration'"));
        if (stream == null)
            throw new IllegalArgumentException("Argument 'stream' must not be null.", new NullPointerException("'stream'"));

        long retHandle = addPreparedSignatureNative(getHandle(), getHandle(document), document, getHandle(configuration), configuration, stream, getHandle(outputOptions), outputOptions);

        if (retHandle == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                case 3: throw new IllegalArgumentException(getLastErrorMessage());
                case 4: throw new java.io.IOException(getLastErrorMessage());
                case 5: throw new com.pdftools.NotFoundException(getLastErrorMessage());
                case 12: throw new com.pdftools.LicenseException(getLastErrorMessage());
                case 19: throw new com.pdftools.UnsupportedFeatureException(getLastErrorMessage());
                case 24: throw new com.pdftools.HttpException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }

        return com.pdftools.sign.PreparedDocument.createDynamicObject(retHandle);
    }

    /**
     * <h1>Sign a prepared signature</h1>
     * Sign a document that contains a prepared signature created using {@link Signer#addPreparedSignature }.
     * Note that the {@link com.pdftools.sign.Signer#signPreparedSignature configuration} must be compatible to the configuration used when preparing the signature.
     * @param document 
     * The input document to sign
     * @param configuration 
     * The signature configuration
     * @param stream 
     * The stream where the signed document is written
     * @return 
     *
     * @throws com.pdftools.LicenseException The license check has failed.
     * @throws java.io.IOException Writing to the {@link com.pdftools.sign.Signer#signPreparedSignature stream} failed.
     * @throws IllegalArgumentException If the {@link com.pdftools.sign.Signer#signPreparedSignature document} does not contain a prepared signature created by {@link Signer#addPreparedSignature }
     * @throws IllegalArgumentException If the {@link com.pdftools.sign.Signer#signPreparedSignature configuration} is invalid, e.g. because the creating provider has been closed.
     * @throws com.pdftools.RetryException If an unexpected error occurs that can be resolved by retrying the operation.
     *         For example, if a signature service returns an unexpectedly large signature.
     * @throws com.pdftools.RetryException If a resource required by the cryptographic provider is temporarily unavailable.
     * @throws com.pdftools.HttpException If a network error occurs, e.g. downloading revocation information (OCSP, CRL) or a time-stamp.
     * @throws com.pdftools.UnsupportedFeatureException If the cryptographic provider does not support the requested signing algorithm.
     * @throws com.pdftools.PermissionException If the cryptographic provider does not allow the signing operation.
     * @throws IllegalArgumentException if {@code document} is {@code null}
     * @throws IllegalArgumentException if {@code configuration} is {@code null}
     * @throws IllegalArgumentException if {@code stream} is {@code null}
     */
    public com.pdftools.pdf.Document signPreparedSignature(com.pdftools.pdf.Document document, com.pdftools.sign.SignatureConfiguration configuration, com.pdftools.sys.Stream stream) 
        throws 
            java.io.IOException,
            com.pdftools.LicenseException,
            com.pdftools.UnsupportedFeatureException,
            com.pdftools.PermissionException,
            com.pdftools.HttpException,

            com.pdftools.RetryException
    {
        if (document == null)
            throw new IllegalArgumentException("Argument 'document' must not be null.", new NullPointerException("'document'"));
        if (configuration == null)
            throw new IllegalArgumentException("Argument 'configuration' must not be null.", new NullPointerException("'configuration'"));
        if (stream == null)
            throw new IllegalArgumentException("Argument 'stream' must not be null.", new NullPointerException("'stream'"));

        long retHandle = signPreparedSignatureNative(getHandle(), getHandle(document), document, getHandle(configuration), configuration, stream);

        if (retHandle == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                case 3: throw new IllegalArgumentException(getLastErrorMessage());
                case 4: throw new java.io.IOException(getLastErrorMessage());
                case 12: throw new com.pdftools.LicenseException(getLastErrorMessage());
                case 19: throw new com.pdftools.UnsupportedFeatureException(getLastErrorMessage());
                case 23: throw new com.pdftools.PermissionException(getLastErrorMessage());
                case 24: throw new com.pdftools.HttpException(getLastErrorMessage());
                case 25: throw new com.pdftools.RetryException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }

        return com.pdftools.pdf.Document.createDynamicObject(retHandle);
    }

    /**
     * <h1>Process a document</h1>
     * <p>
     * Apply document-level processing options without any signature operation.
     * For example:
     * <ul>
     * <li>
     * To encrypt or decrypt PDF documents that may be signed (see the samples "Encrypt" and "Decrypt").
     * </li>
     * <li>
     * To remove signatures and unsigned signature fields (see {@link OutputOptions#getRemoveSignatures }).
     * </li>
     * <li>
     * To add validation information to existing signatures (see {@link OutputOptions#getAddValidationInformation }).
     * </li>
     * </ul></p>
     * <p>
     * Non-critical processing errors raise a {@link Signer.WarningListener }.
     * It is recommended to review the {@link WarningCategory } and handle them if necessary for the application.
     * </p>
     * @param document 
     * The input document to process
     * @param stream 
     * The stream where the output document is written
     * @return 
     *
     * @throws com.pdftools.LicenseException The license check has failed.
     * @throws java.io.IOException Writing to the {@link com.pdftools.sign.Signer#process stream} failed.
     * @throws com.pdftools.UnsupportedFeatureException The input PDF contains unrendered XFA form fields.
     *         See {@link com.pdftools.pdf.Document#getXfa pdftools.pdf.Document.getXfa} for more information on how to detect and handle XFA documents.
     * @throws com.pdftools.HttpException If a network error occurs, e.g. downloading revocation information (OCSP, CRL).
     * @throws IllegalArgumentException if {@code document} is {@code null}
     * @throws IllegalArgumentException if {@code stream} is {@code null}
     */
    public com.pdftools.pdf.Document process(com.pdftools.pdf.Document document, com.pdftools.sys.Stream stream) 
        throws 
            java.io.IOException,
            com.pdftools.LicenseException,
            com.pdftools.UnsupportedFeatureException,

            com.pdftools.HttpException
    {
        if (document == null)
            throw new IllegalArgumentException("Argument 'document' must not be null.", new NullPointerException("'document'"));
        if (stream == null)
            throw new IllegalArgumentException("Argument 'stream' must not be null.", new NullPointerException("'stream'"));

        return process(document, stream, null, null);
    }

    /**
     * <h1>Process a document</h1>
     * <p>
     * Apply document-level processing options without any signature operation.
     * For example:
     * <ul>
     * <li>
     * To encrypt or decrypt PDF documents that may be signed (see the samples "Encrypt" and "Decrypt").
     * </li>
     * <li>
     * To remove signatures and unsigned signature fields (see {@link OutputOptions#getRemoveSignatures }).
     * </li>
     * <li>
     * To add validation information to existing signatures (see {@link OutputOptions#getAddValidationInformation }).
     * </li>
     * </ul></p>
     * <p>
     * Non-critical processing errors raise a {@link Signer.WarningListener }.
     * It is recommended to review the {@link WarningCategory } and handle them if necessary for the application.
     * </p>
     * @param document 
     * The input document to process
     * @param stream 
     * The stream where the output document is written
     * @param outputOptions 
     * The document-level processing options
     * @return 
     *
     * @throws com.pdftools.LicenseException The license check has failed.
     * @throws java.io.IOException Writing to the {@link com.pdftools.sign.Signer#process stream} failed.
     * @throws com.pdftools.UnsupportedFeatureException The input PDF contains unrendered XFA form fields.
     *         See {@link com.pdftools.pdf.Document#getXfa pdftools.pdf.Document.getXfa} for more information on how to detect and handle XFA documents.
     * @throws com.pdftools.HttpException If a network error occurs, e.g. downloading revocation information (OCSP, CRL).
     * @throws IllegalArgumentException if {@code document} is {@code null}
     * @throws IllegalArgumentException if {@code stream} is {@code null}
     */
    public com.pdftools.pdf.Document process(com.pdftools.pdf.Document document, com.pdftools.sys.Stream stream, com.pdftools.sign.OutputOptions outputOptions) 
        throws 
            java.io.IOException,
            com.pdftools.LicenseException,
            com.pdftools.UnsupportedFeatureException,

            com.pdftools.HttpException
    {
        if (document == null)
            throw new IllegalArgumentException("Argument 'document' must not be null.", new NullPointerException("'document'"));
        if (stream == null)
            throw new IllegalArgumentException("Argument 'stream' must not be null.", new NullPointerException("'stream'"));

        return process(document, stream, outputOptions, null);
    }

    /**
     * <h1>Process a document</h1>
     * <p>
     * Apply document-level processing options without any signature operation.
     * For example:
     * <ul>
     * <li>
     * To encrypt or decrypt PDF documents that may be signed (see the samples "Encrypt" and "Decrypt").
     * </li>
     * <li>
     * To remove signatures and unsigned signature fields (see {@link OutputOptions#getRemoveSignatures }).
     * </li>
     * <li>
     * To add validation information to existing signatures (see {@link OutputOptions#getAddValidationInformation }).
     * </li>
     * </ul></p>
     * <p>
     * Non-critical processing errors raise a {@link Signer.WarningListener }.
     * It is recommended to review the {@link WarningCategory } and handle them if necessary for the application.
     * </p>
     * @param document 
     * The input document to process
     * @param stream 
     * The stream where the output document is written
     * @param outputOptions 
     * The document-level processing options
     * @param provider 
     * The cryptographic provider to use to add validation information to existing signatures of input document
     * (see {@link OutputOptions#getAddValidationInformation }).
     * Can be {@code null} if no validation information is added or to use the default provider.
     * @return 
     *
     * @throws com.pdftools.LicenseException The license check has failed.
     * @throws java.io.IOException Writing to the {@link com.pdftools.sign.Signer#process stream} failed.
     * @throws com.pdftools.UnsupportedFeatureException The input PDF contains unrendered XFA form fields.
     *         See {@link com.pdftools.pdf.Document#getXfa pdftools.pdf.Document.getXfa} for more information on how to detect and handle XFA documents.
     * @throws com.pdftools.HttpException If a network error occurs, e.g. downloading revocation information (OCSP, CRL).
     * @throws IllegalArgumentException if {@code document} is {@code null}
     * @throws IllegalArgumentException if {@code stream} is {@code null}
     */
    public com.pdftools.pdf.Document process(com.pdftools.pdf.Document document, com.pdftools.sys.Stream stream, com.pdftools.sign.OutputOptions outputOptions, com.pdftools.crypto.providers.Provider provider) 
        throws 
            java.io.IOException,
            com.pdftools.LicenseException,
            com.pdftools.UnsupportedFeatureException,

            com.pdftools.HttpException
    {
        if (document == null)
            throw new IllegalArgumentException("Argument 'document' must not be null.", new NullPointerException("'document'"));
        if (stream == null)
            throw new IllegalArgumentException("Argument 'stream' must not be null.", new NullPointerException("'stream'"));

        long retHandle = processNative(getHandle(), getHandle(document), document, stream, getHandle(outputOptions), outputOptions, getHandle(provider), provider);

        if (retHandle == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                case 4: throw new java.io.IOException(getLastErrorMessage());
                case 12: throw new com.pdftools.LicenseException(getLastErrorMessage());
                case 19: throw new com.pdftools.UnsupportedFeatureException(getLastErrorMessage());
                case 24: throw new com.pdftools.HttpException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }

        return com.pdftools.pdf.Document.createDynamicObject(retHandle);
    }




    private native long addWarningHandlerNative(long handle, WarningNativeClass eventClass);
    private native boolean removeWarningHandlerNative(long handle, long context);

    private static native long newNative();
    private native long signNative(long handle, long document, com.pdftools.pdf.Document documentObj, long configuration, com.pdftools.sign.SignatureConfiguration configurationObj, com.pdftools.sys.Stream stream, long outputOptions, com.pdftools.sign.OutputOptions outputOptionsObj);
    private native long certifyNative(long handle, long document, com.pdftools.pdf.Document documentObj, long configuration, com.pdftools.sign.SignatureConfiguration configurationObj, com.pdftools.sys.Stream stream, long permissions, com.pdftools.sign.MdpPermissionOptions permissionsObj, long outputOptions, com.pdftools.sign.OutputOptions outputOptionsObj);
    private native long addTimestampNative(long handle, long document, com.pdftools.pdf.Document documentObj, long configuration, com.pdftools.sign.TimestampConfiguration configurationObj, com.pdftools.sys.Stream stream, long outputOptions, com.pdftools.sign.OutputOptions outputOptionsObj);
    private native long addSignatureFieldNative(long handle, long document, com.pdftools.pdf.Document documentObj, long options, com.pdftools.sign.SignatureFieldOptions optionsObj, com.pdftools.sys.Stream stream, long outputOptions, com.pdftools.sign.OutputOptions outputOptionsObj);
    private native long addPreparedSignatureNative(long handle, long document, com.pdftools.pdf.Document documentObj, long configuration, com.pdftools.sign.SignatureConfiguration configurationObj, com.pdftools.sys.Stream stream, long outputOptions, com.pdftools.sign.OutputOptions outputOptionsObj);
    private native long signPreparedSignatureNative(long handle, long document, com.pdftools.pdf.Document documentObj, long configuration, com.pdftools.sign.SignatureConfiguration configurationObj, com.pdftools.sys.Stream stream);
    private native long processNative(long handle, long document, com.pdftools.pdf.Document documentObj, com.pdftools.sys.Stream stream, long outputOptions, com.pdftools.sign.OutputOptions outputOptionsObj, long provider, com.pdftools.crypto.providers.Provider providerObj);

}

