/****************************************************************************
 *
 * File:            TimestampConfiguration.java
 *
 * Description:     PDFTOOLS TimestampConfiguration Class
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.sign;

import com.pdftools.sys.*;
import com.pdftools.internal.*;
import java.util.EnumSet;
import java.time.OffsetDateTime;
/**
 * <h1>Configuration for adding time-stamps</h1>
 * <p>
 * This configuration controls the creation of time-stamps in {@link Signer#addTimestamp }.</p>
 * <p>
 * Use a {@link com.pdftools.crypto.providers.Provider pdftools.crypto.providers.Provider} to create a time-stamp configuration.</p>
 * <p>
 * Note that this object can be re-used to add time-stamps to multiple documents for mass-signing applications.
 * </p>
 */
public abstract class TimestampConfiguration extends NativeObject 
{
    protected TimestampConfiguration(long handle) 
    {
        super(handle);
    }

    /**
     * @hidden
     */
    public static TimestampConfiguration createDynamicObject(long handle)
    {
        int type = getType(handle);
        switch (type)
        {
            case 1:
                return com.pdftools.crypto.providers.globalsigndss.TimestampConfiguration.createDynamicObject(handle);
            case 2:
                return com.pdftools.crypto.providers.swisscomsigsrv.TimestampConfiguration.createDynamicObject(handle);
            case 3:
                return com.pdftools.crypto.providers.pkcs11.TimestampConfiguration.createDynamicObject(handle);
            case 4:
                return com.pdftools.crypto.providers.builtin.TimestampConfiguration.createDynamicObject(handle);
            default:
                return null;
        }
    }




     /**
     * <h1>The name of the existing signature field (Getter)</h1>
     * <p>
     * The {@link com.pdftools.pdf.SignatureField#getFieldName pdftools.pdf.SignatureField.getFieldName} of an existing, unsigned signature field to time-stamp.
     * Note that when an existing signature field is used, the appearance's position is defined by the existing field.
     * Therefore, make sure the {@link TimestampConfiguration#getAppearance } fits into the {@link com.pdftools.pdf.SignatureField#getBoundingBox pdftools.pdf.SignatureField.getBoundingBox}.</p>
     * <p>
     * If {@code null}, a new signature field is created using a unique field name.</p>
     * <p>
     * Default: {@code null}</p>
     *
     * @throws IllegalStateException If the creating provider has already been closed
     */
    public String getFieldName()
    {
        String retVal = getFieldNameNative(getHandle());
        if (retVal == null)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                case 2: throw new IllegalStateException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }

     /**
     * <h1>The name of the existing signature field (Setter)</h1>
     * <p>
     * The {@link com.pdftools.pdf.SignatureField#getFieldName pdftools.pdf.SignatureField.getFieldName} of an existing, unsigned signature field to time-stamp.
     * Note that when an existing signature field is used, the appearance's position is defined by the existing field.
     * Therefore, make sure the {@link TimestampConfiguration#getAppearance } fits into the {@link com.pdftools.pdf.SignatureField#getBoundingBox pdftools.pdf.SignatureField.getBoundingBox}.</p>
     * <p>
     * If {@code null}, a new signature field is created using a unique field name.</p>
     * <p>
     * Default: {@code null}</p>
     *
     * @throws IllegalStateException If the creating provider has already been closed
     */
    public void setFieldName(String value)
    {
        boolean retVal = setFieldNameNative(getHandle(), value);
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                case 2: throw new IllegalStateException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }
    }


     /**
     * <h1>The visual appearance of the time-stamp (Getter)</h1>
     * <p>
     * The visual appearance or {@code null} to create a time-stamp without a visual appearance.</p>
     * <p>
     * For time-stamps, not all text variables are available,
     * most notably the {@code [signature:name]}.</p>
     * <p>
     * Default: {@code null}</p>
     *
     * @throws IllegalStateException If the creating provider has already been closed
     */
    public com.pdftools.sign.Appearance getAppearance()
    {
        long retHandle = getAppearanceNative(getHandle());
        if (retHandle == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                case 2: throw new IllegalStateException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
            return null;
        }
        return com.pdftools.sign.Appearance.createDynamicObject(retHandle);

    }

     /**
     * <h1>The visual appearance of the time-stamp (Setter)</h1>
     * <p>
     * The visual appearance or {@code null} to create a time-stamp without a visual appearance.</p>
     * <p>
     * For time-stamps, not all text variables are available,
     * most notably the {@code [signature:name]}.</p>
     * <p>
     * Default: {@code null}</p>
     *
     * @throws IllegalStateException If the creating provider has already been closed
     */
    public void setAppearance(com.pdftools.sign.Appearance value)
    {
        boolean retVal = setAppearanceNative(getHandle(), getHandle(value), value);
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                case 2: throw new IllegalStateException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }
    }




    private static native int getType(long handle);
    private native String getFieldNameNative(long handle);
    private native boolean setFieldNameNative(long handle, String value);
    private native long getAppearanceNative(long handle);
    private native boolean setAppearanceNative(long handle, long value, com.pdftools.sign.Appearance valueObj);

}

