/****************************************************************************
 *
 * File:            WarningCategory.java
 *
 * Description:     WarningCategory Enumeration
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.sign;

/**
 * <h1>The warning category</h1>
 * The category of the warning of {@link Signer.WarningListener }.
 */
public enum WarningCategory  
{
    /**
     * 
     * PDF/A conformance of input file removed due to file encryption (i.e. {@link com.pdftools.pdf.OutputOptions#getEncryption pdftools.pdf.OutputOptions.getEncryption} is not {@code null}).
     * Removal of PDF/A conformance is necessary, because encryption is not allowed by the PDF/A standard.
     */
    PDF_A_REMOVED(1),
    /**
     * 
     * <p>
     * When processing signed documents, their encryption parameters (user password, owner password, permissions) cannot be changed.
     * Therefore, the property {@link com.pdftools.pdf.OutputOptions#getEncryption pdftools.pdf.OutputOptions.getEncryption} has no effect.</p>
     * <p>
     * This warning is generated so that the following situations can be detected:
     * <ul>
     * <li>
     * If {@link com.pdftools.pdf.OutputOptions#getEncryption pdftools.pdf.OutputOptions.getEncryption} is {@code null} and the input document is encrypted.
     * The output document is also encrypted.
     * </li>
     * <li>
     * If {@link com.pdftools.pdf.OutputOptions#getEncryption pdftools.pdf.OutputOptions.getEncryption} not {@code null} and the input document is encrypted using different encryption parameters.
     * The output document is also encrypted, preserving the encryption parameters of the input document.
     * </li>
     * <li>
     * If {@link com.pdftools.pdf.OutputOptions#getEncryption pdftools.pdf.OutputOptions.getEncryption} not {@code null} and the input document is not encrypted.
     * The output document is not encrypted.
     * </li>
     * </ul></p>
     * <p>
     * Encryption parameters of signed documents can be changed by removing all existing signatures using the property {@link OutputOptions#getRemoveSignatures }.
     * In this case, this warning is not generated.
     * </p>
     */
    SIGNED_DOC_ENCRYPTION_UNCHANGED(2),
    /**
     * 
     * <p>
     * Error adding validation information to existing signatures of input document as requested by
     * {@link OutputOptions#getAddValidationInformation }.
     * The warning's {@code context} contains a description of the affected signature.</p>
     * <p>
     * Potential causes of this warning are:
     * <ul>
     * <li>
     * <em>Missing issuer certificate:</em>
     * All certificates of the trust chain are required to add validation information.
     * Preferably, the certificates should be present in the cryptographic provider's certificate store.
     * Alternatively, if supported by the certificate,
     * the issuer certificate is downloaded from the certificate authority's server and
     * stored in the user's {@code Certificates} directory (see {@link com.pdftools.crypto.providers.builtin.Provider pdftools.crypto.providers.builtin.Provider}).
     * </li>
     * <li>
     * <em>Network problem:</em>
     * The network must allow OCSP and CRL responses to be downloaded from the certificate authority's server.
     * Make sure your proxy configuration (see {@link com.pdftools.Sdk#getProxy pdftools.Sdk.getProxy}) is correct.
     * </li>
     * </ul></p>
     */
    ADD_VALIDATION_INFORMATION_FAILED(3);

    WarningCategory(int value) 
    {
        this.value = value;
    }

    /**
     * @hidden
     */
    public static WarningCategory fromValue(int value) 
    {
        switch (value) 
        {
            case 1: return PDF_A_REMOVED;
            case 2: return SIGNED_DOC_ENCRYPTION_UNCHANGED;
            case 3: return ADD_VALIDATION_INFORMATION_FAILED;
        }
        throw new IllegalArgumentException("Unknown value for WarningCategory: " + value);
    }

    /**
     * @hidden
     */
    public int getValue() 
    {
        return value;
    }

    private int value;
}
