/****************************************************************************
 *
 * File:            Certificate.java
 *
 * Description:     PDFTOOLS Certificate Class
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.signaturevalidation;

import com.pdftools.sys.*;
import com.pdftools.internal.*;
import java.util.EnumSet;
import java.time.OffsetDateTime;
/**
 * <h1>A X.509 certificate</h1>
 */
public class Certificate extends NativeObject 
{
    protected Certificate(long handle) 
    {
        super(handle);
    }

    /**
     * @hidden
     */
    public static Certificate createDynamicObject(long handle)
    {
        return new Certificate(handle);
    }




     /**
     * <h1>The name (subject) of the certificate (Getter)</h1>
     * The common name (CN) of the person or authority that owns the certificate.
     */
    public String getSubjectName()
    {
        String retVal = getSubjectNameNative(getHandle());
        if (retVal == null)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }


     /**
     * <h1>The subject of the certificate (Getter)</h1>
     * The distinguished name (DN) of the person or authority that owns the certificate.
     * Formatted according to RFC 4514.
     */
    public String getSubject()
    {
        String retVal = getSubjectNative(getHandle());
        if (retVal == null)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }


     /**
     * <h1>The name of the certificate's issuer (CA) (Getter)</h1>
     * The common name (CN) of the certificate authority (CA) that issued the certificate.
     */
    public String getIssuerName()
    {
        String retVal = getIssuerNameNative(getHandle());
        if (retVal == null)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }


     /**
     * <h1>The date after which the certificate is no longer valid. (Getter)</h1>
     * 
     */
    public OffsetDateTime getNotAfter()
    {
        OffsetDateTime retVal = getNotAfterNative(getHandle());
        if (retVal == null)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }


     /**
     * <h1>The date on which the certificate becomes valid. (Getter)</h1>
     * 
     */
    public OffsetDateTime getNotBefore()
    {
        OffsetDateTime retVal = getNotBeforeNative(getHandle());
        if (retVal == null)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }


     /**
     * <h1>The certificate's fingerprint (Getter)</h1>
     * The hex string representation of the certificate’s SHA-1 digest.
     */
    public String getFingerprint()
    {
        String retVal = getFingerprintNative(getHandle());
        if (retVal == null)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }


     /**
     * <h1>The raw data of the certificate as a byte array (Getter)</h1>
     * 
     */
    public byte[] getRawData()
    {
        byte[] retVal = getRawDataNative(getHandle());
        if (retVal == null)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
        return retVal;
    }


     /**
     * <h1>Source of the certificate (Getter)</h1>
     * 
     */
    public EnumSet<com.pdftools.signaturevalidation.DataSource> getSource()
    {
        int retVal = getSourceNative(getHandle());
        if (retVal == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: break;
                default: throwLastRuntimeException();
            }
        }
        return getEnumSet(retVal, com.pdftools.signaturevalidation.DataSource.class);
    }


     /**
     * <h1>Whether the certificate is valid according to the validation profile used (Getter)</h1>
     * 
     */
    public com.pdftools.signaturevalidation.ConstraintResult getValidity()
    {
        long retHandle = getValidityNative(getHandle());
        if (retHandle == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }
        return com.pdftools.signaturevalidation.ConstraintResult.createDynamicObject(retHandle);

    }




    private native String getSubjectNameNative(long handle);
    private native String getSubjectNative(long handle);
    private native String getIssuerNameNative(long handle);
    private native OffsetDateTime getNotAfterNative(long handle);
    private native OffsetDateTime getNotBeforeNative(long handle);
    private native String getFingerprintNative(long handle);
    private native byte[] getRawDataNative(long handle);
    private native int getSourceNative(long handle);
    private native long getValidityNative(long handle);

}

