/****************************************************************************
 *
 * File:            CustomTrustList.java
 *
 * Description:     PDFTOOLS CustomTrustList Class
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.signaturevalidation;

import com.pdftools.sys.*;
import com.pdftools.internal.*;
import java.util.EnumSet;
import java.time.OffsetDateTime;
/**
 * <h1>The custom collection of trusted certificates</h1>
 * This class defines a custom collection of trusted certificates.
 * They define the certificates used for {@link DataSource#CUSTOM_TRUST_LIST } and can be set in the validation profile with {@link com.pdftools.signaturevalidation.profiles.Profile#setCustomTrustList profiles.Profile.setCustomTrustList}.
 */
public class CustomTrustList extends NativeObject 
{
    protected CustomTrustList(long handle) 
    {
        super(handle);
    }

    /**
     * @hidden
     */
    public static CustomTrustList createDynamicObject(long handle)
    {
        return new CustomTrustList(handle);
    }


    /**
     * 
     */
    public CustomTrustList()
    {
        this(newHelper());
    }

    private static long newHelper()
    {
        long handle = newNative();
        if (handle == 0)
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                default: throwLastRuntimeException();
            }
        }

        return handle;
    }


    /**
     * <h1>Add one or more certificates</h1>
     * Add certificates to the trust list.
     * @param certificate 
     * The sequence of certificates in either PEM (.pem, ASCII text) or DER (.cer, binary) form
     *
     * @throws com.pdftools.CorruptException If the certificate is corrupt and cannot be read
     * @throws IllegalArgumentException if {@code certificate} is {@code null}
     */
    public void addCertificates(com.pdftools.sys.Stream certificate) 
        throws 
            com.pdftools.CorruptException
    {
        if (certificate == null)
            throw new IllegalArgumentException("Argument 'certificate' must not be null.", new NullPointerException("'certificate'"));

        boolean retVal = addCertificatesNative(getHandle(), certificate);
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                case 16: throw new com.pdftools.CorruptException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }
    }

    /**
     * <h1>Add certificates from a PFX (PKCS#12) archive</h1>
     * Add certificates to the trust list.
     * @param stream 
     * The certificates in PKCS#12 format (.p12, .pfx)
     *
     * @throws com.pdftools.CorruptException The PFX (PKCS#12) archive is corrupt and cannot be read.
     * @throws com.pdftools.PasswordException The password is invalid.
     * @throws IllegalArgumentException if {@code stream} is {@code null}
     */
    public void addArchive(com.pdftools.sys.Stream stream) 
        throws 
            com.pdftools.CorruptException,

            com.pdftools.PasswordException
    {
        if (stream == null)
            throw new IllegalArgumentException("Argument 'stream' must not be null.", new NullPointerException("'stream'"));

        addArchive(stream, null);
    }

    /**
     * <h1>Add certificates from a PFX (PKCS#12) archive</h1>
     * Add certificates to the trust list.
     * @param stream 
     * The certificates in PKCS#12 format (.p12, .pfx)
     * @param password 
     * The password required to decrypt the archive.
     *
     * @throws com.pdftools.CorruptException The PFX (PKCS#12) archive is corrupt and cannot be read.
     * @throws com.pdftools.PasswordException The password is invalid.
     * @throws IllegalArgumentException if {@code stream} is {@code null}
     */
    public void addArchive(com.pdftools.sys.Stream stream, String password) 
        throws 
            com.pdftools.CorruptException,

            com.pdftools.PasswordException
    {
        if (stream == null)
            throw new IllegalArgumentException("Argument 'stream' must not be null.", new NullPointerException("'stream'"));

        boolean retVal = addArchiveNative(getHandle(), stream, password);
        if (!retVal) 
        {
            switch (getLastErrorCode())
            {
                case 0: throw new RuntimeException("An unexpected error occurred");
                case 16: throw new com.pdftools.CorruptException(getLastErrorMessage());
                case 17: throw new com.pdftools.PasswordException(getLastErrorMessage());

                default: throwLastRuntimeException();
            }
        }
    }




    private static native long newNative();
    private native boolean addCertificatesNative(long handle, com.pdftools.sys.Stream certificate);
    private native boolean addArchiveNative(long handle, com.pdftools.sys.Stream stream, String password);

}

