/****************************************************************************
 *
 * File:            DataSource.java
 *
 * Description:     DataSource Enumeration
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.signaturevalidation;

import com.pdftools.internal.Flag;

/**
 * <h1>The source of data such as certificates, OCRPs or CRLs</h1>
 */
public enum DataSource implements Flag 
{
    /**
     * <h1>Data embedded in the signature</h1>
     * Allowed data: certificates, OCSP, CRL
     */
    EMBED_IN_SIGNATURE(0x0001),
    /**
     * <h1>Data embedded in the document security store (DSS)</h1>
     * Allowed data: certificates, OCSP, CRL
     */
    EMBED_IN_DOCUMENT(0x0002),
    /**
     * <h1>Data retrieved online or from the local download cache</h1>
     * <p>
     * Allowed data: issuer certificates (for certificates that have caIssuers extension), OCSP, CRL</p>
     * <p>
     * Note: only data for certificates that are time-valid at the current time can be downloaded.
     * For example, OCSP and CRL can only be downloaded for certificates that have not yet expired.
     * </p>
     */
    DOWNLOAD(0x0004),
    /**
     * <h1>Data from the local system cache</h1>
     * Allowed data: certificates
     */
    SYSTEM(0x0008),
    /**
     * <h1>From the Adobe Approved Trust List (AATL)</h1>
     * <p>
     * NOTE: Support for this trust list has not yet been implemented.</p>
     * <p>
     * Allowed data: issuer certificates
     * </p>
     */
    AATL(0x0100),
    /**
     * <h1>From the European Trust List (EUTL)</h1>
     * <p>
     * NOTE: Support for this trust list has not yet been implemented.</p>
     * <p>
     * Allowed data: issuer certificates
     * </p>
     */
    EUTL(0x0200),
    /**
     * <h1>From the custom trust list</h1>
     * <p>
     * The list of certificates defined by the {@link CustomTrustList }.
     * If no custom trust list has been defined, this value has no effect.</p>
     * <p>
     * Allowed data: issuer certificates
     * </p>
     */
    CUSTOM_TRUST_LIST(0x0400);

    DataSource(int value) 
    {
        this.value = value;
    }

    /**
     * @hidden
     */
    public int getFlag() 
    {
        return value;
    }

    private int value;
}
