/****************************************************************************
 *
 * File:            SubIndication.java
 *
 * Description:     SubIndication Enumeration
 *
 * Author:          PDF Tools AG
 * 
 * Copyright:       Copyright (C) 2023 - 2025 PDF Tools AG, Switzerland
 *                  All rights reserved.
 * 
 * Notice:          By downloading and using this artifact, you accept PDF Tools AG's
 *                  [license agreement](https://www.pdf-tools.com/license-agreement/),
 *                  [privacy policy](https://www.pdf-tools.com/privacy-policy/),
 *                  and allow PDF Tools AG to track your usage data.
 *
 ***************************************************************************/

package com.pdftools.signaturevalidation;

/**
 * <h1>Sub-status indication of the signature validation process</h1>
 * See ETSI TS 102 853 and ETSI EN 319 102-1.
 */
public enum SubIndication  
{
    /**
     * <h1>The signer's certificate has been revoked.</h1>
     */
    REVOKED(1),
    /**
     * <h1>The signature is invalid because at least one hash of the signed data object(s) included in the signing process does not match the corresponding hash value in the signature.</h1>
     */
    HASH_FAILURE(2),
    /**
     * <h1>The signature is invalid because the signature value in the signature could not be verified using the signer's public key in the signer's certificate.</h1>
     */
    SIG_CRYPTO_FAILURE(3),
    /**
     * <h1>The signature is considered invalid because one or more properties of the signature do not match the validation constraints.</h1>
     */
    SIG_CONSTRAINTS_FAILURE(4),
    /**
     * <h1>The signature is considered invalid because the certificate chain used in the validation process does not match the validation constraints related to the certificate.</h1>
     */
    CHAIN_CONSTRAINTS_FAILURE(5),
    /**
     * <h1>The signature is considered invalid because at least one of the algorithms used in an element (e.g. the signature value, a certificate, etc.) has been considered unreliable. Either the algorithm used to invalidate the signature or the size of the keys used by the algorithm are no longer considered secure. The Signature Validation Algorithm has detected that this element was generated after this algorithm was deemed insecure.</h1>
     */
    CRYPTO_CONSTRAINTS_FAILURE(6),
    /**
     * <h1>The signature is considered invalid because the Signature Validation Algorithm has detected that the signing time is after the expiration date (notAfter) of the signer's certificate.</h1>
     */
    EXPIRED(7),
    /**
     * <h1>The signature is considered invalid because the Signature Validation Algorithm has detected that the signing time is before the issue date (notBefore) of the signer's certificate.</h1>
     */
    NOT_YET_VALID(8),
    /**
     * <h1>The signature is not conformant to one of the base standards</h1>
     */
    FORMAT_FAILURE(9),
    /**
     * <h1>The formal policy file could not be processed (e.g. not accessible, not parsable, etc.)</h1>
     */
    POLICY_PROCESSING_ERROR(10),
    /**
     * <h1>The signature was created using a policy and commitment type that is unknown to the SVA.</h1>
     */
    UNKNOWN_COMMITMENT_TYPE(11),
    /**
     * <h1>Constraints on the order of signature time-stamps and/or signed data object (s) time-stamps are not respected.</h1>
     */
    TIMESTAMP_ORDER_FAILURE(12),
    /**
     * <h1>The signer's certificate cannot be identified.</h1>
     */
    NO_SIGNER_CERTIFICATE_FOUND(13),
    /**
     * <h1>No certificate chain has been found for the identified signer's certificate.</h1>
     */
    NO_CERTIFICATE_CHAIN_FOUND(14),
    /**
     * <h1>The signer's certificate was revoked at the validation date/time. The Signature Validation Algorithm cannot determine whether the signing time is before or after the revocation time.</h1>
     */
    REVOKED_NO_POE(15),
    /**
     * <h1>At least one certificate chain was found, but an intermediate CA certificate has been revoked.</h1>
     */
    REVOKED_CA_NO_POE(16),
    /**
     * <h1>The signer's certificate is expired or not yet valid at the validation date/time. The Signature Validation Algorithm cannot determine that the signing time is within the validity interval of the signer's certificate.</h1>
     */
    OUT_OF_BOUNDS_NO_POE(17),
    /**
     * <h1>At least one of the algorithms used in an element (e.g. the signature value, a certificate, etc.) to validate the signature or the size of the keys used in the algorithm are no longer considered reliable at the validation date/time. The Signature Validation Algorithm cannot determine whether the element was generated before or after the algorithm or the size of the keys were considered unreliable.</h1>
     */
    CRYPTO_CONSTRAINTS_FAILURE_NO_POE(18),
    /**
     * <h1>A proof of existence that proves whether a signed object was produced before a compromising event (e.g. broken algorithm) is missing</h1>
     */
    NO_POE(19),
    /**
     * <h1>Insufficient information to fulfill all constraints. It may be possible to fulfill all constraints with additional revocation information that will be available at a later point of time.</h1>
     */
    TRY_LATER(20),
    /**
     * <h1>The policy to use for validation could not be identified.</h1>
     */
    NO_POLICY(21),
    /**
     * <h1>Cannot obtain signed data.</h1>
     */
    SIGNED_DATA_NOT_FOUND(22),
    /**
     * <h1>The certificate's chain is incomplete. The Signature Validation Algorithm cannot determine whether the certificate is trusted.</h1>
     */
    INCOMPLETE_CERTIFICATE_CHAIN(512),
    /**
     * <h1>The certificate has no revocation information. The Signature Validation Algorithm cannot determine whether the certificate has been revoked.</h1>
     */
    CERTIFICATE_NO_REVOCATION_INFORMATION(513),
    /**
     * <h1>No revocation information is available in the revocation information sources. The Signature Validation Algorithm cannot determine whether the certificate has been revoked.</h1>
     */
    MISSING_REVOCATION_INFORMATION(514),
    /**
     * <h1>The certificate has expired and no revocation information is available in the signature or document. The Signature Validation Algorithm cannot determine whether the certificate has been revoked.</h1>
     */
    EXPIRED_NO_REVOCATION_INFORMATION(515),
    /**
     * <h1>The certificate is not trusted because there is no valid path to a trust anchor.</h1>
     */
    UNTRUSTED(516),
    /**
     * <h1>Any other reason</h1>
     */
    GENERIC(1024);

    SubIndication(int value) 
    {
        this.value = value;
    }

    /**
     * @hidden
     */
    public static SubIndication fromValue(int value) 
    {
        switch (value) 
        {
            case 1: return REVOKED;
            case 2: return HASH_FAILURE;
            case 3: return SIG_CRYPTO_FAILURE;
            case 4: return SIG_CONSTRAINTS_FAILURE;
            case 5: return CHAIN_CONSTRAINTS_FAILURE;
            case 6: return CRYPTO_CONSTRAINTS_FAILURE;
            case 7: return EXPIRED;
            case 8: return NOT_YET_VALID;
            case 9: return FORMAT_FAILURE;
            case 10: return POLICY_PROCESSING_ERROR;
            case 11: return UNKNOWN_COMMITMENT_TYPE;
            case 12: return TIMESTAMP_ORDER_FAILURE;
            case 13: return NO_SIGNER_CERTIFICATE_FOUND;
            case 14: return NO_CERTIFICATE_CHAIN_FOUND;
            case 15: return REVOKED_NO_POE;
            case 16: return REVOKED_CA_NO_POE;
            case 17: return OUT_OF_BOUNDS_NO_POE;
            case 18: return CRYPTO_CONSTRAINTS_FAILURE_NO_POE;
            case 19: return NO_POE;
            case 20: return TRY_LATER;
            case 21: return NO_POLICY;
            case 22: return SIGNED_DATA_NOT_FOUND;
            case 512: return INCOMPLETE_CERTIFICATE_CHAIN;
            case 513: return CERTIFICATE_NO_REVOCATION_INFORMATION;
            case 514: return MISSING_REVOCATION_INFORMATION;
            case 515: return EXPIRED_NO_REVOCATION_INFORMATION;
            case 516: return UNTRUSTED;
            case 1024: return GENERIC;
        }
        throw new IllegalArgumentException("Unknown value for SubIndication: " + value);
    }

    /**
     * @hidden
     */
    public int getValue() 
    {
        return value;
    }

    private int value;
}
