package com.pinterest.ktlint.ruleset.standard.rules

import com.pinterest.ktlint.rule.engine.core.api.ElementType
import com.pinterest.ktlint.rule.engine.core.api.IndentConfig
import com.pinterest.ktlint.rule.engine.core.api.Rule
import com.pinterest.ktlint.rule.engine.core.api.RuleId
import com.pinterest.ktlint.rule.engine.core.api.editorconfig.EditorConfig
import com.pinterest.ktlint.rule.engine.core.api.editorconfig.INDENT_SIZE_PROPERTY
import com.pinterest.ktlint.rule.engine.core.api.editorconfig.INDENT_STYLE_PROPERTY
import com.pinterest.ktlint.rule.engine.core.api.findCompositeParentElementOfType
import com.pinterest.ktlint.rule.engine.core.api.indent
import com.pinterest.ktlint.rule.engine.core.api.isPartOfCompositeElementOfType
import com.pinterest.ktlint.rule.engine.core.api.isWhiteSpace
import com.pinterest.ktlint.rule.engine.core.api.nextLeaf
import com.pinterest.ktlint.rule.engine.core.api.nextSibling
import com.pinterest.ktlint.rule.engine.core.api.prevLeaf
import com.pinterest.ktlint.rule.engine.core.api.prevSibling
import com.pinterest.ktlint.rule.engine.core.api.upsertWhitespaceAfterMe
import com.pinterest.ktlint.rule.engine.core.api.upsertWhitespaceBeforeMe
import com.pinterest.ktlint.ruleset.standard.StandardRule
import org.jetbrains.kotlin.com.intellij.lang.ASTNode
import org.jetbrains.kotlin.utils.addToStdlib.applyIf

/**
 * Lints and formats the spacing before and after the angle brackets of a type argument list.
 */
public class TypeArgumentListSpacingRule :
    StandardRule(
        id = "type-argument-list-spacing",
        usesEditorConfigProperties =
            setOf(
                INDENT_SIZE_PROPERTY,
                INDENT_STYLE_PROPERTY,
            ),
    ),
    Rule.Experimental {
    private var indentConfig = IndentConfig.DEFAULT_INDENT_CONFIG

    override fun beforeFirstNode(editorConfig: EditorConfig) {
        indentConfig =
            IndentConfig(
                indentStyle = editorConfig[INDENT_STYLE_PROPERTY],
                tabWidth = editorConfig[INDENT_SIZE_PROPERTY],
            )
    }

    override fun beforeVisitChildNodes(
        node: ASTNode,
        autoCorrect: Boolean,
        emit: (offset: Int, errorMessage: String, canBeAutoCorrected: Boolean) -> Unit,
    ) {
        when (node.elementType) {
            ElementType.TYPE_ARGUMENT_LIST -> {
                visitFunctionDeclaration(node, autoCorrect, emit)
                visitInsideTypeArgumentList(node, autoCorrect, emit)
            }
            ElementType.SUPER_TYPE_LIST, ElementType.SUPER_EXPRESSION ->
                visitInsideTypeArgumentList(node, autoCorrect, emit)
        }
    }

    private fun visitFunctionDeclaration(
        node: ASTNode,
        autoCorrect: Boolean,
        emit: (offset: Int, errorMessage: String, canBeAutoCorrected: Boolean) -> Unit,
    ) {
        // No whitespace expected before type argument list of function call
        //    val list = listOf <String>()
        node
            .prevLeaf(includeEmpty = true)
            ?.takeIf { it.elementType == ElementType.WHITE_SPACE }
            ?.let { noWhitespaceExpected(it, autoCorrect, emit) }

        // No whitespace expected after type argument list of function call
        //    val list = listOf<String> ()
        node
            .takeUnless {
                // unless it is part of a type reference:
                //    fun foo(): List<Foo> { ... }
                //    var bar: List<Bar> = emptyList()
                it.isPartOfCompositeElementOfType(ElementType.TYPE_REFERENCE)
            }
            ?.takeUnless {
                // unless it is part of a call expression followed by lambda:
                //    bar<Foo> { ... }
                it.isPartOfCallExpressionFollowedByLambda()
            }
            ?.lastChildNode
            ?.nextLeaf(includeEmpty = true)
            ?.takeIf { it.elementType == ElementType.WHITE_SPACE }
            ?.let { noWhitespaceExpected(it, autoCorrect, emit) }
    }

    private fun visitInsideTypeArgumentList(
        node: ASTNode,
        autoCorrect: Boolean,
        emit: (offset: Int, errorMessage: String, canBeAutoCorrected: Boolean) -> Unit,
    ) {
        val multiline = node.textContains('\n')
        val expectedIndent =
            node
                .indent()
                .applyIf(multiline) {
                    plus(indentConfig.indent)
                }

        node
            .findChildByType(ElementType.LT)
            ?.nextSibling()
            ?.let { nextSibling ->
                if (multiline) {
                    if (nextSibling.text != expectedIndent) {
                        emit(nextSibling.startOffset, "Expected newline", true)
                        if (autoCorrect) {
                            nextSibling.upsertWhitespaceAfterMe(expectedIndent)
                        }
                    }
                } else {
                    if (nextSibling.isWhiteSpace()) {
                        // Disallow
                        //    val list = listOf< String>()
                        noWhitespaceExpected(nextSibling, autoCorrect, emit)
                    }
                }
            }

        node
            .findChildByType(ElementType.GT)
            ?.prevSibling()
            ?.let { prevSibling ->
                if (multiline) {
                    if (prevSibling.text != expectedIndent) {
                        emit(prevSibling.startOffset, "Expected newline", true)
                        if (autoCorrect) {
                            prevSibling.upsertWhitespaceBeforeMe(expectedIndent)
                        }
                    }
                } else {
                    if (prevSibling.isWhiteSpace()) {
                        // Disallow
                        //    val list = listOf<String >()
                        noWhitespaceExpected(prevSibling, autoCorrect, emit)
                    }
                }
            }
    }

    private fun noWhitespaceExpected(
        node: ASTNode,
        autoCorrect: Boolean,
        emit: (offset: Int, errorMessage: String, canBeAutoCorrected: Boolean) -> Unit,
    ) {
        if (node.text != "") {
            emit(
                node.startOffset,
                "No whitespace expected at this position",
                true,
            )
            if (autoCorrect) {
                node.treeParent.removeChild(node)
            }
        }
    }
}

private fun ASTNode.isPartOfCallExpressionFollowedByLambda(): Boolean =
    findCompositeParentElementOfType(ElementType.CALL_EXPRESSION)
        ?.takeIf { it.elementType == ElementType.CALL_EXPRESSION }
        ?.findChildByType(ElementType.LAMBDA_ARGUMENT)
        .let { it != null }

public val TYPE_ARGUMENT_LIST_SPACING_RULE_ID: RuleId = TypeArgumentListSpacingRule().ruleId
