/*
 * Decompiled with CFR 0.152.
 */
package com.powsybl.ieeecdf.converter;

import com.google.auto.service.AutoService;
import com.google.common.io.ByteStreams;
import com.powsybl.commons.PowsyblException;
import com.powsybl.commons.datasource.DataSource;
import com.powsybl.commons.datasource.ReadOnlyDataSource;
import com.powsybl.commons.parameters.Parameter;
import com.powsybl.commons.parameters.ParameterDefaultValueConfig;
import com.powsybl.commons.parameters.ParameterType;
import com.powsybl.ieeecdf.model.IeeeCdfBranch;
import com.powsybl.ieeecdf.model.IeeeCdfBus;
import com.powsybl.ieeecdf.model.IeeeCdfModel;
import com.powsybl.ieeecdf.model.IeeeCdfReader;
import com.powsybl.ieeecdf.model.IeeeCdfTitle;
import com.powsybl.iidm.network.Bus;
import com.powsybl.iidm.network.BusAdder;
import com.powsybl.iidm.network.Generator;
import com.powsybl.iidm.network.GeneratorAdder;
import com.powsybl.iidm.network.Importer;
import com.powsybl.iidm.network.LineAdder;
import com.powsybl.iidm.network.LoadAdder;
import com.powsybl.iidm.network.Network;
import com.powsybl.iidm.network.NetworkFactory;
import com.powsybl.iidm.network.PhaseTapChanger;
import com.powsybl.iidm.network.PhaseTapChangerAdder;
import com.powsybl.iidm.network.RatioTapChangerAdder;
import com.powsybl.iidm.network.ShuntCompensatorAdder;
import com.powsybl.iidm.network.Substation;
import com.powsybl.iidm.network.SubstationAdder;
import com.powsybl.iidm.network.Terminal;
import com.powsybl.iidm.network.TopologyKind;
import com.powsybl.iidm.network.TwoWindingsTransformer;
import com.powsybl.iidm.network.TwoWindingsTransformerAdder;
import com.powsybl.iidm.network.VoltageLevel;
import com.powsybl.iidm.network.VoltageLevelAdder;
import com.powsybl.iidm.network.extensions.SlackTerminal;
import com.powsybl.iidm.network.util.ContainersMapping;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.UncheckedIOException;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Properties;
import java.util.function.Function;
import java.util.function.ToDoubleFunction;
import java.util.stream.Collectors;
import org.apache.commons.math3.complex.Complex;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@AutoService(value={Importer.class})
public class IeeeCdfImporter
implements Importer {
    private static final Logger LOGGER = LoggerFactory.getLogger(IeeeCdfImporter.class);
    private static final String FORMAT = "IEEE-CDF";
    private static final String EXT = "txt";
    private static final Parameter IGNORE_BASE_VOLTAGE_PARAMETER = new Parameter("ignore-base-voltage", ParameterType.BOOLEAN, "Ignore base voltage specified in the file", (Object)Boolean.FALSE);
    private static final double DEFAULT_ACTIVE_POWER_LIMIT = 9999.0;
    static final ToDoubleFunction<IeeeCdfBus> DEFAULT_NOMINAL_VOLTAGE_PROVIDER = ieeeCdfBus -> ieeeCdfBus.getBaseVoltage() == 0.0 ? 1.0 : ieeeCdfBus.getBaseVoltage();
    private final ToDoubleFunction<IeeeCdfBus> nominalVoltageProvider;

    public IeeeCdfImporter() {
        this(DEFAULT_NOMINAL_VOLTAGE_PROVIDER);
    }

    public IeeeCdfImporter(ToDoubleFunction<IeeeCdfBus> nominalVoltageProvider) {
        this.nominalVoltageProvider = Objects.requireNonNull(nominalVoltageProvider);
    }

    public String getFormat() {
        return FORMAT;
    }

    public List<Parameter> getParameters() {
        return Collections.singletonList(IGNORE_BASE_VOLTAGE_PARAMETER);
    }

    public String getComment() {
        return "IEEE Common Data Format to IIDM converter";
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public boolean exists(ReadOnlyDataSource dataSource) {
        try {
            if (!dataSource.exists(null, EXT)) return false;
            try (BufferedReader reader = new BufferedReader(new InputStreamReader(dataSource.newInputStream(null, EXT)));){
                String titleLine = reader.readLine();
                if (titleLine == null) return false;
                boolean bl = titleLine.length() >= 44 && titleLine.charAt(3) == '/' && titleLine.charAt(6) == '/' && (titleLine.charAt(43) == 'S' || titleLine.charAt(43) == 'W');
                return bl;
            }
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    public void copy(ReadOnlyDataSource fromDataSource, DataSource toDataSource) {
        Objects.requireNonNull(fromDataSource);
        Objects.requireNonNull(toDataSource);
        try (InputStream is = fromDataSource.newInputStream(null, EXT);
             OutputStream os = toDataSource.newOutputStream(null, EXT, false);){
            ByteStreams.copy((InputStream)is, (OutputStream)os);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private static boolean isTransformer(IeeeCdfBranch ieeeCdfBranch) {
        return ieeeCdfBranch.getType() != null && (ieeeCdfBranch.getType() != IeeeCdfBranch.Type.TRANSMISSION_LINE || ieeeCdfBranch.getFinalTurnsRatio() != 0.0);
    }

    private static String getBusId(int busNum) {
        return "B" + busNum;
    }

    private void createBuses(IeeeCdfModel ieeeCdfModel, ContainersMapping containerMapping, PerUnitContext perUnitContext, Network network) {
        for (IeeeCdfBus ieeeCdfBus : ieeeCdfModel.getBuses()) {
            String voltageLevelId = containerMapping.getVoltageLevelId(ieeeCdfBus.getNumber());
            String substationId = containerMapping.getSubstationId(voltageLevelId);
            Substation substation = IeeeCdfImporter.createSubstation(network, substationId);
            VoltageLevel voltageLevel = this.createVoltageLevel(ieeeCdfBus, perUnitContext, voltageLevelId, substation, network);
            Bus bus = IeeeCdfImporter.createBus(ieeeCdfBus, voltageLevel);
            IeeeCdfImporter.createLoad(ieeeCdfBus, voltageLevel);
            IeeeCdfImporter.createShuntCompensator(ieeeCdfBus, perUnitContext, voltageLevel);
            switch (ieeeCdfBus.getType()) {
                case UNREGULATED: {
                    break;
                }
                case HOLD_MVAR_GENERATION_WITHIN_VOLTAGE_LIMITS: {
                    IeeeCdfImporter.newGeneratorAdder(ieeeCdfBus, voltageLevel).setTargetQ(ieeeCdfBus.getReactiveGeneration()).setVoltageRegulatorOn(false).add();
                    break;
                }
                case HOLD_VOLTAGE_WITHIN_VAR_LIMITS: 
                case HOLD_VOLTAGE_AND_ANGLE: {
                    Generator generator = IeeeCdfImporter.newGeneratorAdder(ieeeCdfBus, voltageLevel).setTargetV(ieeeCdfBus.getDesiredVoltage() * voltageLevel.getNominalV()).setVoltageRegulatorOn(true).add();
                    if (ieeeCdfBus.getMinReactivePowerOrVoltageLimit() != 0.0 || ieeeCdfBus.getMaxReactivePowerOrVoltageLimit() != 0.0) {
                        generator.newMinMaxReactiveLimits().setMinQ(ieeeCdfBus.getMinReactivePowerOrVoltageLimit()).setMaxQ(ieeeCdfBus.getMaxReactivePowerOrVoltageLimit()).add();
                    }
                    generator.setTargetQ(ieeeCdfBus.getReactiveGeneration());
                    break;
                }
                default: {
                    throw new IllegalStateException("Unexpected bus type: " + ieeeCdfBus.getType());
                }
            }
            if (ieeeCdfBus.getType() != IeeeCdfBus.Type.HOLD_VOLTAGE_AND_ANGLE) continue;
            SlackTerminal.attach((Bus)bus);
        }
    }

    private static Bus createBus(IeeeCdfBus ieeeCdfBus, VoltageLevel voltageLevel) {
        String busId = IeeeCdfImporter.getBusId(ieeeCdfBus.getNumber());
        Bus bus = ((BusAdder)((BusAdder)voltageLevel.getBusBreakerView().newBus().setId(busId)).setName(ieeeCdfBus.getName())).add();
        bus.setV(ieeeCdfBus.getFinalVoltage() * voltageLevel.getNominalV()).setAngle(ieeeCdfBus.getFinalAngle());
        return bus;
    }

    private static Substation createSubstation(Network network, String substationId) {
        Substation substation = network.getSubstation(substationId);
        if (substation == null) {
            substation = ((SubstationAdder)network.newSubstation().setId(substationId)).add();
        }
        return substation;
    }

    private double getNominalV(IeeeCdfBus ieeeCdfBus, PerUnitContext perUnitContext) {
        if (perUnitContext.isIgnoreBaseVoltage()) {
            return 1.0;
        }
        return this.nominalVoltageProvider.applyAsDouble(ieeeCdfBus);
    }

    private VoltageLevel createVoltageLevel(IeeeCdfBus ieeeCdfBus, PerUnitContext perUnitContext, String voltageLevelId, Substation substation, Network network) {
        double nominalV = this.getNominalV(ieeeCdfBus, perUnitContext);
        VoltageLevel voltageLevel = network.getVoltageLevel(voltageLevelId);
        if (voltageLevel == null) {
            voltageLevel = ((VoltageLevelAdder)substation.newVoltageLevel().setId(voltageLevelId)).setNominalV(nominalV).setTopologyKind(TopologyKind.BUS_BREAKER).add();
        }
        return voltageLevel;
    }

    private static void createLoad(IeeeCdfBus ieeeCdfBus, VoltageLevel voltageLevel) {
        if (ieeeCdfBus.getActiveLoad() != 0.0 || ieeeCdfBus.getReactiveLoad() != 0.0) {
            String busId = IeeeCdfImporter.getBusId(ieeeCdfBus.getNumber());
            ((LoadAdder)((LoadAdder)((LoadAdder)voltageLevel.newLoad().setId(busId + "-L")).setConnectableBus(busId)).setBus(busId)).setP0(ieeeCdfBus.getActiveLoad()).setQ0(ieeeCdfBus.getReactiveLoad()).add();
        }
    }

    private static GeneratorAdder newGeneratorAdder(IeeeCdfBus ieeeCdfBus, VoltageLevel voltageLevel) {
        String busId = IeeeCdfImporter.getBusId(ieeeCdfBus.getNumber());
        return ((GeneratorAdder)((GeneratorAdder)((GeneratorAdder)voltageLevel.newGenerator().setId(busId + "-G")).setConnectableBus(busId)).setBus(busId)).setTargetP(ieeeCdfBus.getActiveGeneration()).setMaxP(9999.0).setMinP(-9999.0);
    }

    private static void createShuntCompensator(IeeeCdfBus ieeeCdfBus, PerUnitContext perUnitContext, VoltageLevel voltageLevel) {
        if (ieeeCdfBus.getShuntSusceptance() != 0.0) {
            String busId = IeeeCdfImporter.getBusId(ieeeCdfBus.getNumber());
            double zb = Math.pow(voltageLevel.getNominalV(), 2.0) / perUnitContext.getSb();
            ((ShuntCompensatorAdder)((ShuntCompensatorAdder)((ShuntCompensatorAdder)voltageLevel.newShuntCompensator().setId(busId + "-SH")).setConnectableBus(busId)).setBus(busId)).setSectionCount(1).newLinearModel().setMaximumSectionCount(1).setBPerSection(ieeeCdfBus.getShuntSusceptance() / zb).add().add();
        }
    }

    private static String getBranchId(char type, int tapBusNumber, int zBusNumber, int circuit, Network network) {
        String id;
        int uniqueCircuit = circuit;
        while (network.getIdentifiable(id = "" + type + tapBusNumber + "-" + zBusNumber + "-" + uniqueCircuit++) != null) {
        }
        return id;
    }

    private static void createLine(IeeeCdfBranch ieeeCdfBranch, ContainersMapping containerMapping, PerUnitContext perUnitContext, Network network) {
        String id = IeeeCdfImporter.getBranchId('L', ieeeCdfBranch.getTapBusNumber(), ieeeCdfBranch.getzBusNumber(), ieeeCdfBranch.getCircuit(), network);
        String bus1Id = IeeeCdfImporter.getBusId(ieeeCdfBranch.getTapBusNumber());
        String bus2Id = IeeeCdfImporter.getBusId(ieeeCdfBranch.getzBusNumber());
        String voltageLevel1Id = containerMapping.getVoltageLevelId(ieeeCdfBranch.getTapBusNumber());
        String voltageLevel2Id = containerMapping.getVoltageLevelId(ieeeCdfBranch.getzBusNumber());
        VoltageLevel voltageLevel1 = network.getVoltageLevel(voltageLevel1Id);
        VoltageLevel voltageLevel2 = network.getVoltageLevel(voltageLevel2Id);
        double nominalV1 = voltageLevel1.getNominalV();
        double nominalV2 = voltageLevel2.getNominalV();
        double sBase = perUnitContext.getSb();
        double r = IeeeCdfImporter.impedanceToEngineeringUnitsForLine(ieeeCdfBranch.getResistance(), nominalV1, nominalV2, sBase);
        double x = IeeeCdfImporter.impedanceToEngineeringUnitsForLine(ieeeCdfBranch.getReactance(), nominalV1, nominalV2, sBase);
        Complex ytr = IeeeCdfImporter.impedanceToAdmittance(r, x);
        double g1 = IeeeCdfImporter.admittanceEndToEngineeringUnitsForLine(ytr.getReal(), 0.0, nominalV1, nominalV2, sBase);
        double b1 = IeeeCdfImporter.admittanceEndToEngineeringUnitsForLine(ytr.getImaginary(), ieeeCdfBranch.getChargingSusceptance() * 0.5, nominalV1, nominalV2, sBase);
        double g2 = IeeeCdfImporter.admittanceEndToEngineeringUnitsForLine(ytr.getReal(), 0.0, nominalV2, nominalV1, sBase);
        double b2 = IeeeCdfImporter.admittanceEndToEngineeringUnitsForLine(ytr.getImaginary(), ieeeCdfBranch.getChargingSusceptance() * 0.5, nominalV2, nominalV1, sBase);
        ((LineAdder)((LineAdder)((LineAdder)((LineAdder)((LineAdder)((LineAdder)((LineAdder)network.newLine().setId(id)).setBus1(bus1Id)).setConnectableBus1(bus1Id)).setVoltageLevel1(voltageLevel1Id)).setBus2(bus2Id)).setConnectableBus2(bus2Id)).setVoltageLevel2(voltageLevel2Id)).setR(r).setX(x).setG1(g1).setB1(b1).setG2(g2).setB2(b2).add();
    }

    private static Complex impedanceToAdmittance(double r, double x) {
        return r == 0.0 && x == 0.0 ? new Complex(0.0, 0.0) : new Complex(r, x).reciprocal();
    }

    private static double impedanceToEngineeringUnitsForLine(double impedance, double nominalVoltageAtEnd, double nominalVoltageAtOtherEnd, double sBase) {
        return impedance * nominalVoltageAtEnd * nominalVoltageAtOtherEnd / sBase;
    }

    private static double admittanceEndToEngineeringUnitsForLine(double transmissionAdmittance, double shuntAdmittanceAtEnd, double nominalVoltageAtEnd, double nominalVoltageAtOtherEnd, double sBase) {
        return shuntAdmittanceAtEnd * sBase / (nominalVoltageAtEnd * nominalVoltageAtEnd) - (1.0 - nominalVoltageAtOtherEnd / nominalVoltageAtEnd) * transmissionAdmittance;
    }

    private static TwoWindingsTransformer createTransformer(IeeeCdfBranch ieeeCdfBranch, ContainersMapping containerMapping, PerUnitContext perUnitContext, Network network) {
        String id = IeeeCdfImporter.getBranchId('T', ieeeCdfBranch.getTapBusNumber(), ieeeCdfBranch.getzBusNumber(), ieeeCdfBranch.getCircuit(), network);
        String bus1Id = IeeeCdfImporter.getBusId(ieeeCdfBranch.getTapBusNumber());
        String bus2Id = IeeeCdfImporter.getBusId(ieeeCdfBranch.getzBusNumber());
        String voltageLevel1Id = containerMapping.getVoltageLevelId(ieeeCdfBranch.getTapBusNumber());
        String voltageLevel2Id = containerMapping.getVoltageLevelId(ieeeCdfBranch.getzBusNumber());
        VoltageLevel voltageLevel1 = network.getVoltageLevel(voltageLevel1Id);
        VoltageLevel voltageLevel2 = network.getVoltageLevel(voltageLevel2Id);
        double zb = Math.pow(voltageLevel2.getNominalV(), 2.0) / perUnitContext.getSb();
        return ((TwoWindingsTransformerAdder)((TwoWindingsTransformerAdder)((TwoWindingsTransformerAdder)((TwoWindingsTransformerAdder)((TwoWindingsTransformerAdder)((TwoWindingsTransformerAdder)((TwoWindingsTransformerAdder)voltageLevel2.getSubstation().map(Substation::newTwoWindingsTransformer).orElseThrow(() -> new PowsyblException("Substation null! Transformer must be within a substation")).setId(id)).setBus1(bus1Id)).setConnectableBus1(bus1Id)).setVoltageLevel1(voltageLevel1Id)).setBus2(bus2Id)).setConnectableBus2(bus2Id)).setVoltageLevel2(voltageLevel2Id)).setRatedU1(voltageLevel1.getNominalV() * ieeeCdfBranch.getFinalTurnsRatio()).setRatedU2(voltageLevel2.getNominalV()).setR(ieeeCdfBranch.getResistance() * zb).setX(ieeeCdfBranch.getReactance() * zb).setG(0.0).setB(ieeeCdfBranch.getChargingSusceptance() / zb).add();
    }

    private static Terminal getRegulatingTerminal(IeeeCdfBranch ieeeCdfBranch, TwoWindingsTransformer transformer) {
        Terminal regulatingTerminal = null;
        if (ieeeCdfBranch.getSide() != null) {
            switch (ieeeCdfBranch.getSide()) {
                case CONTROLLED_BUS_IS_ONE_OF_THE_TERMINALS: {
                    int controlBusNum = ieeeCdfBranch.getControlBusNumber();
                    if (controlBusNum == 0) break;
                    String controlBusId = IeeeCdfImporter.getBusId(controlBusNum);
                    if (controlBusId.equals(transformer.getTerminal1().getBusBreakerView().getBus().getId())) {
                        regulatingTerminal = transformer.getTerminal1();
                        break;
                    }
                    if (controlBusId.equals(transformer.getTerminal2().getBusBreakerView().getBus().getId())) {
                        regulatingTerminal = transformer.getTerminal2();
                        break;
                    }
                    throw new UnsupportedOperationException("Remote control bus not yet supported: " + transformer.getId());
                }
                case CONTROLLED_BUS_IS_NEAR_THE_TAP_SIDE: {
                    regulatingTerminal = transformer.getTerminal1();
                    break;
                }
                case CONTROLLED_BUS_IS_NEAR_THE_IMPEDANCE_SIDE: {
                    regulatingTerminal = transformer.getTerminal2();
                    break;
                }
                default: {
                    throw new IllegalStateException("Unknown branch side: " + ieeeCdfBranch.getSide());
                }
            }
        }
        return regulatingTerminal;
    }

    private static void createTransformerWithVoltageControl(IeeeCdfBranch ieeeCdfBranch, ContainersMapping containerMapping, PerUnitContext perUnitContext, Network network) {
        Bus regulatingBus;
        TwoWindingsTransformer transformer = IeeeCdfImporter.createTransformer(ieeeCdfBranch, containerMapping, perUnitContext, network);
        boolean regulating = false;
        double targetV = Double.NaN;
        Terminal regulatingTerminal = IeeeCdfImporter.getRegulatingTerminal(ieeeCdfBranch, transformer);
        if (regulatingTerminal != null && (regulatingBus = regulatingTerminal.getBusView().getBus()) != null) {
            regulating = true;
            targetV = regulatingBus.getV();
        }
        ArrayList<Double> rhos = new ArrayList<Double>();
        rhos.add(1.0);
        if (ieeeCdfBranch.getMinTapOrPhaseShift() != 0.0 && ieeeCdfBranch.getMaxTapOrPhaseShift() != 0.0) {
            LOGGER.warn("Tap steps are not yet supported ({})", (Object)transformer.getId());
        }
        RatioTapChangerAdder ratioTapChangerAdder = transformer.newRatioTapChanger().setLoadTapChangingCapabilities(true).setRegulating(regulating).setRegulationTerminal(regulatingTerminal).setTargetV(targetV).setTargetDeadband(regulating ? 0.0 : Double.NaN).setTapPosition(0);
        Iterator iterator = rhos.iterator();
        while (iterator.hasNext()) {
            double rho = (Double)iterator.next();
            ratioTapChangerAdder.beginStep().setRho(rho).setR(0.0).setX(0.0).setG(0.0).setB(0.0).endStep();
        }
        ratioTapChangerAdder.add();
    }

    private static void createTransformerWithActivePowerControl(IeeeCdfBranch ieeeCdfBranch, ContainersMapping containerMapping, PerUnitContext perUnitContext, Network network) {
        TwoWindingsTransformer transformer = IeeeCdfImporter.createTransformer(ieeeCdfBranch, containerMapping, perUnitContext, network);
        PhaseTapChangerAdder phaseTapChangerAdder = transformer.newPhaseTapChanger().setRegulationMode(PhaseTapChanger.RegulationMode.FIXED_TAP).setRegulating(false).setTapPosition(0);
        ArrayList<Double> alphas = new ArrayList<Double>();
        alphas.add(-ieeeCdfBranch.getFinalAngle());
        if (ieeeCdfBranch.getMinTapOrPhaseShift() != 0.0 && ieeeCdfBranch.getMaxTapOrPhaseShift() != 0.0) {
            LOGGER.warn("Phase shift steps are not yet supported ({})", (Object)transformer.getId());
        }
        Iterator iterator = alphas.iterator();
        while (iterator.hasNext()) {
            double alpha = (Double)iterator.next();
            phaseTapChangerAdder.beginStep().setAlpha(alpha).setRho(1.0).setR(0.0).setX(0.0).setG(0.0).setB(0.0).endStep();
        }
        phaseTapChangerAdder.add();
    }

    private static void createBranches(IeeeCdfModel ieeeCdfModel, ContainersMapping containerMapping, PerUnitContext perUnitContext, Network network) {
        block7: for (IeeeCdfBranch ieeeCdfBranch : ieeeCdfModel.getBranches()) {
            if (ieeeCdfBranch.getType() == null) {
                IeeeCdfImporter.createLine(ieeeCdfBranch, containerMapping, perUnitContext, network);
                continue;
            }
            switch (ieeeCdfBranch.getType()) {
                case TRANSMISSION_LINE: {
                    if (ieeeCdfBranch.getFinalTurnsRatio() == 0.0) {
                        IeeeCdfImporter.createLine(ieeeCdfBranch, containerMapping, perUnitContext, network);
                        continue block7;
                    }
                    IeeeCdfImporter.createTransformer(ieeeCdfBranch, containerMapping, perUnitContext, network);
                    continue block7;
                }
                case FIXED_TAP: {
                    IeeeCdfImporter.createTransformer(ieeeCdfBranch, containerMapping, perUnitContext, network);
                    continue block7;
                }
                case VARIABLE_TAP_FOR_VOLTAVE_CONTROL: {
                    IeeeCdfImporter.createTransformerWithVoltageControl(ieeeCdfBranch, containerMapping, perUnitContext, network);
                    continue block7;
                }
                case VARIABLE_TAP_FOR_REACTIVE_POWER_CONTROL: {
                    throw new UnsupportedOperationException("Transformers not yet implemented: " + ieeeCdfBranch.getType());
                }
                case VARIABLE_PHASE_ANGLE_FOR_ACTIVE_POWER_CONTROL: {
                    IeeeCdfImporter.createTransformerWithActivePowerControl(ieeeCdfBranch, containerMapping, perUnitContext, network);
                    continue block7;
                }
            }
            throw new IllegalStateException("Unexpected branch type: " + ieeeCdfBranch.getType());
        }
    }

    public Network importData(ReadOnlyDataSource dataSource, NetworkFactory networkFactory, Properties parameters) {
        Network network;
        Objects.requireNonNull(dataSource);
        Objects.requireNonNull(networkFactory);
        BufferedReader reader = new BufferedReader(new InputStreamReader(dataSource.newInputStream(null, EXT)));
        try {
            IeeeCdfModel ieeeCdfModel = new IeeeCdfReader().read(reader);
            boolean ignoreBaseVoltage = Parameter.readBoolean((String)FORMAT, (Properties)parameters, (Parameter)IGNORE_BASE_VOLTAGE_PARAMETER, (ParameterDefaultValueConfig)ParameterDefaultValueConfig.INSTANCE);
            network = this.convert(ieeeCdfModel, networkFactory, dataSource.getBaseName(), ignoreBaseVoltage);
        }
        catch (Throwable throwable) {
            try {
                try {
                    reader.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        }
        reader.close();
        return network;
    }

    Network convert(IeeeCdfModel ieeeCdfModel, NetworkFactory networkFactory, String networkId, boolean ignoreBaseVoltage) {
        PerUnitContext perUnitContext = new PerUnitContext(ieeeCdfModel.getTitle().getMvaBase(), ignoreBaseVoltage);
        Network network = networkFactory.createNetwork(networkId, FORMAT);
        IeeeCdfTitle ieeeCdfTitle = ieeeCdfModel.getTitle();
        if (ieeeCdfTitle.getDate() != null) {
            ZonedDateTime caseDateTime = ieeeCdfTitle.getDate().atStartOfDay(ZoneOffset.UTC.normalized());
            network.setCaseDate(ZonedDateTime.ofInstant(caseDateTime.toInstant(), ZoneOffset.UTC));
        }
        Map busNumToIeeeCdfBus = ieeeCdfModel.getBuses().stream().collect(Collectors.toMap(IeeeCdfBus::getNumber, Function.identity()));
        ContainersMapping containerMapping = ContainersMapping.create((List)ieeeCdfModel.getBuses(), (List)ieeeCdfModel.getBranches(), IeeeCdfBus::getNumber, IeeeCdfBranch::getTapBusNumber, IeeeCdfBranch::getzBusNumber, branch -> branch.getResistance() == 0.0 && branch.getReactance() == 0.0, IeeeCdfImporter::isTransformer, busNumber -> this.getNominalVFromBusNumber(busNumToIeeeCdfBus, (int)busNumber, perUnitContext), busNums -> "VL" + busNums.stream().sorted().findFirst().orElseThrow(() -> new PowsyblException("Unexpected empty busNums")), substationNums -> "S" + substationNums.stream().sorted().findFirst().orElseThrow(() -> new PowsyblException("Unexpected empty substationNums")));
        this.createBuses(ieeeCdfModel, containerMapping, perUnitContext, network);
        IeeeCdfImporter.createBranches(ieeeCdfModel, containerMapping, perUnitContext, network);
        return network;
    }

    private double getNominalVFromBusNumber(Map<Integer, IeeeCdfBus> busNumToIeeeCdfBus, int busNumber, PerUnitContext perUnitContext) {
        if (!busNumToIeeeCdfBus.containsKey(busNumber)) {
            throw new PowsyblException("busId without IeeeCdfBus" + busNumber);
        }
        return this.getNominalV(busNumToIeeeCdfBus.get(busNumber), perUnitContext);
    }

    private static final class PerUnitContext {
        private final double sb;
        private final boolean ignoreBaseVoltage;

        private PerUnitContext(double sb, boolean ignoreBaseVoltage) {
            this.sb = sb;
            this.ignoreBaseVoltage = ignoreBaseVoltage;
        }

        private double getSb() {
            return this.sb;
        }

        public boolean isIgnoreBaseVoltage() {
            return this.ignoreBaseVoltage;
        }
    }
}

