/*
 * Decompiled with CFR 0.152.
 */
package com.powsybl.iidm.network;

import com.powsybl.commons.PowsyblException;
import com.powsybl.commons.report.ReportNode;
import com.powsybl.iidm.network.AcDcConverter;
import com.powsybl.iidm.network.Battery;
import com.powsybl.iidm.network.Branch;
import com.powsybl.iidm.network.Connectable;
import com.powsybl.iidm.network.DanglingLine;
import com.powsybl.iidm.network.DcNode;
import com.powsybl.iidm.network.EnergySource;
import com.powsybl.iidm.network.Generator;
import com.powsybl.iidm.network.HvdcLine;
import com.powsybl.iidm.network.Identifiable;
import com.powsybl.iidm.network.LineCommutatedConverter;
import com.powsybl.iidm.network.Load;
import com.powsybl.iidm.network.LoadType;
import com.powsybl.iidm.network.LoadingLimits;
import com.powsybl.iidm.network.Network;
import com.powsybl.iidm.network.OperationalLimitsGroup;
import com.powsybl.iidm.network.PhaseTapChanger;
import com.powsybl.iidm.network.PhaseTapChangerHolder;
import com.powsybl.iidm.network.RatioTapChanger;
import com.powsybl.iidm.network.RatioTapChangerHolder;
import com.powsybl.iidm.network.ShuntCompensator;
import com.powsybl.iidm.network.StaticVarCompensator;
import com.powsybl.iidm.network.TapChanger;
import com.powsybl.iidm.network.Terminal;
import com.powsybl.iidm.network.ThreeWindingsTransformer;
import com.powsybl.iidm.network.TopologyKind;
import com.powsybl.iidm.network.TwoWindingsTransformer;
import com.powsybl.iidm.network.Validable;
import com.powsybl.iidm.network.ValidationException;
import com.powsybl.iidm.network.ValidationLevel;
import com.powsybl.iidm.network.VoltageLevel;
import com.powsybl.iidm.network.VscConverterStation;
import com.powsybl.iidm.network.util.NetworkReports;
import java.time.ZonedDateTime;
import java.util.Collection;
import java.util.Objects;
import java.util.Set;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class ValidationUtil {
    private static final Logger LOGGER = LoggerFactory.getLogger(ValidationUtil.class);
    private static final String ACTIVE_POWER_SETPOINT = "active power setpoint";
    private static final String MAXIMUM_P = "maximum P";
    private static final String UNIQUE_REGULATING_TAP_CHANGER_MSG = "Only one regulating control enabled is allowed";
    private static final String VOLTAGE_REGULATOR_ON = "voltage regulator is on";
    private static final String VOLTAGE_SETPOINT = "voltage setpoint";

    private ValidationUtil() {
    }

    public static void checkAccessOfRemovedEquipment(String id, boolean removed) {
        if (removed) {
            throw new PowsyblException("Cannot access removed equipment " + id);
        }
    }

    public static void checkModifyOfRemovedEquipment(String id, boolean removed) {
        if (removed) {
            throw new PowsyblException("Cannot modify removed equipment " + id);
        }
    }

    public static void checkAccessOfRemovedEquipment(String id, boolean removed, String attribute) {
        if (removed) {
            throw new PowsyblException("Cannot access " + attribute + " of removed equipment " + id);
        }
    }

    public static void checkModifyOfRemovedEquipment(String id, boolean removed, String attribute) {
        if (removed) {
            throw new PowsyblException("Cannot modify " + attribute + " of removed equipment " + id);
        }
    }

    public static PowsyblException createUndefinedValueGetterException() {
        return new PowsyblException("This getter cannot be used if the value is not defined");
    }

    public static PowsyblException createUnsetMethodException() {
        return new PowsyblException("Unset method is not defined. Implement SCADA mode in order to use it");
    }

    private static ValidationException createInvalidValueException(Validable validable, double value, String valueName) {
        return ValidationUtil.createInvalidValueException(validable, value, valueName, null);
    }

    private static ValidationException createInvalidValueException(Validable validable, double value, String valueName, String reason) {
        Object r = reason == null ? "" : " (" + reason + ")";
        return new ValidationException(validable, "invalid value (" + value + ") for " + valueName + (String)r);
    }

    private static String createInvalidValueMessage(double value, String valueName, String reason) {
        return "invalid value (" + value + ") for " + valueName + (String)(reason == null ? "" : " (" + reason + ")");
    }

    private static void logError(Validable validable, String message) {
        LOGGER.error("{}{}", (Object)validable.getMessageHeader(), (Object)message);
    }

    private static void throwExceptionOrLogError(Validable validable, String message, ActionOnError actionOnError, Consumer<String> reporter) {
        if (actionOnError == ActionOnError.THROW_EXCEPTION) {
            throw new ValidationException(validable, message);
        }
        if (actionOnError == ActionOnError.LOG_ERROR) {
            ValidationUtil.logError(validable, message);
            reporter.accept(validable.getMessageHeader().id());
        }
    }

    private static void throwExceptionOrIgnore(Validable validable, String message, ActionOnError actionOnError) {
        if (actionOnError == ActionOnError.THROW_EXCEPTION) {
            throw new ValidationException(validable, message);
        }
    }

    public static void throwExceptionOrIgnore(Validable validable, String message, ValidationLevel validationLevel) {
        ValidationUtil.throwExceptionOrIgnore(validable, message, ValidationUtil.checkValidationActionOnError(validationLevel));
    }

    private static void throwExceptionOrLogErrorForInvalidValue(Validable validable, double value, String valueName, ActionOnError actionOnError, Consumer<String> reporter) {
        ValidationUtil.throwExceptionOrLogErrorForInvalidValue(validable, value, valueName, null, actionOnError, reporter);
    }

    private static void throwExceptionOrLogErrorForInvalidValue(Validable validable, double value, String valueName, String reason, ActionOnError actionOnError, Consumer<String> reporter) {
        if (actionOnError == ActionOnError.THROW_EXCEPTION) {
            throw ValidationUtil.createInvalidValueException(validable, value, valueName, reason);
        }
        if (actionOnError == ActionOnError.LOG_ERROR) {
            ValidationUtil.logError(validable, ValidationUtil.createInvalidValueMessage(value, valueName, reason));
            reporter.accept(validable.getMessageHeader().id());
        }
    }

    public static ValidationLevel checkActivePowerSetpoint(Validable validable, double activePowerSetpoint, ValidationLevel validationLevel, ReportNode reportNode) {
        return ValidationUtil.checkActivePowerSetpoint(validable, activePowerSetpoint, ValidationUtil.checkValidationActionOnError(validationLevel), reportNode);
    }

    private static ValidationLevel checkActivePowerSetpoint(Validable validable, double activePowerSetpoint, ActionOnError actionOnError, ReportNode reportNode) {
        if (Double.isNaN(activePowerSetpoint)) {
            ValidationUtil.throwExceptionOrLogErrorForInvalidValue(validable, activePowerSetpoint, ACTIVE_POWER_SETPOINT, actionOnError, id -> NetworkReports.activePowerSetpointInvalid(reportNode, id, activePowerSetpoint));
            return ValidationLevel.EQUIPMENT;
        }
        return ValidationLevel.STEADY_STATE_HYPOTHESIS;
    }

    public static ValidationLevel checkHvdcActivePowerSetpoint(Validable validable, double activePowerSetpoint, ValidationLevel validationLevel, ReportNode reportNode) {
        return ValidationUtil.checkHvdcActivePowerSetpoint(validable, activePowerSetpoint, ValidationUtil.checkValidationActionOnError(validationLevel), reportNode);
    }

    private static ValidationLevel checkHvdcActivePowerSetpoint(Validable validable, double activePowerSetpoint, ActionOnError actionOnError, ReportNode reportNode) {
        if (Double.isNaN(activePowerSetpoint)) {
            ValidationUtil.throwExceptionOrLogErrorForInvalidValue(validable, activePowerSetpoint, ACTIVE_POWER_SETPOINT, actionOnError, id -> NetworkReports.activePowerSetpointInvalid(reportNode, id, activePowerSetpoint));
            return ValidationLevel.EQUIPMENT;
        }
        if (activePowerSetpoint < 0.0) {
            throw ValidationUtil.createInvalidValueException(validable, activePowerSetpoint, ACTIVE_POWER_SETPOINT, "active power setpoint should not be negative");
        }
        return ValidationLevel.STEADY_STATE_HYPOTHESIS;
    }

    public static void checkActivePowerLimits(Validable validable, double minP, double maxP) {
        if (minP > maxP) {
            throw new ValidationException(validable, "invalid active limits [" + minP + ", " + maxP + "]");
        }
    }

    public static ValidationLevel checkTargetDeadband(Validable validable, String validableType, boolean regulating, double targetDeadband, ValidationLevel validationLevel, ReportNode reportNode) {
        return ValidationUtil.checkTargetDeadband(validable, validableType, regulating, targetDeadband, ValidationUtil.checkValidationActionOnError(validationLevel), reportNode);
    }

    private static ValidationLevel checkTargetDeadband(Validable validable, String validableType, boolean regulating, double targetDeadband, ActionOnError actionOnError, ReportNode reportNode) {
        if (regulating && Double.isNaN(targetDeadband)) {
            ValidationUtil.throwExceptionOrLogError(validable, "Undefined value for target deadband of regulating " + validableType, actionOnError, id -> NetworkReports.targetDeadbandUndefinedValue(reportNode, validableType, id));
            return ValidationLevel.EQUIPMENT;
        }
        if (targetDeadband < 0.0) {
            throw new ValidationException(validable, "Unexpected value for target deadband of " + validableType + ": " + targetDeadband + " < 0");
        }
        return ValidationLevel.STEADY_STATE_HYPOTHESIS;
    }

    public static ValidationLevel checkVoltageControl(Validable validable, boolean voltageRegulatorOn, double voltageSetpoint, ValidationLevel validationLevel, ReportNode reportNode) {
        return ValidationUtil.checkVoltageControl(validable, voltageRegulatorOn, voltageSetpoint, ValidationUtil.checkValidationActionOnError(validationLevel), reportNode);
    }

    private static ValidationLevel checkVoltageControl(Validable validable, boolean voltageRegulatorOn, double voltageSetpoint, ActionOnError actionOnError, ReportNode reportNode) {
        if (voltageRegulatorOn) {
            if (Double.isNaN(voltageSetpoint)) {
                ValidationUtil.throwExceptionOrLogErrorForInvalidValue(validable, voltageSetpoint, VOLTAGE_SETPOINT, VOLTAGE_REGULATOR_ON, actionOnError, id -> NetworkReports.voltageSetpointInvalidVoltageRegulatorOn(reportNode, id, voltageSetpoint));
                return ValidationLevel.EQUIPMENT;
            }
            if (voltageSetpoint <= 0.0) {
                throw ValidationUtil.createInvalidValueException(validable, voltageSetpoint, VOLTAGE_SETPOINT, VOLTAGE_REGULATOR_ON);
            }
        }
        return ValidationLevel.STEADY_STATE_HYPOTHESIS;
    }

    public static ValidationLevel checkVoltageControl(Validable validable, Boolean voltageRegulatorOn, double voltageSetpoint, double reactivePowerSetpoint, ValidationLevel validationLevel, ReportNode reportNode) {
        return ValidationUtil.checkVoltageControl(validable, voltageRegulatorOn, voltageSetpoint, reactivePowerSetpoint, ValidationUtil.checkValidationActionOnError(validationLevel), reportNode);
    }

    private static ValidationLevel checkVoltageControl(Validable validable, Boolean voltageRegulatorOn, double voltageSetpoint, double reactivePowerSetpoint, ActionOnError actionOnError, ReportNode reportNode) {
        if (voltageRegulatorOn == null) {
            throw new ValidationException(validable, "voltage regulator status is not set");
        }
        if (voltageRegulatorOn.booleanValue()) {
            if (Double.isNaN(voltageSetpoint)) {
                ValidationUtil.throwExceptionOrLogErrorForInvalidValue(validable, voltageSetpoint, VOLTAGE_SETPOINT, VOLTAGE_REGULATOR_ON, actionOnError, id -> NetworkReports.voltageSetpointInvalidVoltageRegulatorOn(reportNode, id, voltageSetpoint));
                return ValidationLevel.EQUIPMENT;
            }
            if (voltageSetpoint <= 0.0) {
                throw ValidationUtil.createInvalidValueException(validable, voltageSetpoint, VOLTAGE_SETPOINT, VOLTAGE_REGULATOR_ON);
            }
        } else if (Double.isNaN(reactivePowerSetpoint)) {
            ValidationUtil.throwExceptionOrLogErrorForInvalidValue(validable, reactivePowerSetpoint, "reactive power setpoint", "voltage regulator is off", actionOnError, id -> NetworkReports.reactivePowerSetpointInvalidVoltageRegulatorOff(reportNode, id, reactivePowerSetpoint));
            return ValidationLevel.EQUIPMENT;
        }
        return ValidationLevel.STEADY_STATE_HYPOTHESIS;
    }

    public static void checkRatedS(Validable validable, double ratedS) {
        if (!Double.isNaN(ratedS) && ratedS <= 0.0) {
            throw new ValidationException(validable, "Invalid value of rated S " + ratedS);
        }
    }

    public static void checkEnergySource(Validable validable, EnergySource energySource) {
        if (energySource == null) {
            throw new ValidationException(validable, "energy source is not set");
        }
    }

    public static void checkMinP(Validable validable, double minP) {
        if (Double.isNaN(minP)) {
            throw ValidationUtil.createInvalidValueException(validable, minP, "minimum P");
        }
    }

    public static void checkMaxP(Validable validable, double maxP) {
        if (Double.isNaN(maxP)) {
            throw ValidationUtil.createInvalidValueException(validable, maxP, MAXIMUM_P);
        }
    }

    public static void checkHvdcMaxP(Validable validable, double maxP) {
        if (Double.isNaN(maxP)) {
            throw ValidationUtil.createInvalidValueException(validable, maxP, MAXIMUM_P);
        }
        if (maxP < 0.0) {
            throw ValidationUtil.createInvalidValueException(validable, maxP, MAXIMUM_P, "maximum P should not be negative");
        }
    }

    public static void checkRegulatingTerminal(Validable validable, Terminal regulatingTerminal, Network network) {
        if (regulatingTerminal != null && regulatingTerminal.getVoltageLevel().getNetwork() != network) {
            throw new ValidationException(validable, "regulating terminal is not part of the network");
        }
    }

    public static void checkLoadType(Validable validable, LoadType loadType) {
        if (loadType == null) {
            throw new ValidationException(validable, "load type is null");
        }
    }

    public static ValidationLevel checkP0(Validable validable, double p0, ValidationLevel validationLevel, ReportNode reportNode) {
        return ValidationUtil.checkP0(validable, p0, ValidationUtil.checkValidationActionOnError(validationLevel), reportNode);
    }

    private static ValidationLevel checkP0(Validable validable, double p0, ActionOnError actionOnError, ReportNode reportNode) {
        if (Double.isNaN(p0)) {
            ValidationUtil.throwExceptionOrLogError(validable, "p0 is invalid", actionOnError, id -> NetworkReports.invalidP0(reportNode, id));
            return ValidationLevel.EQUIPMENT;
        }
        return ValidationLevel.STEADY_STATE_HYPOTHESIS;
    }

    public static ValidationLevel checkQ0(Validable validable, double q0, ValidationLevel validationLevel, ReportNode reportNode) {
        return ValidationUtil.checkQ0(validable, q0, ValidationUtil.checkValidationActionOnError(validationLevel), reportNode);
    }

    private static ValidationLevel checkQ0(Validable validable, double q0, ActionOnError actionOnError, ReportNode reportNode) {
        if (Double.isNaN(q0)) {
            ValidationUtil.throwExceptionOrLogError(validable, "q0 is invalid", actionOnError, id -> NetworkReports.invalidQ0(reportNode, id));
            return ValidationLevel.EQUIPMENT;
        }
        return ValidationLevel.STEADY_STATE_HYPOTHESIS;
    }

    public static void checkR(Validable validable, double r) {
        if (Double.isNaN(r)) {
            throw new ValidationException(validable, "r is invalid");
        }
    }

    public static void checkX(Validable validable, double x) {
        if (Double.isNaN(x)) {
            throw new ValidationException(validable, "x is invalid");
        }
    }

    public static void checkG1(Validable validable, double g1) {
        if (Double.isNaN(g1)) {
            throw new ValidationException(validable, "g1 is invalid");
        }
    }

    public static void checkG2(Validable validable, double g2) {
        if (Double.isNaN(g2)) {
            throw new ValidationException(validable, "g2 is invalid");
        }
    }

    public static void checkB1(Validable validable, double b1) {
        if (Double.isNaN(b1)) {
            throw new ValidationException(validable, "b1 is invalid");
        }
    }

    public static void checkB2(Validable validable, double b2) {
        if (Double.isNaN(b2)) {
            throw new ValidationException(validable, "b2 is invalid");
        }
    }

    public static void checkG(Validable validable, double g) {
        if (Double.isNaN(g)) {
            throw new ValidationException(validable, "g is invalid");
        }
    }

    public static void checkB(Validable validable, double b) {
        if (Double.isNaN(b)) {
            throw new ValidationException(validable, "b is invalid");
        }
    }

    public static void checkNominalV(Validable validable, double nominalV) {
        if (Double.isNaN(nominalV) || nominalV <= 0.0) {
            throw new ValidationException(validable, "nominal voltage is invalid");
        }
    }

    public static void checkVoltageLimits(Validable validable, double lowVoltageLimit, double highVoltageLimit) {
        if (lowVoltageLimit < 0.0) {
            throw new ValidationException(validable, "low voltage limit is < 0");
        }
        if (highVoltageLimit < 0.0) {
            throw new ValidationException(validable, "high voltage limit is < 0");
        }
        if (lowVoltageLimit > highVoltageLimit) {
            throw new ValidationException(validable, "Inconsistent voltage limit range [" + lowVoltageLimit + ", " + highVoltageLimit + "]");
        }
    }

    public static void checkTopologyKind(Validable validable, TopologyKind topologyKind) {
        if (topologyKind == null) {
            throw new ValidationException(validable, "topology kind is invalid");
        }
    }

    public static void checkCaseDate(Validable validable, ZonedDateTime caseDate) {
        if (caseDate == null) {
            throw new ValidationException(validable, "case date is invalid");
        }
    }

    public static void checkForecastDistance(Validable validable, int forecastDistance) {
        if (forecastDistance < 0) {
            throw new ValidationException(validable, "forecast distance < 0");
        }
    }

    public static void checkBPerSection(Validable validable, double sectionB) {
        if (Double.isNaN(sectionB)) {
            throw new ValidationException(validable, "section susceptance is invalid");
        }
    }

    public static void checkMaximumSectionCount(Validable validable, int maximumSectionCount) {
        if (maximumSectionCount <= 0) {
            throw new ValidationException(validable, "the maximum number of section (" + maximumSectionCount + ") should be greater than 0");
        }
    }

    public static ValidationLevel checkSections(Validable validable, Integer currentSectionCount, int maximumSectionCount, ValidationLevel validationLevel, ReportNode reportNode) {
        return ValidationUtil.checkSections(validable, currentSectionCount, maximumSectionCount, ValidationUtil.checkValidationActionOnError(validationLevel), reportNode);
    }

    private static ValidationLevel checkSections(Validable validable, Integer currentSectionCount, int maximumSectionCount, ActionOnError actionOnError, ReportNode reportNode) {
        ValidationUtil.checkMaximumSectionCount(validable, maximumSectionCount);
        if (currentSectionCount == null) {
            ValidationUtil.throwExceptionOrLogError(validable, "the current number of section is undefined", actionOnError, id -> NetworkReports.undefinedShuntCompensatorSection(reportNode, id));
            return ValidationLevel.EQUIPMENT;
        }
        if (currentSectionCount < 0) {
            throw new ValidationException(validable, "the current number of section (" + currentSectionCount + ") should be greater than or equal to 0");
        }
        if (currentSectionCount > maximumSectionCount) {
            throw new ValidationException(validable, "the current number (" + currentSectionCount + ") of section should be lesser than the maximum number of section (" + maximumSectionCount + ")");
        }
        return ValidationLevel.STEADY_STATE_HYPOTHESIS;
    }

    public static void checkRatedU(Validable validable, double ratedU, String num) {
        if (Double.isNaN(ratedU)) {
            throw new ValidationException(validable, "rated U" + num + " is invalid");
        }
    }

    public static void checkRatedU1(Validable validable, double ratedU1) {
        ValidationUtil.checkRatedU(validable, ratedU1, "1");
    }

    public static void checkRatedU2(Validable validable, double ratedU2) {
        ValidationUtil.checkRatedU(validable, ratedU2, "2");
    }

    public static ValidationLevel checkSvcRegulator(Validable validable, Boolean regulating, double voltageSetpoint, double reactivePowerSetpoint, StaticVarCompensator.RegulationMode regulationMode, ValidationLevel validationLevel, ReportNode reportNode) {
        return ValidationUtil.checkSvcRegulator(validable, regulating, voltageSetpoint, reactivePowerSetpoint, regulationMode, ValidationUtil.checkValidationActionOnError(validationLevel), reportNode);
    }

    private static ValidationLevel checkSvcRegulator(Validable validable, Boolean regulating, double voltageSetpoint, double reactivePowerSetpoint, StaticVarCompensator.RegulationMode regulationMode, ActionOnError actionOnError, ReportNode reportNode) {
        if (regulating == null) {
            throw new ValidationException(validable, "regulating is not set");
        }
        if (regulationMode == null) {
            ValidationUtil.throwExceptionOrLogError(validable, "Regulation mode is invalid", actionOnError, id -> NetworkReports.svcRegulationModeInvalid(reportNode, id));
            return ValidationLevel.EQUIPMENT;
        }
        if (regulating.booleanValue()) {
            if (regulationMode == StaticVarCompensator.RegulationMode.VOLTAGE && Double.isNaN(voltageSetpoint)) {
                ValidationUtil.throwExceptionOrLogErrorForInvalidValue(validable, voltageSetpoint, VOLTAGE_SETPOINT, actionOnError, id -> NetworkReports.svcVoltageSetpointInvalid(reportNode, id, voltageSetpoint));
                return ValidationLevel.EQUIPMENT;
            }
            if (regulationMode == StaticVarCompensator.RegulationMode.REACTIVE_POWER && Double.isNaN(reactivePowerSetpoint)) {
                ValidationUtil.throwExceptionOrLogErrorForInvalidValue(validable, reactivePowerSetpoint, "reactive power setpoint", actionOnError, id -> NetworkReports.svcReactivePowerSetpointInvalid(reportNode, id, reactivePowerSetpoint));
                return ValidationLevel.EQUIPMENT;
            }
        }
        return ValidationLevel.STEADY_STATE_HYPOTHESIS;
    }

    public static void checkBmin(Validable validable, double bMin) {
        if (Double.isNaN(bMin)) {
            throw new ValidationException(validable, "bmin is invalid");
        }
    }

    public static void checkBmax(Validable validable, double bMax) {
        if (Double.isNaN(bMax)) {
            throw new ValidationException(validable, "bmax is invalid");
        }
    }

    public static void checkDoubleParamPositive(Validable validable, double param, String paramName) {
        if (Double.isNaN(param) || param < 0.0) {
            throw new ValidationException(validable, paramName + " is invalid");
        }
    }

    public static ValidationLevel checkRatioTapChangerRegulation(Validable validable, boolean regulating, boolean loadTapChangingCapabilities, Terminal regulationTerminal, RatioTapChanger.RegulationMode regulationMode, double regulationValue, Network network, ValidationLevel validationLevel, ReportNode reportNode) {
        return ValidationUtil.checkRatioTapChangerRegulation(validable, regulating, loadTapChangingCapabilities, regulationTerminal, regulationMode, regulationValue, network, ValidationUtil.checkValidationActionOnError(validationLevel), reportNode);
    }

    private static ValidationLevel checkRatioTapChangerRegulation(Validable validable, boolean regulating, boolean loadTapChangingCapabilities, Terminal regulationTerminal, RatioTapChanger.RegulationMode regulationMode, double regulationValue, Network network, ActionOnError actionOnError, ReportNode reportNode) {
        ValidationLevel validationLevel = ValidationLevel.STEADY_STATE_HYPOTHESIS;
        if (regulating) {
            if (!loadTapChangingCapabilities) {
                ValidationUtil.throwExceptionOrLogError(validable, "regulation cannot be enabled on ratio tap changer without load tap changing capabilities", actionOnError, id -> NetworkReports.rtcRegulationCannotBeEnabledWithoutLoadTapChanging(reportNode, id));
                validationLevel = ValidationLevel.min(validationLevel, ValidationLevel.EQUIPMENT);
            }
            if (Objects.isNull((Object)regulationMode)) {
                ValidationUtil.throwExceptionOrLogError(validable, "regulation mode of regulating ratio tap changer must be given", actionOnError, id -> NetworkReports.regulatingRtcNoRegulationMode(reportNode, id));
                validationLevel = ValidationLevel.min(validationLevel, ValidationLevel.EQUIPMENT);
            }
            if (Double.isNaN(regulationValue)) {
                ValidationUtil.throwExceptionOrLogError(validable, "a regulation value has to be set for a regulating ratio tap changer", actionOnError, id -> NetworkReports.regulatingRtcNoRegulationValue(reportNode, id));
                validationLevel = ValidationLevel.min(validationLevel, ValidationLevel.EQUIPMENT);
            }
            if (regulationMode == RatioTapChanger.RegulationMode.VOLTAGE && regulationValue <= 0.0) {
                ValidationUtil.throwExceptionOrLogError(validable, "bad target voltage " + regulationValue, actionOnError, id -> NetworkReports.regulatingRtcBadTargetVoltage(reportNode, id, regulationValue));
                validationLevel = ValidationLevel.min(validationLevel, ValidationLevel.EQUIPMENT);
            }
            if (regulationTerminal == null) {
                ValidationUtil.throwExceptionOrLogError(validable, "a regulation terminal has to be set for a regulating ratio tap changer", actionOnError, id -> NetworkReports.regulatingRtcNoRegulationTerminal(reportNode, id));
                validationLevel = ValidationLevel.min(validationLevel, ValidationLevel.EQUIPMENT);
            }
        }
        if (regulationTerminal != null && regulationTerminal.getVoltageLevel().getNetwork() != network) {
            throw new ValidationException(validable, "regulation terminal is not part of the network");
        }
        return validationLevel;
    }

    public static ValidationLevel checkPhaseTapChangerRegulation(Validable validable, PhaseTapChanger.RegulationMode regulationMode, double regulationValue, boolean regulating, boolean loadTapChangingCapabilities, Terminal regulationTerminal, Network network, ValidationLevel validationLevel, ReportNode reportNode) {
        return ValidationUtil.checkPhaseTapChangerRegulation(validable, regulationMode, regulationValue, regulating, loadTapChangingCapabilities, regulationTerminal, network, ValidationUtil.checkValidationActionOnError(validationLevel), reportNode);
    }

    private static ValidationLevel checkPhaseTapChangerRegulation(Validable validable, PhaseTapChanger.RegulationMode regulationMode, double regulationValue, boolean regulating, boolean loadTapChangingCapabilities, Terminal regulationTerminal, Network network, ActionOnError actionOnError, ReportNode reportNode) {
        ValidationLevel validationLevel = ValidationLevel.STEADY_STATE_HYPOTHESIS;
        if (loadTapChangingCapabilities && regulationMode == null) {
            ValidationUtil.throwExceptionOrLogError(validable, "phase regulation mode is not set", actionOnError, id -> NetworkReports.ptcPhaseRegulationModeNotSet(reportNode, id));
            validationLevel = ValidationLevel.min(validationLevel, ValidationLevel.EQUIPMENT);
        }
        if (regulating && regulationMode != null) {
            validationLevel = ValidationUtil.checkRegulatingPhaseTapChanger(validable, regulationMode, regulationValue, regulationTerminal, loadTapChangingCapabilities, actionOnError, reportNode);
        }
        if (regulationTerminal != null && regulationTerminal.getVoltageLevel().getNetwork() != network) {
            throw new ValidationException(validable, "phase regulation terminal is not part of the network");
        }
        return validationLevel;
    }

    private static ValidationLevel checkRegulatingPhaseTapChanger(Validable validable, PhaseTapChanger.RegulationMode regulationMode, double regulationValue, Terminal regulationTerminal, boolean loadTapChangingCapabilities, ActionOnError actionOnError, ReportNode reportNode) {
        if (regulationMode == PhaseTapChanger.RegulationMode.CURRENT_LIMITER && regulationValue < 0.0) {
            throw new ValidationException(validable, "phase tap changer in CURRENT_LIMITER mode must have a non-negative regulation value");
        }
        ValidationLevel validationLevel = ValidationLevel.STEADY_STATE_HYPOTHESIS;
        if (!loadTapChangingCapabilities) {
            ValidationUtil.throwExceptionOrLogError(validable, "regulation cannot be enabled on phase tap changer without load tap changing capabilities", actionOnError, id -> NetworkReports.ptcPhaseRegulationCannotBeEnabledWithoutLoadTapChanging(reportNode, id));
            validationLevel = ValidationLevel.min(validationLevel, ValidationLevel.EQUIPMENT);
        }
        if (Double.isNaN(regulationValue)) {
            ValidationUtil.throwExceptionOrLogError(validable, "phase regulation is on and threshold/setpoint value is not set", actionOnError, id -> NetworkReports.ptcPhaseRegulationRegulationValueNotSet(reportNode, id));
            validationLevel = ValidationLevel.min(validationLevel, ValidationLevel.EQUIPMENT);
        }
        if (regulationTerminal == null) {
            ValidationUtil.throwExceptionOrLogError(validable, "phase regulation is on and regulated terminal is not set", actionOnError, id -> NetworkReports.ptcPhaseRegulationNoRegulatedTerminal(reportNode, id));
            validationLevel = ValidationLevel.min(validationLevel, ValidationLevel.EQUIPMENT);
        }
        return validationLevel;
    }

    public static ValidationLevel checkOnlyOneTapChangerRegulatingEnabled(Validable validable, Set<TapChanger<?, ?, ?, ?>> tapChangersNotIncludingTheModified, boolean regulating, ValidationLevel validationLevel, ReportNode reportNode) {
        return ValidationUtil.checkOnlyOneTapChangerRegulatingEnabled(validable, tapChangersNotIncludingTheModified, regulating, ValidationUtil.checkValidationActionOnError(validationLevel), reportNode);
    }

    private static ValidationLevel checkOnlyOneTapChangerRegulatingEnabled(Validable validable, Set<TapChanger<?, ?, ?, ?>> tapChangersNotIncludingTheModified, boolean regulating, ActionOnError actionOnError, ReportNode reportNode) {
        if (regulating && tapChangersNotIncludingTheModified.stream().anyMatch(TapChanger::isRegulating)) {
            ValidationUtil.throwExceptionOrLogError(validable, UNIQUE_REGULATING_TAP_CHANGER_MSG, actionOnError, id -> NetworkReports.tooManyRegulatingControlEnabled(reportNode, id));
            return ValidationLevel.EQUIPMENT;
        }
        return ValidationLevel.STEADY_STATE_HYPOTHESIS;
    }

    public static ValidationLevel checkConvertersMode(Validable validable, HvdcLine.ConvertersMode converterMode, ValidationLevel validationLevel, ReportNode reportNode) {
        return ValidationUtil.checkConvertersMode(validable, converterMode, ValidationUtil.checkValidationActionOnError(validationLevel), reportNode);
    }

    private static ValidationLevel checkConvertersMode(Validable validable, HvdcLine.ConvertersMode converterMode, ActionOnError actionOnError, ReportNode reportNode) {
        if (converterMode == null) {
            ValidationUtil.throwExceptionOrLogError(validable, "converter mode is invalid", actionOnError, id -> NetworkReports.hvdcConverterModeInvalid(reportNode, id));
            return ValidationLevel.EQUIPMENT;
        }
        return ValidationLevel.STEADY_STATE_HYPOTHESIS;
    }

    public static void checkPowerFactor(Validable validable, double powerFactor) {
        if (Double.isNaN(powerFactor)) {
            throw new ValidationException(validable, "power factor is invalid");
        }
        if (Math.abs(powerFactor) > 1.0) {
            throw new ValidationException(validable, "power factor is invalid, it should be between -1 and 1");
        }
    }

    public static void checkPositivePowerFactor(Validable validable, double powerFactor) {
        if (Double.isNaN(powerFactor)) {
            throw new ValidationException(validable, "power factor is invalid");
        }
        if (powerFactor < 0.0 || powerFactor > 1.0) {
            throw new ValidationException(validable, "power factor is invalid, it must be between 0 and 1");
        }
    }

    public static DcNode checkAndGetDcNode(Network network, Validable validable, String dcNodeId, String attribute) {
        if (dcNodeId == null) {
            throw new ValidationException(validable, attribute + " is not set");
        }
        DcNode dcNode = network.getDcNode(dcNodeId);
        if (dcNode == null) {
            throw new ValidationException(validable, "DcNode '" + dcNodeId + "' not found");
        }
        return dcNode;
    }

    public static void checkSameParentNetwork(Network validableNetwork, Validable validable, DcNode dcNode) {
        if (validableNetwork != dcNode.getParentNetwork()) {
            throw new ValidationException(validable, "DC Node '" + dcNode.getId() + "' is in network '" + dcNode.getParentNetwork().getId() + "' but DC Equipment is in '" + validableNetwork.getId() + "'");
        }
    }

    public static void checkSameParentNetwork(Network validableNetwork, Validable validable, DcNode dcNode1, DcNode dcNode2) {
        if (dcNode1.getParentNetwork() != dcNode2.getParentNetwork()) {
            throw new ValidationException(validable, "DC Nodes '" + dcNode1.getId() + "' and '" + dcNode2.getId() + "' are in different networks '" + dcNode1.getParentNetwork().getId() + "' and '" + dcNode2.getParentNetwork().getId() + "'");
        }
        if (validableNetwork != dcNode1.getParentNetwork()) {
            throw new ValidationException(validable, "DC Nodes '" + dcNode1.getId() + "' and '" + dcNode2.getId() + "' are in network '" + dcNode1.getParentNetwork().getId() + "' but DC Equipment is in '" + validableNetwork.getId() + "'");
        }
    }

    public static ValidationLevel checkAcDcConverterControl(Validable validable, AcDcConverter.ControlMode controlMode, double targetP, double targetVdc, ValidationLevel validationLevel, ReportNode reportNode) {
        return ValidationUtil.checkAcDcConverterControl(validable, controlMode, targetP, targetVdc, ValidationUtil.checkValidationActionOnError(validationLevel), reportNode);
    }

    private static ValidationLevel checkAcDcConverterControl(Validable validable, AcDcConverter.ControlMode controlMode, double targetP, double targetVdc, ActionOnError actionOnError, ReportNode reportNode) {
        ValidationLevel validationLevel = ValidationLevel.STEADY_STATE_HYPOTHESIS;
        if (controlMode == null) {
            ValidationUtil.throwExceptionOrLogError(validable, "controlMode is not set", actionOnError, id -> NetworkReports.acDcConverterControlModeNotSet(reportNode, id));
            validationLevel = ValidationLevel.min(validationLevel, ValidationLevel.EQUIPMENT);
        }
        if (Objects.equals((Object)controlMode, (Object)AcDcConverter.ControlMode.P_PCC) && Double.isNaN(targetP)) {
            ValidationUtil.throwExceptionOrLogError(validable, "targetP is invalid", actionOnError, id -> NetworkReports.acDcConverterControlTargetPInvalid(reportNode, id));
            validationLevel = ValidationLevel.min(validationLevel, ValidationLevel.EQUIPMENT);
        }
        if (Objects.equals((Object)controlMode, (Object)AcDcConverter.ControlMode.V_DC) && Double.isNaN(targetVdc)) {
            ValidationUtil.throwExceptionOrLogError(validable, "targetVdc is invalid", actionOnError, id -> NetworkReports.acDcConverterControlInvalidTargetVDc(reportNode, id));
            validationLevel = ValidationLevel.min(validationLevel, ValidationLevel.EQUIPMENT);
        }
        return validationLevel;
    }

    public static void checkAcDcConverterPccTerminal(Validable validable, boolean twoAcTerminals, Terminal pccTerminal, VoltageLevel voltageLevel) {
        if (twoAcTerminals && pccTerminal == null) {
            throw new ValidationException(validable, "converter has two AC terminals and pccTerminal is not set");
        }
        if (pccTerminal != null) {
            Connectable c = pccTerminal.getConnectable();
            if (twoAcTerminals && !(c instanceof Branch) && !(c instanceof ThreeWindingsTransformer)) {
                throw new ValidationException(validable, "converter has two AC terminals and pccTerminal is not a line or transformer terminal");
            }
            if (!(twoAcTerminals || c instanceof Branch || c instanceof ThreeWindingsTransformer || c instanceof AcDcConverter)) {
                throw new ValidationException(validable, "pccTerminal is not a line or transformer or the converter terminal");
            }
            if (!twoAcTerminals && c instanceof AcDcConverter && c != validable) {
                throw new ValidationException(validable, "pccTerminal cannot be the terminal of another converter");
            }
            if (c.getParentNetwork() != voltageLevel.getParentNetwork()) {
                throw new ValidationException(validable, "pccTerminal is not in the same parent network as the voltage level");
            }
        }
    }

    public static void checkLccReactiveModel(Validable validable, LineCommutatedConverter.ReactiveModel reactiveModel) {
        if (reactiveModel == null) {
            throw new ValidationException(validable, "reactiveModel is not set");
        }
    }

    public static ValidationLevel checkLoadingLimits(Validable validable, double permanentLimit, Collection<LoadingLimits.TemporaryLimit> temporaryLimits, ValidationLevel validationLevel, ReportNode reportNode) {
        return ValidationUtil.checkLoadingLimits(validable, permanentLimit, temporaryLimits, ValidationUtil.checkValidationActionOnError(validationLevel), reportNode);
    }

    private static ValidationLevel checkLoadingLimits(Validable validable, double permanentLimit, Collection<LoadingLimits.TemporaryLimit> temporaryLimits, ActionOnError actionOnError, ReportNode reportNode) {
        ValidationLevel validationLevel = ValidationUtil.checkPermanentLimit(validable, permanentLimit, temporaryLimits, actionOnError, reportNode);
        ValidationUtil.checkTemporaryLimits(validable, permanentLimit, temporaryLimits);
        return validationLevel;
    }

    public static ValidationLevel checkPermanentLimit(Validable validable, double permanentLimit, Collection<LoadingLimits.TemporaryLimit> temporaryLimits, ValidationLevel validationLevel, ReportNode reportNode) {
        return ValidationUtil.checkPermanentLimit(validable, permanentLimit, temporaryLimits, ValidationUtil.checkValidationActionOnError(validationLevel), reportNode);
    }

    private static ValidationLevel checkPermanentLimit(Validable validable, double permanentLimit, Collection<LoadingLimits.TemporaryLimit> temporaryLimits, ActionOnError actionOnError, ReportNode reportNode) {
        ValidationLevel validationLevel = ValidationLevel.STEADY_STATE_HYPOTHESIS;
        if (Double.isNaN(permanentLimit) && !temporaryLimits.isEmpty()) {
            ValidationUtil.throwExceptionOrLogError(validable, "permanent limit must be defined if temporary limits are present", actionOnError, id -> NetworkReports.temporaryLimitsButPermanentLimitUndefined(reportNode, id));
            validationLevel = ValidationLevel.min(validationLevel, ValidationLevel.EQUIPMENT);
        }
        if (permanentLimit < 0.0) {
            throw new ValidationException(validable, "permanent limit must be >= 0");
        }
        if (permanentLimit == 0.0) {
            LOGGER.info("{}permanent limit is set to 0", (Object)validable.getMessageHeader());
        }
        return validationLevel;
    }

    private static void checkTemporaryLimits(Validable validable, double permanentLimit, Collection<LoadingLimits.TemporaryLimit> temporaryLimits) {
        if (LOGGER.isDebugEnabled()) {
            double previousLimit = Double.NaN;
            boolean wrongOrderMessageAlreadyLogged = false;
            for (LoadingLimits.TemporaryLimit tl : temporaryLimits) {
                if (tl.getValue() <= permanentLimit) {
                    LOGGER.debug("{}, temporary limit should be greater than permanent limit", (Object)validable.getMessageHeader());
                }
                if (!wrongOrderMessageAlreadyLogged && !Double.isNaN(previousLimit) && tl.getValue() <= previousLimit) {
                    LOGGER.debug("{} : temporary limits should be in ascending value order", (Object)validable.getMessageHeader());
                    wrongOrderMessageAlreadyLogged = true;
                }
                previousLimit = tl.getValue();
            }
        }
    }

    public static ValidationLevel checkLossFactor(Validable validable, float lossFactor, ValidationLevel validationLevel, ReportNode reportNode) {
        return ValidationUtil.checkLossFactor(validable, lossFactor, ValidationUtil.checkValidationActionOnError(validationLevel), reportNode);
    }

    private static ValidationLevel checkLossFactor(Validable validable, float lossFactor, ActionOnError actionOnError, ReportNode reportNode) {
        ValidationLevel validationLevel = ValidationLevel.STEADY_STATE_HYPOTHESIS;
        if (Double.isNaN(lossFactor)) {
            ValidationUtil.throwExceptionOrLogError(validable, "loss factor is invalid is undefined", actionOnError, id -> NetworkReports.hvdcUndefinedLossFactor(reportNode, id));
            validationLevel = ValidationLevel.min(validationLevel, ValidationLevel.EQUIPMENT);
        } else if (lossFactor < 0.0f || lossFactor > 100.0f) {
            throw new ValidationException(validable, "loss factor must be >= 0 and <= 100");
        }
        return validationLevel;
    }

    private static ValidationLevel checkRtc(Validable validable, RatioTapChanger rtc, Network network, ActionOnError actionOnError, ReportNode reportNode) {
        ValidationLevel validationLevel = ValidationLevel.STEADY_STATE_HYPOTHESIS;
        if (rtc.findTapPosition().isEmpty()) {
            ValidationUtil.throwExceptionOrLogError(validable, "tap position is not set", actionOnError, id -> NetworkReports.tapPositionNotSet(reportNode, id));
            validationLevel = ValidationLevel.min(validationLevel, ValidationLevel.EQUIPMENT);
        }
        validationLevel = ValidationLevel.min(validationLevel, ValidationUtil.checkRatioTapChangerRegulation(validable, rtc.isRegulating(), rtc.hasLoadTapChangingCapabilities(), rtc.getRegulationTerminal(), rtc.getRegulationMode(), rtc.getRegulationValue(), network, actionOnError, reportNode));
        validationLevel = ValidationLevel.min(validationLevel, ValidationUtil.checkTargetDeadband(validable, "ratio tap changer", rtc.isRegulating(), rtc.getTargetDeadband(), actionOnError, reportNode));
        return validationLevel;
    }

    private static ValidationLevel checkPtc(Validable validable, PhaseTapChanger ptc, Network network, ActionOnError actionOnError, ReportNode reportNode) {
        ValidationLevel validationLevel = ValidationLevel.STEADY_STATE_HYPOTHESIS;
        if (ptc.findTapPosition().isEmpty()) {
            ValidationUtil.throwExceptionOrLogError(validable, "tap position is not set", actionOnError, id -> NetworkReports.tapPositionNotSet(reportNode, id));
            validationLevel = ValidationLevel.min(validationLevel, ValidationLevel.EQUIPMENT);
        }
        validationLevel = ValidationLevel.min(validationLevel, ValidationUtil.checkPhaseTapChangerRegulation(validable, ptc.getRegulationMode(), ptc.getRegulationValue(), ptc.isRegulating(), ptc.hasLoadTapChangingCapabilities(), ptc.getRegulationTerminal(), network, actionOnError, reportNode));
        validationLevel = ValidationLevel.min(validationLevel, ValidationUtil.checkTargetDeadband(validable, "phase tap changer", ptc.isRegulating(), ptc.getTargetDeadband(), actionOnError, reportNode));
        return validationLevel;
    }

    private static ValidationLevel checkThreeWindingsTransformer(Validable validable, ThreeWindingsTransformer twt, ActionOnError actionOnError, ReportNode reportNode) {
        ValidationLevel validationLevel = ValidationLevel.STEADY_STATE_HYPOTHESIS;
        for (ThreeWindingsTransformer.Leg leg : twt.getLegs()) {
            if (leg.hasRatioTapChanger()) {
                validationLevel = ValidationLevel.min(validationLevel, ValidationUtil.checkRtc(validable, leg.getRatioTapChanger(), twt.getNetwork(), actionOnError, reportNode));
            }
            if (!leg.hasPhaseTapChanger()) continue;
            validationLevel = ValidationLevel.min(validationLevel, ValidationUtil.checkPtc(validable, leg.getPhaseTapChanger(), twt.getNetwork(), actionOnError, reportNode));
        }
        long regulatingTc = twt.getLegStream().map(RatioTapChangerHolder::getRatioTapChanger).filter(Objects::nonNull).filter(TapChanger::isRegulating).count() + twt.getLegStream().map(PhaseTapChangerHolder::getPhaseTapChanger).filter(Objects::nonNull).filter(TapChanger::isRegulating).count();
        if (regulatingTc > 1L) {
            throw new ValidationException(validable, UNIQUE_REGULATING_TAP_CHANGER_MSG);
        }
        validationLevel = ValidationUtil.checkOperationalLimitsGroups(validable, twt.getLeg1().getOperationalLimitsGroups(), validationLevel, actionOnError, reportNode);
        validationLevel = ValidationUtil.checkOperationalLimitsGroups(validable, twt.getLeg2().getOperationalLimitsGroups(), validationLevel, actionOnError, reportNode);
        validationLevel = ValidationUtil.checkOperationalLimitsGroups(validable, twt.getLeg3().getOperationalLimitsGroups(), validationLevel, actionOnError, reportNode);
        return validationLevel;
    }

    private static ValidationLevel checkTwoWindingsTransformer(Validable validable, TwoWindingsTransformer twt, ActionOnError actionOnError, ReportNode reportNode) {
        ValidationLevel validationLevel = ValidationLevel.STEADY_STATE_HYPOTHESIS;
        if (twt.hasRatioTapChanger()) {
            validationLevel = ValidationLevel.min(validationLevel, ValidationUtil.checkRtc(validable, twt.getRatioTapChanger(), twt.getNetwork(), actionOnError, reportNode));
        }
        if (twt.hasPhaseTapChanger()) {
            validationLevel = ValidationLevel.min(validationLevel, ValidationUtil.checkPtc(validable, twt.getPhaseTapChanger(), twt.getNetwork(), actionOnError, reportNode));
        }
        if (twt.getOptionalRatioTapChanger().map(TapChanger::isRegulating).orElse(false).booleanValue() && twt.getOptionalPhaseTapChanger().map(TapChanger::isRegulating).orElse(false).booleanValue()) {
            throw new ValidationException(validable, UNIQUE_REGULATING_TAP_CHANGER_MSG);
        }
        validationLevel = ValidationUtil.checkOperationalLimitsGroups(validable, twt.getOperationalLimitsGroups1(), validationLevel, actionOnError, reportNode);
        validationLevel = ValidationUtil.checkOperationalLimitsGroups(validable, twt.getOperationalLimitsGroups2(), validationLevel, actionOnError, reportNode);
        return validationLevel;
    }

    private static ValidationLevel checkIdentifiable(Identifiable<?> identifiable, ValidationLevel previous, ActionOnError actionOnError, ReportNode reportNode) {
        ValidationLevel validationLevel = previous;
        if (identifiable instanceof Validable) {
            Validable validable = (Validable)((Object)identifiable);
            if (identifiable instanceof Battery) {
                Battery battery = (Battery)identifiable;
                validationLevel = ValidationLevel.min(validationLevel, ValidationUtil.checkP0(validable, battery.getTargetP(), actionOnError, reportNode));
                validationLevel = ValidationLevel.min(validationLevel, ValidationUtil.checkQ0(validable, battery.getTargetQ(), actionOnError, reportNode));
            } else if (identifiable instanceof DanglingLine) {
                DanglingLine danglingLine = (DanglingLine)identifiable;
                validationLevel = ValidationLevel.min(validationLevel, ValidationUtil.checkP0(validable, danglingLine.getP0(), actionOnError, reportNode));
                validationLevel = ValidationLevel.min(validationLevel, ValidationUtil.checkQ0(validable, danglingLine.getQ0(), actionOnError, reportNode));
                validationLevel = ValidationUtil.checkGenerationOnDanglingLine(validationLevel, validable, danglingLine, actionOnError, reportNode);
                validationLevel = ValidationUtil.checkOperationalLimitsGroups(validable, danglingLine.getOperationalLimitsGroups(), validationLevel, actionOnError, reportNode);
            } else if (identifiable instanceof Generator) {
                Generator generator = (Generator)identifiable;
                validationLevel = ValidationLevel.min(validationLevel, ValidationUtil.checkActivePowerSetpoint(validable, generator.getTargetP(), actionOnError, reportNode));
                validationLevel = ValidationLevel.min(validationLevel, ValidationUtil.checkVoltageControl(validable, (Boolean)generator.isVoltageRegulatorOn(), generator.getTargetV(), generator.getTargetQ(), actionOnError, reportNode));
            } else if (identifiable instanceof HvdcLine) {
                HvdcLine hvdcLine = (HvdcLine)identifiable;
                validationLevel = ValidationLevel.min(validationLevel, ValidationUtil.checkConvertersMode(validable, hvdcLine.getConvertersMode(), actionOnError, reportNode));
                validationLevel = ValidationLevel.min(validationLevel, ValidationUtil.checkHvdcActivePowerSetpoint(validable, hvdcLine.getActivePowerSetpoint(), actionOnError, reportNode));
            } else if (identifiable instanceof Load) {
                Load load = (Load)identifiable;
                validationLevel = ValidationLevel.min(validationLevel, ValidationUtil.checkP0(validable, load.getP0(), actionOnError, reportNode));
                validationLevel = ValidationLevel.min(validationLevel, ValidationUtil.checkQ0(validable, load.getQ0(), actionOnError, reportNode));
            } else if (identifiable instanceof ShuntCompensator) {
                ShuntCompensator shunt = (ShuntCompensator)identifiable;
                validationLevel = ValidationLevel.min(validationLevel, ValidationUtil.checkVoltageControl(validable, shunt.isVoltageRegulatorOn(), shunt.getTargetV(), actionOnError, reportNode));
                validationLevel = ValidationLevel.min(validationLevel, ValidationUtil.checkTargetDeadband(validable, "shunt compensator", shunt.isVoltageRegulatorOn(), shunt.getTargetDeadband(), actionOnError, reportNode));
                validationLevel = ValidationLevel.min(validationLevel, ValidationUtil.checkSections(validable, ValidationUtil.getSectionCount(shunt), shunt.getMaximumSectionCount(), actionOnError, reportNode));
            } else if (identifiable instanceof StaticVarCompensator) {
                StaticVarCompensator svc = (StaticVarCompensator)identifiable;
                validationLevel = ValidationLevel.min(validationLevel, ValidationUtil.checkSvcRegulator(validable, (Boolean)svc.isRegulating(), svc.getVoltageSetpoint(), svc.getReactivePowerSetpoint(), svc.getRegulationMode(), actionOnError, reportNode));
            } else if (identifiable instanceof ThreeWindingsTransformer) {
                ThreeWindingsTransformer twt = (ThreeWindingsTransformer)identifiable;
                validationLevel = ValidationLevel.min(validationLevel, ValidationUtil.checkThreeWindingsTransformer(validable, twt, actionOnError, reportNode));
            } else if (identifiable instanceof TwoWindingsTransformer) {
                TwoWindingsTransformer twt = (TwoWindingsTransformer)identifiable;
                validationLevel = ValidationLevel.min(validationLevel, ValidationUtil.checkTwoWindingsTransformer(validable, twt, actionOnError, reportNode));
            } else if (identifiable instanceof VscConverterStation) {
                VscConverterStation converterStation = (VscConverterStation)identifiable;
                validationLevel = ValidationLevel.min(validationLevel, ValidationUtil.checkVoltageControl(validable, (Boolean)converterStation.isVoltageRegulatorOn(), converterStation.getVoltageSetpoint(), converterStation.getReactivePowerSetpoint(), actionOnError, reportNode));
            } else if (identifiable instanceof Branch) {
                Branch branch = (Branch)identifiable;
                validationLevel = ValidationUtil.checkOperationalLimitsGroups(validable, branch.getOperationalLimitsGroups1(), validationLevel, actionOnError, reportNode);
                validationLevel = ValidationUtil.checkOperationalLimitsGroups(validable, branch.getOperationalLimitsGroups2(), validationLevel, actionOnError, reportNode);
            }
        }
        return validationLevel;
    }

    private static ValidationLevel checkGenerationOnDanglingLine(ValidationLevel previous, Validable validable, DanglingLine danglingLine, ActionOnError actionOnError, ReportNode reportNode) {
        ValidationLevel validationLevel = previous;
        DanglingLine.Generation generation = danglingLine.getGeneration();
        if (generation != null) {
            validationLevel = ValidationLevel.min(validationLevel, ValidationUtil.checkActivePowerSetpoint(validable, generation.getTargetP(), actionOnError, reportNode));
            validationLevel = ValidationLevel.min(validationLevel, ValidationUtil.checkVoltageControl(validable, (Boolean)generation.isVoltageRegulationOn(), generation.getTargetV(), generation.getTargetQ(), actionOnError, reportNode));
        }
        return validationLevel;
    }

    private static Integer getSectionCount(ShuntCompensator shunt) {
        return shunt.findSectionCount().isPresent() ? Integer.valueOf(shunt.getSectionCount()) : null;
    }

    private static ValidationLevel checkOperationalLimitsGroups(Validable validable, Collection<OperationalLimitsGroup> operationalLimitsGroupCollection, ValidationLevel previous, ActionOnError actionOnError, ReportNode reportNode) {
        ValidationLevel validationLevel = previous;
        for (OperationalLimitsGroup group : operationalLimitsGroupCollection) {
            validationLevel = ValidationUtil.checkOperationalLimitsGroup(validable, group, validationLevel, actionOnError, reportNode);
        }
        return validationLevel;
    }

    private static ValidationLevel checkOperationalLimitsGroup(Validable validable, OperationalLimitsGroup operationalLimitsGroup, ValidationLevel previous, ActionOnError actionOnError, ReportNode reportNode) {
        ValidationLevel[] validationLevel = new ValidationLevel[]{previous};
        operationalLimitsGroup.getCurrentLimits().ifPresent(l -> {
            validationLevel[0] = ValidationUtil.checkLoadingLimits(validable, l, validationLevel[0], actionOnError, reportNode);
        });
        operationalLimitsGroup.getApparentPowerLimits().ifPresent(l -> {
            validationLevel[0] = ValidationUtil.checkLoadingLimits(validable, l, validationLevel[0], actionOnError, reportNode);
        });
        operationalLimitsGroup.getActivePowerLimits().ifPresent(l -> {
            validationLevel[0] = ValidationUtil.checkLoadingLimits(validable, l, validationLevel[0], actionOnError, reportNode);
        });
        return validationLevel[0];
    }

    private static ValidationLevel checkLoadingLimits(Validable validable, LoadingLimits limits, ValidationLevel validationLevel, ActionOnError actionOnError, ReportNode reportNode) {
        return ValidationLevel.min(validationLevel, ValidationUtil.checkLoadingLimits(validable, limits.getPermanentLimit(), limits.getTemporaryLimits(), actionOnError, reportNode));
    }

    public static ValidationLevel validate(Collection<Identifiable<?>> identifiables, boolean allChecks, ActionOnError actionOnError, ValidationLevel previous, ReportNode reportNode) {
        Objects.requireNonNull(identifiables);
        Objects.requireNonNull(previous);
        Objects.requireNonNull(reportNode);
        if (ValidationUtil.checkValidationLevel(previous)) {
            return previous;
        }
        ValidationLevel validationLevel = ValidationLevel.STEADY_STATE_HYPOTHESIS;
        for (Identifiable<?> identifiable : identifiables) {
            validationLevel = ValidationUtil.checkIdentifiable(identifiable, validationLevel, actionOnError, reportNode);
            if (allChecks || validationLevel != ValidationLevel.MINIMUM_VALUE) continue;
            return validationLevel;
        }
        return validationLevel;
    }

    private static boolean checkValidationLevel(ValidationLevel validationLevel) {
        return validationLevel.compareTo(ValidationLevel.STEADY_STATE_HYPOTHESIS) >= 0;
    }

    private static ActionOnError checkValidationActionOnError(ValidationLevel validationLevel) {
        return validationLevel.compareTo(ValidationLevel.STEADY_STATE_HYPOTHESIS) >= 0 ? ActionOnError.THROW_EXCEPTION : ActionOnError.IGNORE;
    }

    public static enum ActionOnError {
        THROW_EXCEPTION,
        LOG_ERROR,
        IGNORE;

    }
}

