/*
 * Decompiled with CFR 0.152.
 */
package com.powsybl.math.matrix;

import com.google.common.base.Strings;
import com.powsybl.math.matrix.AbstractMatrix;
import com.powsybl.math.matrix.DenseLUDecomposition;
import com.powsybl.math.matrix.DenseMatrixFactory;
import com.powsybl.math.matrix.LUDecomposition;
import com.powsybl.math.matrix.Matrix;
import com.powsybl.math.matrix.MatrixException;
import com.powsybl.math.matrix.MatrixFactory;
import com.powsybl.math.matrix.SparseMatrix;
import com.powsybl.math.matrix.SparseMatrixFactory;
import java.io.PrintStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.List;
import java.util.Objects;
import java.util.function.Supplier;

public class DenseMatrix
extends AbstractMatrix {
    private final int rowCount;
    private final int columnCount;
    private final ByteBuffer buffer;

    private static ByteBuffer createBuffer(int rowCount, int columnCount) {
        return ByteBuffer.allocateDirect(rowCount * columnCount * 8).order(ByteOrder.LITTLE_ENDIAN);
    }

    public DenseMatrix(int rowCount, int columnCount, double[] values) {
        this(rowCount, columnCount);
        this.setValues(values);
    }

    public DenseMatrix(int rowCount, int columnCount) {
        this(rowCount, columnCount, () -> DenseMatrix.createBuffer(rowCount, columnCount));
    }

    public DenseMatrix(int rowCount, int columnCount, Supplier<ByteBuffer> bufferSupplier) {
        if (rowCount < 0) {
            throw new MatrixException("row count has to be positive");
        }
        if (columnCount < 0) {
            throw new MatrixException("column count has to be positive");
        }
        this.rowCount = rowCount;
        this.columnCount = columnCount;
        Objects.requireNonNull(bufferSupplier);
        this.buffer = bufferSupplier.get();
        if (this.buffer.capacity() != rowCount * columnCount * 8) {
            throw new MatrixException("values size (" + this.buffer.capacity() + ") is incorrect (should be " + rowCount * columnCount + ")");
        }
    }

    public DenseMatrix(Jama.Matrix matrix) {
        this(matrix.getRowDimension(), matrix.getColumnDimension(), matrix.getColumnPackedCopy());
    }

    public double get(int i, int j) {
        this.checkBounds(i, j);
        return this.getUnsafe(i, j);
    }

    private double getUnsafe(int i, int j) {
        return this.buffer.getDouble(j * 8 * this.rowCount + i * 8);
    }

    @Deprecated
    public double getValue(int i, int j) {
        return this.get(i, j);
    }

    private void setUnsafe(int i, int j, double value) {
        this.buffer.putDouble(j * 8 * this.rowCount + i * 8, value);
    }

    @Override
    public void set(int i, int j, double value) {
        this.checkBounds(i, j);
        this.setUnsafe(i, j, value);
    }

    private void addUnsafe(int i, int j, double value) {
        int index = j * 8 * this.rowCount + i * 8;
        this.buffer.putDouble(index, this.buffer.getDouble(index) + value);
    }

    @Override
    public void add(int i, int j, double value) {
        this.checkBounds(i, j);
        this.addUnsafe(i, j, value);
    }

    @Override
    public Matrix.Element addAndGetElement(int i, int j, double value) {
        this.add(i, j, value);
        return new DenseElement(i, j);
    }

    @Override
    public int addAndGetIndex(int i, int j, double value) {
        this.add(i, j, value);
        return j * this.rowCount + i;
    }

    private void checkElementIndex(int index) {
        if (index < 0 || index >= this.rowCount * this.columnCount) {
            throw new MatrixException("Element index out of bound [0, " + (this.rowCount * this.columnCount - 1) + "]");
        }
    }

    @Override
    public void setAtIndex(int index, double value) {
        this.checkElementIndex(index);
        this.setQuickAtIndex(index, value);
    }

    @Override
    public void setQuickAtIndex(int index, double value) {
        int i = index % this.rowCount;
        int j = index / this.rowCount;
        this.setUnsafe(i, j, value);
    }

    @Override
    public void addAtIndex(int index, double value) {
        this.checkElementIndex(index);
        this.addQuickAtIndex(index, value);
    }

    @Override
    public void addQuickAtIndex(int index, double value) {
        int i = index % this.rowCount;
        int j = index / this.rowCount;
        this.addUnsafe(i, j, value);
    }

    @Override
    public void reset() {
        for (int k = 0; k < this.rowCount * this.columnCount; ++k) {
            this.buffer.putDouble(k * 8, 0.0);
        }
    }

    @Override
    public int getRowCount() {
        return this.rowCount;
    }

    @Override
    public int getColumnCount() {
        return this.columnCount;
    }

    ByteBuffer getBuffer() {
        return this.buffer;
    }

    void setValues(double[] values) {
        if (values.length != this.rowCount * this.columnCount) {
            throw new MatrixException("Incorrect values array size " + values.length + ", expected " + this.rowCount * this.columnCount);
        }
        for (int i = 0; i < values.length; ++i) {
            this.buffer.putDouble(i * 8, values[i]);
        }
    }

    private double[] getValuesCopy() {
        double[] values = new double[this.rowCount * this.columnCount];
        this.buffer.asDoubleBuffer().get(values);
        return values;
    }

    Jama.Matrix toJamaMatrix() {
        return new Jama.Matrix(this.getValuesCopy(), this.rowCount);
    }

    @Override
    public LUDecomposition decomposeLU() {
        return new DenseLUDecomposition(this);
    }

    @Override
    public Matrix times(Matrix other, double scalar) {
        return this.times(other.toDense(), scalar);
    }

    public DenseMatrix times(DenseMatrix other, double scalar) {
        Objects.requireNonNull(other);
        if (other.rowCount != this.columnCount) {
            throw new MatrixException("Invalid matrices inner dimension");
        }
        DenseMatrix result = new DenseMatrix(this.rowCount, other.columnCount);
        double[] otherColumnJ = new double[this.columnCount];
        for (int j = 0; j < other.columnCount; ++j) {
            for (int k = 0; k < this.columnCount; ++k) {
                otherColumnJ[k] = other.getUnsafe(k, j);
            }
            for (int i = 0; i < this.rowCount; ++i) {
                double s = 0.0;
                for (int k = 0; k < this.columnCount; ++k) {
                    s += this.getUnsafe(i, k) * otherColumnJ[k];
                }
                result.setUnsafe(i, j, s * scalar);
            }
        }
        return result;
    }

    public DenseMatrix times(DenseMatrix other) {
        return this.times(other, 1.0);
    }

    public DenseMatrix add(DenseMatrix other, double alpha, double beta) {
        Objects.requireNonNull(other);
        if (other.rowCount != this.rowCount || other.columnCount != this.columnCount) {
            throw new MatrixException("Incompatible matrices dimensions");
        }
        DenseMatrix result = new DenseMatrix(this.rowCount, this.columnCount);
        for (int i = 0; i < this.rowCount; ++i) {
            for (int j = 0; j < this.columnCount; ++j) {
                result.setUnsafe(i, j, alpha * this.getUnsafe(i, j) + beta * other.getUnsafe(i, j));
            }
        }
        return result;
    }

    @Override
    public Matrix add(Matrix other, double alpha, double beta) {
        return this.add(other.toDense(), alpha, beta);
    }

    @Override
    public void iterateNonZeroValue(Matrix.ElementHandler handler) {
        Objects.requireNonNull(handler);
        for (int j = 0; j < this.getColumnCount(); ++j) {
            this.iterateNonZeroValueOfColumn(j, handler);
        }
    }

    @Override
    public void iterateNonZeroValueOfColumn(int j, Matrix.ElementHandler handler) {
        for (int i = 0; i < this.getRowCount(); ++i) {
            double value = this.get(i, j);
            if (value == 0.0) continue;
            handler.onElement(i, j, value);
        }
    }

    @Override
    public DenseMatrix toDense() {
        return this;
    }

    @Override
    public SparseMatrix toSparse() {
        return (SparseMatrix)this.to(new SparseMatrixFactory());
    }

    @Override
    public Matrix to(MatrixFactory factory) {
        Objects.requireNonNull(factory);
        if (factory instanceof DenseMatrixFactory) {
            return this;
        }
        return this.copy(factory);
    }

    @Override
    protected int getEstimatedNonZeroValueCount() {
        return this.getRowCount() * this.getColumnCount();
    }

    @Override
    public DenseMatrix transpose() {
        int transposedRowCount = this.columnCount;
        int transposedColumnCount = this.rowCount;
        ByteBuffer transposedBuffer = DenseMatrix.createBuffer(transposedRowCount, transposedColumnCount);
        for (int i = 0; i < this.rowCount; ++i) {
            for (int j = 0; j < this.columnCount; ++j) {
                double value = this.buffer.getDouble(j * 8 * this.rowCount + i * 8);
                transposedBuffer.putDouble(i * 8 * transposedRowCount + j * 8, value);
            }
        }
        return new DenseMatrix(transposedRowCount, transposedColumnCount, () -> transposedBuffer);
    }

    @Override
    public void print(PrintStream out) {
        this.print(out, null, null);
    }

    @Override
    public void print(PrintStream out, List<String> rowNames, List<String> columnNames) {
        int rowNamesWidth = this.getMaxWidthAmongRowNames(rowNames);
        int[] width = this.getMaxWidthForEachColumn(columnNames);
        if (columnNames != null) {
            if (rowNames != null) {
                out.print(Strings.repeat((String)" ", (int)(rowNamesWidth + 1)));
            }
            for (int j = 0; j < this.getColumnCount(); ++j) {
                out.print(Strings.padStart((String)columnNames.get(j), (int)(width[j] + 1), (char)' '));
            }
            out.println();
        }
        for (int i = 0; i < this.getRowCount(); ++i) {
            if (rowNames != null) {
                out.print(Strings.padStart((String)rowNames.get(i), (int)(rowNamesWidth + 1), (char)' '));
            }
            for (int j = 0; j < this.getColumnCount(); ++j) {
                out.print(Strings.padStart((String)Double.toString(this.get(i, j)), (int)(width[j] + 1), (char)' '));
            }
            out.println();
        }
    }

    private int getMaxWidthAmongRowNames(List<String> rowNames) {
        int rowNamesWidth = 0;
        if (rowNames != null) {
            for (String rowName : rowNames) {
                rowNamesWidth = Math.max(rowNamesWidth, rowName.length());
            }
        }
        return rowNamesWidth;
    }

    private int[] getMaxWidthForEachColumn(List<String> columnNames) {
        int[] width = new int[this.getColumnCount()];
        for (int i = 0; i < this.getRowCount(); ++i) {
            for (int j = 0; j < this.getColumnCount(); ++j) {
                width[j] = Math.max(width[j], Double.toString(this.get(i, j)).length());
                if (columnNames == null) continue;
                width[j] = Math.max(width[j], columnNames.get(j).length());
            }
        }
        return width;
    }

    public int hashCode() {
        return this.rowCount + this.columnCount + this.buffer.hashCode();
    }

    public boolean equals(Object obj) {
        if (obj instanceof DenseMatrix) {
            DenseMatrix other = (DenseMatrix)obj;
            return this.rowCount == other.rowCount && this.columnCount == other.columnCount && this.buffer.equals(other.buffer);
        }
        return false;
    }

    class DenseElement
    implements Matrix.Element {
        private final int i;
        private final int j;

        DenseElement(int i, int j) {
            this.i = i;
            this.j = j;
        }

        @Override
        public void set(double value) {
            DenseMatrix.this.set(this.i, this.j, value);
        }

        @Override
        public void add(double value) {
            DenseMatrix.this.add(this.i, this.j, value);
        }
    }
}

