/*
 * Decompiled with CFR 0.152.
 */
package com.powsybl.openloadflow.network;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonGenerator;
import com.google.common.base.Stopwatch;
import com.powsybl.commons.report.ReportNode;
import com.powsybl.openloadflow.graph.GraphConnectivity;
import com.powsybl.openloadflow.graph.GraphConnectivityFactory;
import com.powsybl.openloadflow.network.AbstractPropertyBag;
import com.powsybl.openloadflow.network.ElementType;
import com.powsybl.openloadflow.network.GraphVizGraphBuilder;
import com.powsybl.openloadflow.network.LfBranch;
import com.powsybl.openloadflow.network.LfBus;
import com.powsybl.openloadflow.network.LfElement;
import com.powsybl.openloadflow.network.LfGenerator;
import com.powsybl.openloadflow.network.LfHvdc;
import com.powsybl.openloadflow.network.LfLoad;
import com.powsybl.openloadflow.network.LfNetworkListener;
import com.powsybl.openloadflow.network.LfNetworkLoader;
import com.powsybl.openloadflow.network.LfNetworkParameters;
import com.powsybl.openloadflow.network.LfNetworkStateUpdateParameters;
import com.powsybl.openloadflow.network.LfNetworkUpdateReport;
import com.powsybl.openloadflow.network.LfOverloadManagementSystem;
import com.powsybl.openloadflow.network.LfSecondaryVoltageControl;
import com.powsybl.openloadflow.network.LfShunt;
import com.powsybl.openloadflow.network.LfTopoConfig;
import com.powsybl.openloadflow.network.LfZeroImpedanceNetwork;
import com.powsybl.openloadflow.network.LoadFlowModel;
import com.powsybl.openloadflow.network.MostMeshedSlackBusSelector;
import com.powsybl.openloadflow.network.PiModel;
import com.powsybl.openloadflow.network.PropertyBag;
import com.powsybl.openloadflow.network.ReferenceBusSelector;
import com.powsybl.openloadflow.network.SelectedGeneratorReferenceBus;
import com.powsybl.openloadflow.network.SelectedReferenceBus;
import com.powsybl.openloadflow.network.SelectedSlackBus;
import com.powsybl.openloadflow.network.SlackBusSelector;
import com.powsybl.openloadflow.network.TransformerVoltageControl;
import com.powsybl.openloadflow.network.VoltageControl;
import com.powsybl.openloadflow.util.Markers;
import com.powsybl.openloadflow.util.Reports;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.io.Writer;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.anarres.graphviz.builder.GraphVizGraph;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LfNetwork
extends AbstractPropertyBag
implements PropertyBag {
    private static final Logger LOGGER = LoggerFactory.getLogger(LfNetwork.class);
    private static final SlackBusSelector SLACK_BUS_SELECTOR_FALLBACK = new MostMeshedSlackBusSelector();
    private final int numCC;
    private final int numSC;
    private final SlackBusSelector slackBusSelector;
    private final ReferenceBusSelector referenceBusSelector;
    private final int maxSlackBusCount;
    private final Map<String, LfBus> busesById = new LinkedHashMap<String, LfBus>();
    private final List<LfBus> busesByIndex = new ArrayList<LfBus>();
    private LfBus referenceBus;
    private List<LfBus> slackBuses;
    private Set<LfBus> excludedSlackBuses = Collections.emptySet();
    private LfGenerator referenceGenerator;
    private final List<LfBranch> branches = new ArrayList<LfBranch>();
    private final Map<String, LfBranch> branchesById = new HashMap<String, LfBranch>();
    private int shuntCount = 0;
    private final List<LfShunt> shuntsByIndex = new ArrayList<LfShunt>();
    private final Map<String, LfShunt> shuntsById = new HashMap<String, LfShunt>();
    private final Map<String, LfGenerator> generatorsById = new HashMap<String, LfGenerator>();
    private final Map<String, LfLoad> loadsById = new HashMap<String, LfLoad>();
    private final List<LfHvdc> hvdcs = new ArrayList<LfHvdc>();
    private final Map<String, LfHvdc> hvdcsById = new HashMap<String, LfHvdc>();
    private final List<LfNetworkListener> listeners = new ArrayList<LfNetworkListener>();
    private Validity validity = Validity.VALID;
    private final GraphConnectivityFactory<LfBus, LfBranch> connectivityFactory;
    private GraphConnectivity<LfBus, LfBranch> connectivity;
    private final Map<LoadFlowModel, Set<LfZeroImpedanceNetwork>> zeroImpedanceNetworksByModel = new EnumMap<LoadFlowModel, Set<LfZeroImpedanceNetwork>>(LoadFlowModel.class);
    private ReportNode reportNode;
    private final List<LfSecondaryVoltageControl> secondaryVoltageControls = new ArrayList<LfSecondaryVoltageControl>();
    private final List<LfVoltageAngleLimit> voltageAngleLimits = new ArrayList<LfVoltageAngleLimit>();
    protected final List<LfOverloadManagementSystem> overloadManagementSystems = new ArrayList<LfOverloadManagementSystem>();

    public LfNetwork(int numCC, int numSC, SlackBusSelector slackBusSelector, int maxSlackBusCount, GraphConnectivityFactory<LfBus, LfBranch> connectivityFactory, ReferenceBusSelector referenceBusSelector, ReportNode reportNode) {
        this.numCC = numCC;
        this.numSC = numSC;
        this.slackBusSelector = Objects.requireNonNull(slackBusSelector);
        this.maxSlackBusCount = maxSlackBusCount;
        this.connectivityFactory = Objects.requireNonNull(connectivityFactory);
        this.referenceBusSelector = referenceBusSelector;
        this.reportNode = Objects.requireNonNull(reportNode);
    }

    public LfNetwork(int numCC, int numSC, SlackBusSelector slackBusSelector, int maxSlackBusCount, GraphConnectivityFactory<LfBus, LfBranch> connectivityFactory, ReferenceBusSelector referenceBusSelector) {
        this(numCC, numSC, slackBusSelector, maxSlackBusCount, connectivityFactory, referenceBusSelector, ReportNode.NO_OP);
    }

    public int getNumCC() {
        return this.numCC;
    }

    public int getNumSC() {
        return this.numSC;
    }

    public ReportNode getReportNode() {
        return this.reportNode;
    }

    public void setReportNode(ReportNode reportNode) {
        this.reportNode = Objects.requireNonNull(reportNode);
    }

    public LfElement getElement(ElementType elementType, int num) {
        return switch (elementType) {
            default -> throw new IncompatibleClassChangeError();
            case ElementType.BUS -> this.getBus(num);
            case ElementType.BRANCH -> this.getBranch(num);
            case ElementType.SHUNT_COMPENSATOR -> this.getShunt(num);
            case ElementType.HVDC -> this.getHvdc(num);
        };
    }

    private void invalidateSlackAndReference() {
        if (this.slackBuses != null) {
            for (LfBus slackBus : this.slackBuses) {
                slackBus.setSlack(false);
            }
        }
        this.slackBuses = null;
        if (this.referenceBus != null) {
            this.referenceBus.setReference(false);
        }
        this.referenceBus = null;
        if (this.referenceGenerator != null) {
            this.referenceGenerator.setReference(false);
        }
        this.referenceGenerator = null;
    }

    public void updateSlackBusesAndReferenceBus() {
        if (this.slackBuses == null && this.referenceBus == null) {
            SelectedSlackBus selectedSlackBus = this.slackBusSelector.select(this.busesByIndex, this.maxSlackBusCount);
            this.slackBuses = selectedSlackBus.getBuses().stream().filter(bus -> !this.excludedSlackBuses.contains(bus)).toList();
            if (this.slackBuses.isEmpty()) {
                selectedSlackBus = SLACK_BUS_SELECTOR_FALLBACK.select(this.busesByIndex, this.excludedSlackBuses.size() + this.maxSlackBusCount);
                this.slackBuses = selectedSlackBus.getBuses().stream().filter(bus -> !this.excludedSlackBuses.contains(bus)).limit(this.maxSlackBusCount).toList();
            }
            LOGGER.info("Network {}, slack buses are {} (method='{}')", new Object[]{this, this.slackBuses, selectedSlackBus.getSelectionMethod()});
            for (LfBus slackBus : this.slackBuses) {
                slackBus.setSlack(true);
            }
            SelectedReferenceBus selectedReferenceBus = this.referenceBusSelector.select(this);
            this.referenceBus = selectedReferenceBus.getLfBus();
            LOGGER.info("Network {}, reference bus is {} (method='{}')", new Object[]{this, this.referenceBus, selectedReferenceBus.getSelectionMethod()});
            this.referenceBus.setReference(true);
            if (selectedReferenceBus instanceof SelectedGeneratorReferenceBus) {
                SelectedGeneratorReferenceBus generatorReferenceBus = (SelectedGeneratorReferenceBus)selectedReferenceBus;
                this.referenceGenerator = generatorReferenceBus.getLfGenerator();
                LOGGER.info("Network {}, reference generator is {}", (Object)this, (Object)this.referenceGenerator.getId());
                this.referenceGenerator.setReference(true);
            }
            if (this.connectivity != null) {
                this.connectivity.setMainComponentVertex(this.slackBuses.get(0));
            }
        }
    }

    private void invalidateZeroImpedanceNetworks() {
        this.zeroImpedanceNetworksByModel.clear();
    }

    public void addBranch(LfBranch branch) {
        Objects.requireNonNull(branch);
        branch.setNum(this.branches.size());
        this.branches.add(branch);
        this.branchesById.put(branch.getId(), branch);
        this.invalidateSlackAndReference();
        this.connectivity = null;
        this.invalidateZeroImpedanceNetworks();
        if (branch.getBus1() != null) {
            branch.getBus1().addBranch(branch);
        }
        if (branch.getBus2() != null) {
            branch.getBus2().addBranch(branch);
        }
    }

    public List<LfBranch> getBranches() {
        return this.branches;
    }

    public LfBranch getBranch(int num) {
        return this.branches.get(num);
    }

    public LfBranch getBranchById(String branchId) {
        Objects.requireNonNull(branchId);
        return this.branchesById.get(branchId);
    }

    private void addShunt(LfShunt shunt) {
        shunt.setNum(this.shuntCount++);
        this.shuntsByIndex.add(shunt);
        shunt.getOriginalIds().forEach(id -> this.shuntsById.put((String)id, shunt));
    }

    public void addBus(LfBus bus) {
        Objects.requireNonNull(bus);
        bus.setNum(this.busesByIndex.size());
        this.busesByIndex.add(bus);
        this.busesById.put(bus.getId(), bus);
        this.invalidateSlackAndReference();
        this.connectivity = null;
        bus.getShunt().ifPresent(this::addShunt);
        bus.getControllerShunt().ifPresent(this::addShunt);
        bus.getSvcShunt().ifPresent(this::addShunt);
        bus.getGenerators().forEach(gen -> this.generatorsById.put(gen.getId(), (LfGenerator)gen));
        bus.getLoads().forEach(load -> load.getOriginalIds().forEach(id -> this.loadsById.put((String)id, (LfLoad)load)));
    }

    public List<LfBus> getBuses() {
        return this.busesByIndex;
    }

    public LfBus getBusById(String id) {
        Objects.requireNonNull(id);
        return this.busesById.get(id);
    }

    public LfBus getBus(int num) {
        return this.busesByIndex.get(num);
    }

    public LfBus getReferenceBus() {
        this.updateSlackBusesAndReferenceBus();
        return this.referenceBus;
    }

    public LfBus getSlackBus() {
        return this.getSlackBuses().get(0);
    }

    public List<LfBus> getSlackBuses() {
        this.updateSlackBusesAndReferenceBus();
        return this.slackBuses;
    }

    public Set<LfBus> getExcludedSlackBuses() {
        return this.excludedSlackBuses;
    }

    public void setExcludedSlackBuses(Set<LfBus> excludedSlackBuses) {
        Objects.requireNonNull(excludedSlackBuses);
        if (!excludedSlackBuses.equals(this.excludedSlackBuses)) {
            this.excludedSlackBuses = excludedSlackBuses;
            this.invalidateSlackAndReference();
        }
    }

    public LfGenerator getReferenceGenerator() {
        this.updateSlackBusesAndReferenceBus();
        return this.referenceGenerator;
    }

    public List<LfShunt> getShunts() {
        return this.shuntsByIndex;
    }

    public LfShunt getShunt(int num) {
        return this.shuntsByIndex.get(num);
    }

    public LfShunt getShuntById(String id) {
        Objects.requireNonNull(id);
        return this.shuntsById.get(id);
    }

    public LfGenerator getGeneratorById(String id) {
        Objects.requireNonNull(id);
        return this.generatorsById.get(id);
    }

    public LfLoad getLoadById(String id) {
        Objects.requireNonNull(id);
        return this.loadsById.get(id);
    }

    public void addHvdc(LfHvdc hvdc) {
        Objects.requireNonNull(hvdc);
        hvdc.setNum(this.hvdcs.size());
        this.hvdcs.add(hvdc);
        this.hvdcsById.put(hvdc.getId(), hvdc);
        if (hvdc.getBus1() != null) {
            hvdc.getBus1().addHvdc(hvdc);
        }
        if (hvdc.getBus2() != null) {
            hvdc.getBus2().addHvdc(hvdc);
        }
    }

    public List<LfHvdc> getHvdcs() {
        return this.hvdcs;
    }

    public LfHvdc getHvdc(int num) {
        return this.hvdcs.get(num);
    }

    public LfHvdc getHvdcById(String id) {
        Objects.requireNonNull(id);
        return this.hvdcsById.get(id);
    }

    public void updateState(LfNetworkStateUpdateParameters parameters) {
        Stopwatch stopwatch = Stopwatch.createStarted();
        LfNetworkUpdateReport updateReport = new LfNetworkUpdateReport();
        for (LfBus bus : this.busesById.values()) {
            bus.updateState(parameters);
            for (LfGenerator generator : bus.getGenerators()) {
                generator.updateState(parameters);
            }
            bus.getShunt().ifPresent(shunt -> shunt.updateState(parameters));
            bus.getControllerShunt().ifPresent(shunt -> shunt.updateState(parameters));
        }
        this.branches.forEach(branch -> branch.updateState(parameters, updateReport));
        this.hvdcs.forEach(LfHvdc::updateState);
        if (updateReport.closedSwitchCount + updateReport.openedSwitchCount > 0) {
            LOGGER.debug("Switches status update: {} closed and {} opened", (Object)updateReport.closedSwitchCount, (Object)updateReport.openedSwitchCount);
        }
        if (updateReport.connectedBranchSide1Count + updateReport.disconnectedBranchSide1Count + updateReport.connectedBranchSide2Count + updateReport.disconnectedBranchSide2Count > 0) {
            LOGGER.debug("Branches connection status update: {} connected side 1, {} disconnected side1, {} connected side 2, {} disconnected side 2", new Object[]{updateReport.connectedBranchSide1Count, updateReport.disconnectedBranchSide1Count, updateReport.connectedBranchSide2Count, updateReport.disconnectedBranchSide2Count});
        }
        stopwatch.stop();
        LOGGER.debug(Markers.PERFORMANCE_MARKER, "Network {}, IIDM network updated in {} ms", (Object)this, (Object)stopwatch.elapsed(TimeUnit.MILLISECONDS));
    }

    public void writeJson(Path file) {
        try (BufferedWriter writer = Files.newBufferedWriter(file, StandardCharsets.UTF_8, new OpenOption[0]);){
            this.writeJson(writer);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private void writeJson(LfBus bus, JsonGenerator jsonGenerator) throws IOException {
        jsonGenerator.writeStringField("id", bus.getId());
        jsonGenerator.writeNumberField("num", bus.getNum());
        if (bus.getGenerationTargetQ() != 0.0) {
            jsonGenerator.writeNumberField("generationTargetQ", bus.getGenerationTargetQ());
        }
        if (bus.getLoadTargetP() != 0.0) {
            jsonGenerator.writeNumberField("loadTargetP", bus.getLoadTargetP());
        }
        if (bus.getLoadTargetQ() != 0.0) {
            jsonGenerator.writeNumberField("loadTargetQ", bus.getLoadTargetQ());
        }
        bus.getGeneratorVoltageControl().ifPresent(vc -> {
            if (bus.isGeneratorVoltageControlEnabled()) {
                try {
                    if (vc.getControlledBus() != bus) {
                        jsonGenerator.writeNumberField("remoteControlTargetBus", vc.getControlledBus().getNum());
                    }
                    jsonGenerator.writeNumberField("targetV", vc.getTargetValue());
                }
                catch (IOException e) {
                    throw new UncheckedIOException(e);
                }
            }
        });
        if (!Double.isNaN(bus.getV())) {
            jsonGenerator.writeNumberField("v", bus.getV());
        }
        if (!Double.isNaN(bus.getAngle())) {
            jsonGenerator.writeNumberField("angle", bus.getAngle());
        }
    }

    private void writeJson(LfBranch branch, JsonGenerator jsonGenerator) throws IOException {
        jsonGenerator.writeStringField("id", branch.getId());
        jsonGenerator.writeNumberField("num", branch.getNum());
        LfBus bus1 = branch.getBus1();
        LfBus bus2 = branch.getBus2();
        if (bus1 != null) {
            jsonGenerator.writeNumberField("num1", bus1.getNum());
        }
        if (bus2 != null) {
            jsonGenerator.writeNumberField("num2", bus2.getNum());
        }
        PiModel piModel = branch.getPiModel();
        jsonGenerator.writeNumberField("r", piModel.getR());
        jsonGenerator.writeNumberField("x", piModel.getX());
        if (piModel.getG1() != 0.0) {
            jsonGenerator.writeNumberField("g1", piModel.getG1());
        }
        if (piModel.getG2() != 0.0) {
            jsonGenerator.writeNumberField("g2", piModel.getG2());
        }
        if (piModel.getB1() != 0.0) {
            jsonGenerator.writeNumberField("b1", piModel.getB1());
        }
        if (piModel.getB2() != 0.0) {
            jsonGenerator.writeNumberField("b2", piModel.getB2());
        }
        if (piModel.getR1() != 1.0) {
            jsonGenerator.writeNumberField("r1", piModel.getR1());
        }
        if (piModel.getA1() != 0.0) {
            jsonGenerator.writeNumberField("a1", piModel.getA1());
        }
        branch.getPhaseControl().filter(dpc -> branch.isPhaseController()).ifPresent(dpc -> {
            try {
                jsonGenerator.writeFieldName("discretePhaseControl");
                jsonGenerator.writeStartObject();
                jsonGenerator.writeStringField("controller", dpc.getControllerBranch().getId());
                jsonGenerator.writeStringField("controlled", dpc.getControlledBranch().getId());
                jsonGenerator.writeStringField("mode", dpc.getMode().name());
                jsonGenerator.writeStringField("unit", dpc.getUnit().name());
                jsonGenerator.writeStringField("controlledSide", dpc.getControlledSide().name());
                jsonGenerator.writeNumberField("targetValue", dpc.getTargetValue());
                jsonGenerator.writeEndObject();
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        });
    }

    private void writeJson(LfShunt shunt, JsonGenerator jsonGenerator) throws IOException {
        jsonGenerator.writeStringField("id", shunt.getId());
        jsonGenerator.writeNumberField("num", shunt.getNum());
        jsonGenerator.writeNumberField("b", shunt.getB());
    }

    private void writeJson(LfGenerator generator, JsonGenerator jsonGenerator) throws IOException {
        jsonGenerator.writeStringField("id", generator.getId());
        jsonGenerator.writeNumberField("targetP", generator.getTargetP());
        if (!Double.isNaN(generator.getTargetQ())) {
            jsonGenerator.writeNumberField("targetQ", generator.getTargetQ());
        }
        jsonGenerator.writeBooleanField("voltageControl", generator.getGeneratorControlType() == LfGenerator.GeneratorControlType.VOLTAGE);
        jsonGenerator.writeNumberField("minP", generator.getMinP());
        jsonGenerator.writeNumberField("maxP", generator.getMaxP());
        jsonGenerator.writeNumberField("minTargetP", generator.getMinTargetP());
        jsonGenerator.writeNumberField("maxTargetP", generator.getMaxTargetP());
    }

    public void writeJson(Writer writer) {
        Objects.requireNonNull(writer);
        this.updateSlackBusesAndReferenceBus();
        try (JsonGenerator jsonGenerator = new JsonFactory().createGenerator(writer).useDefaultPrettyPrinter();){
            jsonGenerator.writeStartObject();
            jsonGenerator.writeFieldName("buses");
            jsonGenerator.writeStartArray();
            List sortedBuses = this.busesById.values().stream().sorted(Comparator.comparing(LfElement::getId)).collect(Collectors.toList());
            for (LfBus bus : sortedBuses) {
                jsonGenerator.writeStartObject();
                this.writeJson(bus, jsonGenerator);
                bus.getShunt().ifPresent(shunt -> {
                    try {
                        jsonGenerator.writeFieldName("shunt");
                        jsonGenerator.writeStartObject();
                        this.writeJson((LfShunt)shunt, jsonGenerator);
                        jsonGenerator.writeEndObject();
                    }
                    catch (IOException e) {
                        throw new UncheckedIOException(e);
                    }
                });
                List sortedGenerators = bus.getGenerators().stream().sorted(Comparator.comparing(LfGenerator::getId)).collect(Collectors.toList());
                if (!sortedGenerators.isEmpty()) {
                    jsonGenerator.writeFieldName("generators");
                    jsonGenerator.writeStartArray();
                    for (LfGenerator generator : sortedGenerators) {
                        jsonGenerator.writeStartObject();
                        this.writeJson(generator, jsonGenerator);
                        jsonGenerator.writeEndObject();
                    }
                    jsonGenerator.writeEndArray();
                }
                jsonGenerator.writeEndObject();
            }
            jsonGenerator.writeEndArray();
            jsonGenerator.writeFieldName("branches");
            jsonGenerator.writeStartArray();
            List sortedBranches = this.branches.stream().sorted(Comparator.comparing(LfElement::getId)).collect(Collectors.toList());
            for (LfBranch branch : sortedBranches) {
                jsonGenerator.writeStartObject();
                this.writeJson(branch, jsonGenerator);
                jsonGenerator.writeEndObject();
            }
            jsonGenerator.writeEndArray();
            jsonGenerator.writeEndObject();
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private void reportSize(ReportNode reportNode) {
        Reports.reportNetworkSize(reportNode, this.busesById.values().size(), this.branches.size());
        LOGGER.info("Network {} has {} buses and {} branches", new Object[]{this, this.busesById.values().size(), this.branches.size()});
    }

    public void reportBalance(ReportNode reportNode) {
        double activeGeneration = 0.0;
        double reactiveGeneration = 0.0;
        double activeLoad = 0.0;
        double reactiveLoad = 0.0;
        for (LfBus b : this.busesById.values()) {
            activeGeneration += b.getGenerationTargetP() * 100.0;
            reactiveGeneration += b.getGenerationTargetQ() * 100.0;
            activeLoad += b.getLoadTargetP() * 100.0;
            reactiveLoad += b.getLoadTargetQ() * 100.0;
        }
        Reports.reportNetworkBalance(reportNode, activeGeneration, activeLoad, reactiveGeneration, reactiveLoad);
        LOGGER.info("Network {} balance: active generation={} MW, active load={} MW, reactive generation={} MVar, reactive load={} MVar", new Object[]{this, activeGeneration, activeLoad, reactiveGeneration, reactiveLoad});
    }

    public void fix(boolean minImpedance, double lowImpedanceThreshold) {
        if (minImpedance) {
            for (LfBranch branch2 : this.branches) {
                branch2.setMinZ(lowImpedanceThreshold);
            }
        } else {
            this.branches.stream().filter(b -> b.isPhaseController() || b.isPhaseControlled() || b.isTransformerReactivePowerController() || b.isTransformerReactivePowerControlled() || b.getGeneratorReactivePowerControl().isPresent()).forEach(branch -> branch.setMinZ(lowImpedanceThreshold));
        }
    }

    private void validateBuses(LoadFlowModel loadFlowModel, ReportNode reportNode) {
        boolean hasAtLeastOneBusGenerator = false;
        for (LfBus bus : this.busesByIndex) {
            if (bus.getGenerators().isEmpty()) continue;
            hasAtLeastOneBusGenerator = true;
            break;
        }
        if (!hasAtLeastOneBusGenerator) {
            LOGGER.debug("Network {} has no generator and will be considered dead", (Object)this);
            this.validity = Validity.INVALID_NO_GENERATOR;
            return;
        }
        if (loadFlowModel == LoadFlowModel.AC) {
            boolean hasAtLeastOneBusGeneratorVoltageControlEnabled = false;
            for (LfBus bus : this.busesByIndex) {
                if (!bus.isGeneratorVoltageControlEnabled()) continue;
                hasAtLeastOneBusGeneratorVoltageControlEnabled = true;
                break;
            }
            if (!hasAtLeastOneBusGeneratorVoltageControlEnabled) {
                LOGGER.error("Network {} must have at least one bus with generator voltage control enabled", (Object)this);
                if (reportNode != null) {
                    Reports.reportNetworkMustHaveAtLeastOneBusGeneratorVoltageControlEnabled(reportNode);
                }
                this.validity = Validity.INVALID_NO_GENERATOR_VOLTAGE_CONTROL;
            }
        }
    }

    public void validate(LoadFlowModel loadFlowModel, ReportNode reportNode) {
        this.validity = Validity.VALID;
        this.validateBuses(loadFlowModel, reportNode);
    }

    public static <T> List<LfNetwork> load(T network, LfNetworkLoader<T> networkLoader, SlackBusSelector slackBusSelector) {
        return LfNetwork.load(network, networkLoader, new LfNetworkParameters().setSlackBusSelector(slackBusSelector), ReportNode.NO_OP);
    }

    public static <T> List<LfNetwork> load(T network, LfNetworkLoader<T> networkLoader, LfNetworkParameters parameters) {
        return LfNetwork.load(network, networkLoader, parameters, ReportNode.NO_OP);
    }

    public static <T> List<LfNetwork> load(T network, LfNetworkLoader<T> networkLoader, LfNetworkParameters parameters, ReportNode reportNode) {
        return LfNetwork.load(network, networkLoader, new LfTopoConfig(), parameters, reportNode);
    }

    public static <T> List<LfNetwork> load(T network, LfNetworkLoader<T> networkLoader, LfTopoConfig topoConfig, LfNetworkParameters parameters, ReportNode reportNode) {
        Objects.requireNonNull(network);
        Objects.requireNonNull(networkLoader);
        Objects.requireNonNull(parameters);
        List<LfNetwork> lfNetworks = networkLoader.load(network, topoConfig, parameters, reportNode);
        int deadComponentsCount = 0;
        for (LfNetwork lfNetwork : lfNetworks) {
            ReportNode networkReport = Reports.createNetworkInfoReporter(lfNetwork.getReportNode());
            lfNetwork.fix(parameters.isMinImpedance(), parameters.getLowImpedanceThreshold());
            lfNetwork.validate(parameters.getLoadFlowModel(), networkReport);
            switch (lfNetwork.getValidity()) {
                case VALID: {
                    lfNetwork.reportSize(networkReport);
                    lfNetwork.reportBalance(networkReport);
                    Reports.reportAngleReferenceBusAndSlackBuses(networkReport, lfNetwork.getReferenceBus().getId(), lfNetwork.getSlackBuses().stream().map(LfElement::getId).toList());
                    lfNetwork.setReportNode(Reports.includeLfNetworkReportNode(reportNode, lfNetwork.getReportNode()));
                    break;
                }
                case INVALID_NO_GENERATOR_VOLTAGE_CONTROL: {
                    LOGGER.info("Network {} is invalid, no calculation will be done", (Object)lfNetwork);
                    lfNetwork.setReportNode(Reports.includeLfNetworkReportNode(reportNode, lfNetwork.getReportNode()));
                    break;
                }
                case INVALID_NO_GENERATOR: {
                    ++deadComponentsCount;
                }
            }
        }
        if (deadComponentsCount > 0) {
            Reports.reportComponentsWithoutGenerators(reportNode, deadComponentsCount);
            LOGGER.info("No calculation will be done on {} network(s) that have no generators", (Object)deadComponentsCount);
        }
        return lfNetworks;
    }

    public void updateZeroImpedanceCache(LoadFlowModel loadFlowModel) {
        this.zeroImpedanceNetworksByModel.computeIfAbsent(loadFlowModel, m -> LfZeroImpedanceNetwork.create(this, loadFlowModel));
    }

    public Set<LfZeroImpedanceNetwork> getZeroImpedanceNetworks(LoadFlowModel loadFlowModel) {
        this.updateZeroImpedanceCache(loadFlowModel);
        return this.zeroImpedanceNetworksByModel.get((Object)loadFlowModel);
    }

    public GraphConnectivity<LfBus, LfBranch> getConnectivity() {
        if (this.connectivity == null) {
            this.connectivity = Objects.requireNonNull(this.connectivityFactory.create());
            this.getBuses().forEach(this.connectivity::addVertex);
            this.getBranches().stream().filter(b -> b.getBus1() != null && b.getBus2() != null).forEach(b -> this.connectivity.addEdge(b.getBus1(), b.getBus2(), (LfBranch)b));
            this.connectivity.setMainComponentVertex(this.getSlackBuses().get(0));
            if (this.connectivity.supportTemporaryChangesNesting()) {
                this.connectivity.startTemporaryChanges();
            }
        }
        return this.connectivity;
    }

    public void addListener(LfNetworkListener listener) {
        this.listeners.add(listener);
    }

    public void removeListener(LfNetworkListener listener) {
        this.listeners.remove(listener);
    }

    public List<LfNetworkListener> getListeners() {
        return this.listeners;
    }

    public Validity getValidity() {
        return this.validity;
    }

    public void fixTransformerVoltageControls() {
        ArrayList<LfBranch> controllerBranches = new ArrayList<LfBranch>(1);
        this.getConnectivity().startTemporaryChanges();
        for (LfBranch branch : this.branches) {
            if (!branch.isDisabled() && branch.isVoltageController() && branch.isVoltageControlEnabled()) {
                controllerBranches.add(branch);
            }
            if (!branch.isDisabled() || branch.getBus1() == null || branch.getBus2() == null) continue;
            this.getConnectivity().removeEdge(branch);
        }
        for (LfBranch branch : controllerBranches) {
            this.getConnectivity().removeEdge(branch);
        }
        int disabledTransformerCount = 0;
        HashMap<Integer, Boolean> componentNoPVBusesMap = new HashMap<Integer, Boolean>();
        for (LfBranch branch : controllerBranches) {
            boolean noPvBusesInComponent;
            LfBus notControlledSide;
            TransformerVoltageControl voltageControl = branch.getVoltageControl().orElseThrow();
            if (voltageControl.getControlledBus() == branch.getBus1()) {
                notControlledSide = branch.getBus2();
            } else {
                if (voltageControl.getControlledBus() != branch.getBus2()) continue;
                notControlledSide = branch.getBus1();
            }
            if (!(noPvBusesInComponent = componentNoPVBusesMap.computeIfAbsent(this.getConnectivity().getComponentNumber(notControlledSide), k -> this.getConnectivity().getConnectedComponent(notControlledSide).stream().noneMatch(bus -> bus.isGeneratorVoltageControlled() && bus.isGeneratorVoltageControlEnabled())).booleanValue())) continue;
            branch.setVoltageControlEnabled(false);
            LOGGER.trace("Transformer {} voltage control has been disabled because no PV buses on not controlled side connected component", (Object)branch.getId());
            ++disabledTransformerCount;
        }
        this.getConnectivity().undoTemporaryChanges();
        if (disabledTransformerCount > 0) {
            LOGGER.warn("{} transformer voltage controls have been disabled because no PV buses on not controlled side connected component", (Object)disabledTransformerCount);
        }
    }

    public void writeGraphViz(Path file, LoadFlowModel loadFlowModel) {
        try (BufferedWriter writer = Files.newBufferedWriter(file, StandardCharsets.UTF_8, new OpenOption[0]);){
            this.writeGraphViz(writer, loadFlowModel);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    public void writeGraphViz(Writer writer, LoadFlowModel loadFlowModel) {
        try {
            GraphVizGraph gvGraph = new GraphVizGraphBuilder(this).build(loadFlowModel);
            gvGraph.writeTo(writer);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    public String getId() {
        return "{CC" + this.numCC + " SC" + this.numSC + "}";
    }

    public void addSecondaryVoltageControl(LfSecondaryVoltageControl secondaryVoltageControl) {
        this.secondaryVoltageControls.add(Objects.requireNonNull(secondaryVoltageControl));
    }

    public List<LfSecondaryVoltageControl> getSecondaryVoltageControls() {
        return this.secondaryVoltageControls;
    }

    public Optional<LfSecondaryVoltageControl> getSecondaryVoltageControl(String controlZoneName) {
        Objects.requireNonNull(controlZoneName);
        return this.secondaryVoltageControls.stream().filter(lfSvc -> lfSvc.getZoneName().equals(controlZoneName)).findFirst();
    }

    private static boolean filterSecondaryVoltageControl(LfSecondaryVoltageControl secondaryVoltageControl) {
        return !secondaryVoltageControl.getPilotBus().isDisabled();
    }

    public List<LfSecondaryVoltageControl> getEnabledSecondaryVoltageControls() {
        return this.secondaryVoltageControls.stream().filter(LfNetwork::filterSecondaryVoltageControl).toList();
    }

    public void addVoltageAngleLimit(LfVoltageAngleLimit limit) {
        this.voltageAngleLimits.add(Objects.requireNonNull(limit));
    }

    public List<LfVoltageAngleLimit> getVoltageAngleLimits() {
        return this.voltageAngleLimits;
    }

    public <E extends LfElement> List<E> getControllerElements(VoltageControl.Type type) {
        return this.busesByIndex.stream().filter(bus -> bus.isVoltageControlled(type)).filter(bus -> bus.getVoltageControl(type).orElseThrow().getMergeStatus() == VoltageControl.MergeStatus.MAIN).filter(bus -> bus.getVoltageControl(type).orElseThrow().isVisible()).flatMap(bus -> bus.getVoltageControl(type).orElseThrow().getMergedControllerElements().stream()).filter(Predicate.not(LfElement::isDisabled)).map(element -> element).collect(Collectors.toList());
    }

    public List<LfBus> getControlledBuses(VoltageControl.Type type) {
        return this.busesByIndex.stream().filter(bus -> bus.isVoltageControlled(type)).filter(bus -> bus.getVoltageControl(type).orElseThrow().getMergeStatus() == VoltageControl.MergeStatus.MAIN).filter(bus -> bus.getVoltageControl(type).orElseThrow().isVisible()).collect(Collectors.toList());
    }

    public void addOverloadManagementSystem(LfOverloadManagementSystem overloadManagementSystem) {
        this.overloadManagementSystems.add(Objects.requireNonNull(overloadManagementSystem));
    }

    public List<LfOverloadManagementSystem> getOverloadManagementSystems() {
        return this.overloadManagementSystems;
    }

    public String toString() {
        return this.getId();
    }

    public static enum Validity {
        VALID("Valid"),
        INVALID_NO_GENERATOR("Network has no generator"),
        INVALID_NO_GENERATOR_VOLTAGE_CONTROL("Network has no generator with voltage control enabled");

        private final String description;

        private Validity(String description) {
            this.description = description;
        }

        public String toString() {
            return this.description;
        }
    }

    public static class LfVoltageAngleLimit {
        private final String id;
        private final LfBus from;
        private final LfBus to;
        private final double highValue;
        private final double lowValue;

        public LfVoltageAngleLimit(String id, LfBus from, LfBus to, double highValue, double lowValue) {
            this.id = Objects.requireNonNull(id);
            this.from = Objects.requireNonNull(from);
            this.to = Objects.requireNonNull(to);
            this.highValue = highValue;
            this.lowValue = lowValue;
        }

        public String getId() {
            return this.id;
        }

        public LfBus getFrom() {
            return this.from;
        }

        public LfBus getTo() {
            return this.to;
        }

        public double getHighValue() {
            return this.highValue;
        }

        public double getLowValue() {
            return this.lowValue;
        }
    }
}

