/*
 * Decompiled with CFR 0.152.
 */
package com.powsybl.timeseries;

import com.google.common.base.Stopwatch;
import com.powsybl.timeseries.BiList;
import com.powsybl.timeseries.BigDoubleBuffer;
import com.powsybl.timeseries.BigStringBuffer;
import com.powsybl.timeseries.DoubleTimeSeries;
import com.powsybl.timeseries.StringTimeSeries;
import com.powsybl.timeseries.TimeSeries;
import com.powsybl.timeseries.TimeSeriesCsvConfig;
import com.powsybl.timeseries.TimeSeriesDataType;
import com.powsybl.timeseries.TimeSeriesException;
import com.powsybl.timeseries.TimeSeriesIndex;
import com.powsybl.timeseries.TimeSeriesMetadata;
import com.powsybl.timeseries.TimeSeriesVersions;
import gnu.trove.list.array.TIntArrayList;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.StringWriter;
import java.io.UncheckedIOException;
import java.io.Writer;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.time.Instant;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.IntFunction;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.zip.GZIPOutputStream;
import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TimeSeriesTable {
    private static final Logger LOGGER = LoggerFactory.getLogger(TimeSeriesTable.class);
    private final int fromVersion;
    private final int toVersion;
    private List<TimeSeriesMetadata> timeSeriesMetadata;
    private final TimeSeriesIndex tableIndex;
    private final IntFunction<ByteBuffer> byteBufferAllocator;
    private final TIntArrayList timeSeriesIndexDoubleOrString = new TIntArrayList();
    private final TimeSeriesNameMap doubleTimeSeriesNames = new TimeSeriesNameMap();
    private final TimeSeriesNameMap stringTimeSeriesNames = new TimeSeriesNameMap();
    private BigDoubleBuffer doubleBuffer;
    private BigStringBuffer stringBuffer;
    private final Lock initLock = new ReentrantLock();
    private double[] means;
    private double[] stdDevs;
    private final Lock statsLock = new ReentrantLock();

    public TimeSeriesTable(int fromVersion, int toVersion, TimeSeriesIndex tableIndex) {
        this(fromVersion, toVersion, tableIndex, ByteBuffer::allocateDirect);
    }

    public TimeSeriesTable(int fromVersion, int toVersion, TimeSeriesIndex tableIndex, IntFunction<ByteBuffer> byteBufferAllocator) {
        TimeSeriesVersions.check(fromVersion);
        TimeSeriesVersions.check(toVersion);
        if (toVersion < fromVersion) {
            throw new TimeSeriesException("toVersion (" + toVersion + ") is expected to be greater than fromVersion (" + fromVersion + ")");
        }
        this.fromVersion = fromVersion;
        this.toVersion = toVersion;
        this.tableIndex = Objects.requireNonNull(tableIndex);
        this.byteBufferAllocator = Objects.requireNonNull(byteBufferAllocator);
    }

    public static TimeSeriesTable createDirectMem(int fromVersion, int toVersion, TimeSeriesIndex tableIndex) {
        return new TimeSeriesTable(fromVersion, toVersion, tableIndex);
    }

    public static TimeSeriesTable createMem(int fromVersion, int toVersion, TimeSeriesIndex tableIndex) {
        return new TimeSeriesTable(fromVersion, toVersion, tableIndex, ByteBuffer::allocate);
    }

    private void initTable(List<DoubleTimeSeries> doubleTimeSeries, List<StringTimeSeries> stringTimeSeries) {
        this.initLock.lock();
        try {
            int i;
            if (this.timeSeriesMetadata != null) {
                return;
            }
            this.timeSeriesMetadata = new ArrayList<TimeSeriesMetadata>(doubleTimeSeries.size() + stringTimeSeries.size());
            for (DoubleTimeSeries doubleTimeSeries2 : doubleTimeSeries.stream().sorted(Comparator.comparing(ts -> ts.getMetadata().getName())).toList()) {
                this.timeSeriesMetadata.add(doubleTimeSeries2.getMetadata());
                i = this.doubleTimeSeriesNames.add(doubleTimeSeries2.getMetadata().getName());
                this.timeSeriesIndexDoubleOrString.add(i);
            }
            for (StringTimeSeries stringTimeSeries2 : stringTimeSeries.stream().sorted(Comparator.comparing(ts -> ts.getMetadata().getName())).toList()) {
                this.timeSeriesMetadata.add(stringTimeSeries2.getMetadata());
                i = this.stringTimeSeriesNames.add(stringTimeSeries2.getMetadata().getName());
                this.timeSeriesIndexDoubleOrString.add(i);
            }
            if (this.tableIndex == null) {
                throw new TimeSeriesException("None of the time series have a finite index");
            }
            int versionCount = this.toVersion - this.fromVersion + 1;
            long l = (long)versionCount * (long)this.doubleTimeSeriesNames.size() * (long)this.tableIndex.getPointCount();
            this.doubleBuffer = TimeSeriesTable.createDoubleBuffer(this.byteBufferAllocator, l, Double.NaN);
            long stringBufferSize = (long)versionCount * (long)this.stringTimeSeriesNames.size() * (long)this.tableIndex.getPointCount();
            this.stringBuffer = new BigStringBuffer(this.byteBufferAllocator, stringBufferSize);
            if (LOGGER.isInfoEnabled()) {
                LOGGER.info("Allocation of {} for time series table", (Object)FileUtils.byteCountToDisplaySize((long)(this.doubleBuffer.capacity() * 8L + this.stringBuffer.capacity() * 4L)));
            }
            this.means = new double[this.doubleTimeSeriesNames.size() * versionCount];
            Arrays.fill(this.means, Double.NaN);
            this.stdDevs = new double[this.doubleTimeSeriesNames.size() * versionCount];
            Arrays.fill(this.stdDevs, Double.NaN);
        }
        catch (Exception e) {
            LOGGER.error(e.toString(), (Throwable)e);
            this.timeSeriesMetadata = null;
            this.doubleTimeSeriesNames.clear();
            this.stringTimeSeriesNames.clear();
            this.timeSeriesIndexDoubleOrString.clear();
            this.doubleBuffer = null;
            this.stringBuffer = null;
            this.means = null;
            this.stdDevs = null;
            throw e;
        }
        finally {
            this.initLock.unlock();
        }
    }

    public TimeSeriesIndex getTableIndex() {
        return this.tableIndex;
    }

    private static BigDoubleBuffer createDoubleBuffer(IntFunction<ByteBuffer> byteBufferAllocator, long size) {
        return new BigDoubleBuffer(byteBufferAllocator, size);
    }

    private static BigDoubleBuffer createDoubleBuffer(IntFunction<ByteBuffer> byteBufferAllocator, long size, double initialValue) {
        BigDoubleBuffer doubleBuffer = TimeSeriesTable.createDoubleBuffer(byteBufferAllocator, size);
        for (long i = 0L; i < size; ++i) {
            doubleBuffer.put(i, initialValue);
        }
        return doubleBuffer;
    }

    private long getTimeSeriesOffset(int version, int timeSeriesNum) {
        return (long)timeSeriesNum * (long)this.tableIndex.getPointCount() * (long)(this.toVersion - this.fromVersion + 1) + (long)(version - this.fromVersion) * (long)this.tableIndex.getPointCount();
    }

    private int getStatisticsIndex(int version, int timeSeriesNum) {
        return (version - this.fromVersion) * this.doubleTimeSeriesNames.size() + timeSeriesNum;
    }

    private void checkVersionIsInRange(int version) {
        if (version < this.fromVersion || version > this.toVersion) {
            throw new IllegalArgumentException("Version is out of range [" + this.fromVersion + ", " + this.toVersion + "]");
        }
    }

    private int checkTimeSeriesNum(int timeSeriesNum) {
        if (timeSeriesNum < 0 || timeSeriesNum >= this.timeSeriesIndexDoubleOrString.size()) {
            throw new IllegalArgumentException("Time series number is out of range [0, " + (this.timeSeriesIndexDoubleOrString.size() - 1) + "]");
        }
        return this.timeSeriesIndexDoubleOrString.get(timeSeriesNum);
    }

    private void checkPoint(int point) {
        if (point < 0 || point >= this.tableIndex.getPointCount()) {
            throw new IllegalArgumentException("Point is out of range [0, " + (this.tableIndex.getPointCount() - 1) + "]");
        }
    }

    private void loadDouble(int version, DoubleTimeSeries timeSeries) {
        int timeSeriesNum = this.doubleTimeSeriesNames.getIndex(timeSeries.getMetadata().getName());
        long timeSeriesOffset = this.getTimeSeriesOffset(version, timeSeriesNum);
        timeSeries.fillBuffer(this.doubleBuffer, timeSeriesOffset);
        this.invalidateStatistics(version, timeSeriesNum);
    }

    private void loadString(int version, StringTimeSeries timeSeries) {
        int timeSeriesNum = this.stringTimeSeriesNames.getIndex(timeSeries.getMetadata().getName());
        long timeSeriesOffset = this.getTimeSeriesOffset(version, timeSeriesNum);
        timeSeries.fillBuffer(this.stringBuffer, timeSeriesOffset);
    }

    @SafeVarargs
    public final void load(int version, List<? extends TimeSeries> ... timeSeries) {
        this.load(version, Arrays.stream(timeSeries).flatMap(Collection::stream).collect(Collectors.toList()));
    }

    public void load(int version, List<TimeSeries> timeSeriesList) {
        this.checkVersionIsInRange(version);
        Objects.requireNonNull(timeSeriesList);
        if (timeSeriesList.isEmpty()) {
            throw new TimeSeriesException("Empty time series list");
        }
        Stopwatch stopWatch = Stopwatch.createStarted();
        ArrayList<DoubleTimeSeries> doubleTimeSeries = new ArrayList<DoubleTimeSeries>();
        ArrayList<StringTimeSeries> stringTimeSeries = new ArrayList<StringTimeSeries>();
        for (TimeSeries timeSeries : timeSeriesList) {
            Objects.requireNonNull(timeSeries);
            if (timeSeries instanceof DoubleTimeSeries) {
                DoubleTimeSeries dts = (DoubleTimeSeries)timeSeries;
                doubleTimeSeries.add(dts);
                continue;
            }
            if (timeSeries instanceof StringTimeSeries) {
                StringTimeSeries sts = (StringTimeSeries)timeSeries;
                stringTimeSeries.add(sts);
                continue;
            }
            throw new IllegalStateException("Unsupported time series type " + timeSeries.getClass());
        }
        this.initTable(doubleTimeSeries, stringTimeSeries);
        for (DoubleTimeSeries doubleTimeSeries2 : doubleTimeSeries) {
            doubleTimeSeries2.synchronize(this.tableIndex);
            this.loadDouble(version, doubleTimeSeries2);
        }
        for (StringTimeSeries stringTimeSeries2 : stringTimeSeries) {
            stringTimeSeries2.synchronize(this.tableIndex);
            this.loadString(version, stringTimeSeries2);
        }
        LOGGER.info("{} time series (version={}) loaded in {} ms", new Object[]{timeSeriesList.size(), version, stopWatch.elapsed(TimeUnit.MILLISECONDS)});
    }

    public List<String> getTimeSeriesNames() {
        return this.timeSeriesMetadata.stream().map(TimeSeriesMetadata::getName).collect(Collectors.toList());
    }

    public double getDoubleValue(int version, int timeSeriesNum, int point) {
        this.checkVersionIsInRange(version);
        int doubleTimeSeriesNum = this.checkTimeSeriesNum(timeSeriesNum);
        this.checkPoint(point);
        long timeSeriesOffset = this.getTimeSeriesOffset(version, doubleTimeSeriesNum);
        return this.doubleBuffer.get(timeSeriesOffset + (long)point);
    }

    public String getStringValue(int version, int timeSeriesNum, int point) {
        this.checkVersionIsInRange(version);
        int stringTimeSeriesNum = this.checkTimeSeriesNum(timeSeriesNum);
        this.checkPoint(point);
        long timeSeriesOffset = this.getTimeSeriesOffset(version, stringTimeSeriesNum);
        return this.stringBuffer.getString(timeSeriesOffset + (long)point);
    }

    public int getDoubleTimeSeriesIndex(String timeSeriesName) {
        return this.doubleTimeSeriesNames.getIndex(timeSeriesName);
    }

    public int getStringTimeSeriesIndex(String timeSeriesName) {
        return this.stringTimeSeriesNames.getIndex(timeSeriesName);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void invalidateStatistics(int version, int timeSeriesNum) {
        int statisticsIndex = this.getStatisticsIndex(version, timeSeriesNum);
        this.statsLock.lock();
        try {
            this.means[statisticsIndex] = Double.NaN;
            this.stdDevs[statisticsIndex] = Double.NaN;
        }
        finally {
            this.statsLock.unlock();
        }
    }

    private void updateStatistics(int version, int timeSeriesNum) {
        double mean;
        int statisticsIndex = this.getStatisticsIndex(version, timeSeriesNum);
        if (!Double.isNaN(this.means[statisticsIndex]) && !Double.isNaN(this.stdDevs[statisticsIndex])) {
            return;
        }
        long timeSeriesOffset = this.getTimeSeriesOffset(version, timeSeriesNum);
        double sum = 0.0;
        int nbPoints = 0;
        for (int point = 0; point < this.tableIndex.getPointCount(); ++point) {
            double value = this.doubleBuffer.get(timeSeriesOffset + (long)point);
            if (Double.isNaN(value)) continue;
            sum += value;
            ++nbPoints;
        }
        this.means[statisticsIndex] = mean = nbPoints > 0 ? sum / (double)nbPoints : 0.0;
        double stdDev = 0.0;
        for (int point = 0; point < this.tableIndex.getPointCount(); ++point) {
            double value = this.doubleBuffer.get(timeSeriesOffset + (long)point);
            if (Double.isNaN(value)) continue;
            stdDev += (value - mean) * (value - mean);
        }
        this.stdDevs[statisticsIndex] = stdDev = nbPoints > 1 ? Math.sqrt(stdDev / (double)(nbPoints - 1)) : 0.0;
    }

    private void updateStatistics(int version) {
        for (int timeSeriesNum = 0; timeSeriesNum < this.doubleTimeSeriesNames.size(); ++timeSeriesNum) {
            this.updateStatistics(version, timeSeriesNum);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private double getStatistics(int version, int timeSeriesNum, double[] stats) {
        this.checkVersionIsInRange(version);
        int doubleTimeSeriesNum = this.checkTimeSeriesNum(timeSeriesNum);
        int statisticsIndex = this.getStatisticsIndex(version, doubleTimeSeriesNum);
        this.statsLock.lock();
        try {
            this.updateStatistics(version, timeSeriesNum);
            double d = stats[statisticsIndex];
            return d;
        }
        finally {
            this.statsLock.unlock();
        }
    }

    public double getMean(int version, int timeSeriesNum) {
        return this.getStatistics(version, timeSeriesNum, this.means);
    }

    public double getStdDev(int version, int timeSeriesNum) {
        return this.getStatistics(version, timeSeriesNum, this.stdDevs);
    }

    public List<Correlation> findMostCorrelatedTimeSeries(String timeSeriesName, int version) {
        return this.findMostCorrelatedTimeSeries(timeSeriesName, version, 10);
    }

    public List<Correlation> findMostCorrelatedTimeSeries(String timeSeriesName, int version, int maxSize) {
        double[] ppmcc = this.computePpmcc(timeSeriesName, version);
        return IntStream.range(0, ppmcc.length).mapToObj(i -> new Correlation(timeSeriesName, this.doubleTimeSeriesNames.getName(i), Math.abs(ppmcc[i]))).filter(correlation -> !correlation.getTimeSeriesName2().equals(timeSeriesName)).sorted(Comparator.comparingDouble(Correlation::getCoefficient).reversed()).limit(maxSize).collect(Collectors.toList());
    }

    private void computeConstantTimeSeriesPpmcc(double[] r, int version) {
        for (int timeSeriesNum2 = 0; timeSeriesNum2 < this.doubleTimeSeriesNames.size(); ++timeSeriesNum2) {
            int statisticsIndex2 = this.getStatisticsIndex(version, timeSeriesNum2);
            double stdDev2 = this.stdDevs[statisticsIndex2];
            r[timeSeriesNum2] = stdDev2 == 0.0 ? 1.0 : 0.0;
        }
    }

    private void computeVariableTimeSeriesPpmcc(double[] r, int timeSeriesNum1, int statisticsIndex1, double stdDev1, int version) {
        double mean1 = this.means[statisticsIndex1];
        for (int timeSeriesNum2 = 0; timeSeriesNum2 < this.doubleTimeSeriesNames.size(); ++timeSeriesNum2) {
            if (timeSeriesNum2 == timeSeriesNum1) {
                r[timeSeriesNum2] = 1.0;
                continue;
            }
            int statisticsIndex2 = this.getStatisticsIndex(version, timeSeriesNum2);
            double stdDev2 = this.stdDevs[statisticsIndex2];
            r[timeSeriesNum2] = 0.0;
            if (stdDev2 == 0.0) continue;
            double mean2 = this.means[statisticsIndex2];
            long timeSeriesOffset1 = this.getTimeSeriesOffset(version, timeSeriesNum1);
            long timeSeriesOffset2 = this.getTimeSeriesOffset(version, timeSeriesNum2);
            for (int point = 0; point < this.tableIndex.getPointCount(); ++point) {
                double value1 = this.doubleBuffer.get(timeSeriesOffset1 + (long)point);
                double value2 = this.doubleBuffer.get(timeSeriesOffset2 + (long)point);
                int n = timeSeriesNum2;
                r[n] = r[n] + (value1 - mean1) / stdDev1 * (value2 - mean2) / stdDev2;
            }
            int n = timeSeriesNum2;
            r[n] = r[n] / (double)(this.tableIndex.getPointCount() - 1);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public double[] computePpmcc(String timeSeriesName, int version) {
        int timeSeriesNum1 = this.doubleTimeSeriesNames.getIndex(timeSeriesName);
        this.checkVersionIsInRange(version);
        Stopwatch stopWatch = Stopwatch.createStarted();
        double[] r = new double[this.doubleTimeSeriesNames.size()];
        this.statsLock.lock();
        try {
            this.updateStatistics(version);
            int statisticsIndex1 = this.getStatisticsIndex(version, timeSeriesNum1);
            double stdDev1 = this.stdDevs[statisticsIndex1];
            if (stdDev1 == 0.0) {
                this.computeConstantTimeSeriesPpmcc(r, version);
            } else {
                this.computeVariableTimeSeriesPpmcc(r, timeSeriesNum1, statisticsIndex1, stdDev1, version);
            }
        }
        finally {
            this.statsLock.unlock();
        }
        LOGGER.info("PPMCC computed in {} ms", (Object)stopWatch.elapsed(TimeUnit.MILLISECONDS));
        return r;
    }

    private static BufferedWriter createWriter(Path file) throws IOException {
        if (file.getFileName().toString().endsWith(".gz")) {
            return new BufferedWriter(new OutputStreamWriter((OutputStream)new GZIPOutputStream(Files.newOutputStream(file, new OpenOption[0])), StandardCharsets.UTF_8));
        }
        return Files.newBufferedWriter(file, StandardCharsets.UTF_8, new OpenOption[0]);
    }

    public void writeCsv(Path file) {
        try (BufferedWriter writer = TimeSeriesTable.createWriter(file);){
            this.writeCsv(writer, new TimeSeriesCsvConfig());
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    public void writeCsv(Path file, TimeSeriesCsvConfig timeSeriesCsvConfig) {
        try (BufferedWriter writer = TimeSeriesTable.createWriter(file);){
            this.writeCsv(writer, timeSeriesCsvConfig);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    public String toCsvString() {
        String string;
        StringWriter writer = new StringWriter();
        try {
            this.writeCsv(writer, new TimeSeriesCsvConfig());
            string = writer.toString();
        }
        catch (Throwable throwable) {
            try {
                try {
                    writer.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        }
        writer.close();
        return string;
    }

    public String toCsvString(TimeSeriesCsvConfig timeSeriesCsvConfig) {
        String string;
        StringWriter writer = new StringWriter();
        try {
            this.writeCsv(writer, timeSeriesCsvConfig);
            string = writer.toString();
        }
        catch (Throwable throwable) {
            try {
                try {
                    writer.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        }
        writer.close();
        return string;
    }

    private void writeHeader(Writer writer, TimeSeriesCsvConfig timeSeriesCsvConfig) throws IOException {
        writer.write("Time");
        if (timeSeriesCsvConfig.versioned()) {
            writer.write(timeSeriesCsvConfig.separator());
            writer.write("Version");
        }
        if (this.timeSeriesMetadata != null) {
            for (TimeSeriesMetadata metadata : this.timeSeriesMetadata) {
                writer.write(timeSeriesCsvConfig.separator());
                writer.write(metadata.getName());
            }
        }
        writer.write(System.lineSeparator());
    }

    private void fillCache(int point, CsvCache cache, int cachedPoints, int version) {
        for (int i = 0; i < this.timeSeriesMetadata.size(); ++i) {
            int cachedPoint;
            TimeSeriesMetadata metadata = this.timeSeriesMetadata.get(i);
            int timeSeriesNum = this.timeSeriesIndexDoubleOrString.get(i);
            long timeSeriesOffset = this.getTimeSeriesOffset(version, timeSeriesNum);
            if (metadata.getDataType() == TimeSeriesDataType.DOUBLE) {
                for (cachedPoint = 0; cachedPoint < cachedPoints; ++cachedPoint) {
                    cache.doubleCache[cachedPoint * this.doubleTimeSeriesNames.size() + timeSeriesNum] = this.doubleBuffer.get(timeSeriesOffset + (long)point + (long)cachedPoint);
                }
                continue;
            }
            if (metadata.getDataType() == TimeSeriesDataType.STRING) {
                for (cachedPoint = 0; cachedPoint < cachedPoints; ++cachedPoint) {
                    cache.stringCache[cachedPoint * this.stringTimeSeriesNames.size() + timeSeriesNum] = this.stringBuffer.getString(timeSeriesOffset + (long)point + (long)cachedPoint);
                }
                continue;
            }
            throw new IllegalStateException("Unexpected data type " + metadata.getDataType());
        }
    }

    private static void writeDouble(Writer writer, double value) throws IOException {
        if (!Double.isNaN(value)) {
            writer.write(Double.toString(value));
        }
    }

    private static void writeString(Writer writer, String value) throws IOException {
        if (value != null) {
            writer.write(value);
        }
    }

    private void dumpCache(Writer writer, TimeSeriesCsvConfig timeSeriesCsvConfig, int point, CsvCache cache, int cachedPoints, int version) throws IOException {
        for (int cachedPoint = 0; cachedPoint < cachedPoints; ++cachedPoint) {
            this.writeTime(writer, timeSeriesCsvConfig, point, cachedPoint);
            if (timeSeriesCsvConfig.versioned()) {
                writer.write(timeSeriesCsvConfig.separator());
                writer.write(Integer.toString(version));
            }
            for (int i = 0; i < this.timeSeriesMetadata.size(); ++i) {
                TimeSeriesMetadata metadata = this.timeSeriesMetadata.get(i);
                int timeSeriesNum = this.timeSeriesIndexDoubleOrString.get(i);
                writer.write(timeSeriesCsvConfig.separator());
                if (metadata.getDataType() == TimeSeriesDataType.DOUBLE) {
                    double value = cache.doubleCache[cachedPoint * this.doubleTimeSeriesNames.size() + timeSeriesNum];
                    TimeSeriesTable.writeDouble(writer, value);
                    continue;
                }
                if (metadata.getDataType() == TimeSeriesDataType.STRING) {
                    String value = cache.stringCache[cachedPoint * this.stringTimeSeriesNames.size() + timeSeriesNum];
                    TimeSeriesTable.writeString(writer, value);
                    continue;
                }
                throw new IllegalStateException("Unexpected data type " + metadata.getDataType());
            }
            writer.write(System.lineSeparator());
        }
    }

    private void writeTime(Writer writer, TimeSeriesCsvConfig timeSeriesCsvConfig, int point, int cachedPoint) throws IOException {
        Instant instant = this.tableIndex.getInstantAt(point + cachedPoint);
        switch (timeSeriesCsvConfig.timeFormat()) {
            case DATE_TIME: {
                writer.write(ZonedDateTime.ofInstant(instant, ZoneId.systemDefault()).format(timeSeriesCsvConfig.dateTimeFormatter()));
                break;
            }
            case FRACTIONS_OF_SECOND: {
                writer.write(Double.toString((double)instant.getEpochSecond() + (double)instant.getNano() / 1.0E9));
                break;
            }
            case MILLIS: {
                writer.write(Long.toString(instant.toEpochMilli()));
                break;
            }
            case MICROS: {
                writer.write(TimeSeries.writeInstantToMicroString(instant));
                break;
            }
            case NANOS: {
                writer.write(TimeSeries.writeInstantToNanoString(instant));
                break;
            }
            default: {
                throw new IllegalStateException("Unknown time format " + timeSeriesCsvConfig.timeFormat());
            }
        }
    }

    public void writeCsv(Writer writer, TimeSeriesCsvConfig timeSeriesCsvConfig) throws IOException {
        Objects.requireNonNull(writer);
        Objects.requireNonNull(timeSeriesCsvConfig);
        Stopwatch stopWatch = Stopwatch.createStarted();
        try {
            this.writeHeader(writer, timeSeriesCsvConfig);
            if (this.timeSeriesMetadata != null) {
                CsvCache cache = new CsvCache();
                for (int version = this.fromVersion; version <= this.toVersion; ++version) {
                    for (int point = 0; point < this.tableIndex.getPointCount(); point += 10) {
                        int cachedPoints = Math.min(10, this.tableIndex.getPointCount() - point);
                        this.fillCache(point, cache, cachedPoints, version);
                        this.dumpCache(writer, timeSeriesCsvConfig, point, cache, cachedPoints, version);
                    }
                }
            }
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
        LOGGER.info("Csv written in {} ms", (Object)stopWatch.elapsed(TimeUnit.MILLISECONDS));
        writer.flush();
    }

    private static class TimeSeriesNameMap {
        private final BiList<String> names = new BiList();

        private TimeSeriesNameMap() {
        }

        int add(String name) {
            return this.names.add(name);
        }

        String getName(int index) {
            if (index < 0 || index >= this.names.size()) {
                throw new IllegalArgumentException("Time series at index " + index + " not found");
            }
            return this.names.get(index);
        }

        int getIndex(String name) {
            Objects.requireNonNull(name);
            int index = this.names.indexOf(name);
            if (index == -1) {
                throw new IllegalArgumentException("Time series '" + name + "' not found");
            }
            return index;
        }

        int size() {
            return this.names.size();
        }

        void clear() {
            this.names.clear();
        }
    }

    private class CsvCache {
        static final int CACHE_SIZE = 10;
        final double[] doubleCache;
        final String[] stringCache;

        private CsvCache() {
            this.doubleCache = new double[10 * TimeSeriesTable.this.doubleTimeSeriesNames.size()];
            this.stringCache = new String[10 * TimeSeriesTable.this.stringTimeSeriesNames.size()];
        }
    }

    public static class Correlation {
        private final String timeSeriesName1;
        private final String timeSeriesName2;
        private final double coefficient;

        public Correlation(String timeSeriesName1, String timeSeriesName2, double coefficient) {
            this.timeSeriesName1 = Objects.requireNonNull(timeSeriesName1);
            this.timeSeriesName2 = Objects.requireNonNull(timeSeriesName2);
            this.coefficient = coefficient;
        }

        public String getTimeSeriesName1() {
            return this.timeSeriesName1;
        }

        public String getTimeSeriesName2() {
            return this.timeSeriesName2;
        }

        public double getCoefficient() {
            return this.coefficient;
        }

        public String toString() {
            return "Correlation(timeSeriesName1=" + this.timeSeriesName1 + ", timeSeriesName2=" + this.timeSeriesName2 + ", coefficient=" + this.coefficient + ")";
        }
    }
}

