/*
 * Decompiled with CFR 0.152.
 */
package com.powsybl.ucte.converter;

import com.google.auto.service.AutoService;
import com.google.common.base.Enums;
import com.google.common.base.Stopwatch;
import com.google.common.collect.Sets;
import com.google.common.io.ByteStreams;
import com.powsybl.commons.config.PlatformConfig;
import com.powsybl.commons.datasource.DataSource;
import com.powsybl.commons.datasource.ReadOnlyDataSource;
import com.powsybl.commons.parameters.ConfiguredParameter;
import com.powsybl.commons.parameters.Parameter;
import com.powsybl.commons.parameters.ParameterDefaultValueConfig;
import com.powsybl.commons.parameters.ParameterType;
import com.powsybl.commons.report.ReportNode;
import com.powsybl.entsoe.util.EntsoeAreaAdder;
import com.powsybl.entsoe.util.EntsoeFileName;
import com.powsybl.entsoe.util.EntsoeGeographicalCode;
import com.powsybl.iidm.network.Area;
import com.powsybl.iidm.network.AreaAdder;
import com.powsybl.iidm.network.Bus;
import com.powsybl.iidm.network.BusAdder;
import com.powsybl.iidm.network.Country;
import com.powsybl.iidm.network.CurrentLimitsAdder;
import com.powsybl.iidm.network.DanglingLine;
import com.powsybl.iidm.network.DanglingLineAdder;
import com.powsybl.iidm.network.DanglingLineFilter;
import com.powsybl.iidm.network.EnergySource;
import com.powsybl.iidm.network.Generator;
import com.powsybl.iidm.network.GeneratorAdder;
import com.powsybl.iidm.network.Identifiable;
import com.powsybl.iidm.network.Importer;
import com.powsybl.iidm.network.Line;
import com.powsybl.iidm.network.LineAdder;
import com.powsybl.iidm.network.LoadAdder;
import com.powsybl.iidm.network.Network;
import com.powsybl.iidm.network.NetworkFactory;
import com.powsybl.iidm.network.PhaseTapChanger;
import com.powsybl.iidm.network.PhaseTapChangerAdder;
import com.powsybl.iidm.network.RatioTapChangerAdder;
import com.powsybl.iidm.network.RatioTapChangerStep;
import com.powsybl.iidm.network.Substation;
import com.powsybl.iidm.network.SubstationAdder;
import com.powsybl.iidm.network.Switch;
import com.powsybl.iidm.network.TieLine;
import com.powsybl.iidm.network.TieLineAdder;
import com.powsybl.iidm.network.TopologyKind;
import com.powsybl.iidm.network.TwoWindingsTransformer;
import com.powsybl.iidm.network.TwoWindingsTransformerAdder;
import com.powsybl.iidm.network.VoltageLevel;
import com.powsybl.iidm.network.VoltageLevelAdder;
import com.powsybl.iidm.network.extensions.SlackTerminal;
import com.powsybl.ucte.converter.UcteException;
import com.powsybl.ucte.network.UcteAngleRegulation;
import com.powsybl.ucte.network.UcteCountryCode;
import com.powsybl.ucte.network.UcteElement;
import com.powsybl.ucte.network.UcteElementStatus;
import com.powsybl.ucte.network.UcteLine;
import com.powsybl.ucte.network.UcteNetwork;
import com.powsybl.ucte.network.UcteNode;
import com.powsybl.ucte.network.UcteNodeCode;
import com.powsybl.ucte.network.UcteNodeStatus;
import com.powsybl.ucte.network.UcteNodeTypeCode;
import com.powsybl.ucte.network.UctePhaseRegulation;
import com.powsybl.ucte.network.UctePowerPlantType;
import com.powsybl.ucte.network.UcteRegulation;
import com.powsybl.ucte.network.UcteTransformer;
import com.powsybl.ucte.network.UcteVoltageLevelCode;
import com.powsybl.ucte.network.ext.UcteNetworkExt;
import com.powsybl.ucte.network.ext.UcteSubstation;
import com.powsybl.ucte.network.ext.UcteVoltageLevel;
import com.powsybl.ucte.network.io.UcteReader;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.UncheckedIOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@AutoService(value={Importer.class})
public class UcteImporter
implements Importer {
    private static final Logger LOGGER = LoggerFactory.getLogger(UcteImporter.class);
    private static final double LINE_MIN_Z = 0.05;
    private static final String[] EXTENSIONS = new String[]{"uct", "UCT"};
    public static final String COMBINE_PHASE_ANGLE_REGULATION = "ucte.import.combine-phase-angle-regulation";
    public static final String CREATE_AREAS = "ucte.import.create-areas";
    public static final String AREAS_DC_XNODES = "ucte.import.areas-dc-xnodes";
    public static final String UNEXPECTED_UCTE_ELEMENT_STATUS = "Unexpected UcteElementStatus value: ";
    public static final String X_NODE = "_XNode";
    private static final Parameter COMBINE_PHASE_ANGLE_REGULATION_PARAMETER = new Parameter("ucte.import.combine-phase-angle-regulation", ParameterType.BOOLEAN, "Combine phase and angle regulation", (Object)false);
    private static final Parameter CREATE_AREAS_PARAMETER = new Parameter("ucte.import.create-areas", ParameterType.BOOLEAN, "Create Areas", (Object)true);
    private static final Parameter AREAS_DC_XNODES_PARAMETER = new Parameter("ucte.import.areas-dc-xnodes", ParameterType.STRING_LIST, "X-Nodes to be considered as DC when creating area boundaries", List.of());
    private static final List<Parameter> PARAMETERS = List.of(COMBINE_PHASE_ANGLE_REGULATION_PARAMETER, CREATE_AREAS_PARAMETER, AREAS_DC_XNODES_PARAMETER);
    private final ParameterDefaultValueConfig defaultValueConfig;

    public UcteImporter() {
        this(PlatformConfig.defaultConfig());
    }

    public UcteImporter(PlatformConfig platformConfig) {
        this.defaultValueConfig = new ParameterDefaultValueConfig(platformConfig);
    }

    private static double getConductance(UcteTransformer ucteTransfo) {
        double g = 0.0;
        if (!Double.isNaN(ucteTransfo.getConductance())) {
            g = ucteTransfo.getConductance();
        }
        return g;
    }

    private static double getSusceptance(UcteElement ucteElement) {
        double b = 0.0;
        if (!Double.isNaN(ucteElement.getSusceptance())) {
            b = ucteElement.getSusceptance();
        }
        return b;
    }

    private static boolean isFictitious(UcteElement ucteElement) {
        return switch (ucteElement.getStatus()) {
            default -> throw new MatchException(null, null);
            case UcteElementStatus.EQUIVALENT_ELEMENT_IN_OPERATION, UcteElementStatus.EQUIVALENT_ELEMENT_OUT_OF_OPERATION -> true;
            case UcteElementStatus.REAL_ELEMENT_IN_OPERATION, UcteElementStatus.REAL_ELEMENT_OUT_OF_OPERATION, UcteElementStatus.BUSBAR_COUPLER_IN_OPERATION, UcteElementStatus.BUSBAR_COUPLER_OUT_OF_OPERATION -> false;
        };
    }

    private static boolean isFictitious(UcteNode ucteNode) {
        return switch (ucteNode.getStatus()) {
            default -> throw new MatchException(null, null);
            case UcteNodeStatus.EQUIVALENT -> true;
            case UcteNodeStatus.REAL -> false;
        };
    }

    private static EntsoeGeographicalCode getRegionalGeographicalCode(Substation substation) {
        if (substation.getCountry().map(country -> country != Country.DE).orElse(true).booleanValue()) {
            return null;
        }
        EntsoeGeographicalCode res = (EntsoeGeographicalCode)Enums.getIfPresent(EntsoeGeographicalCode.class, (String)substation.getNameOrId().substring(0, 2)).orNull();
        return res == EntsoeGeographicalCode.DE ? null : res;
    }

    private static void createBuses(UcteNetworkExt ucteNetwork, UcteVoltageLevel ucteVoltageLevel, VoltageLevel voltageLevel) {
        for (UcteNodeCode ucteNodeCode : ucteVoltageLevel.getNodes()) {
            UcteNode ucteNode = ucteNetwork.getNode(ucteNodeCode);
            if (ucteNode.getCode().getUcteCountryCode() == UcteCountryCode.XX) continue;
            if (LOGGER.isTraceEnabled()) {
                LOGGER.trace("Create bus '{}'", (Object)ucteNodeCode);
            }
            Bus bus = ((BusAdder)((BusAdder)voltageLevel.getBusBreakerView().newBus().setId(ucteNodeCode.toString())).setFictitious(UcteImporter.isFictitious(ucteNode))).add();
            UcteImporter.addGeographicalNameProperty(ucteNode, bus);
            if (UcteImporter.isValueValid(ucteNode.getActiveLoad()) || UcteImporter.isValueValid(ucteNode.getReactiveLoad())) {
                UcteImporter.createLoad(ucteNode, voltageLevel, bus);
            }
            if (ucteNode.isGenerator()) {
                UcteImporter.createGenerator(ucteNode, voltageLevel, bus);
            }
            if (ucteNode.getTypeCode() != UcteNodeTypeCode.UT) continue;
            SlackTerminal.attach((Bus)bus);
        }
    }

    private static void createBuses(UcteNetworkExt ucteNetwork, Network network) {
        for (UcteSubstation ucteSubstation : ucteNetwork.getSubstations()) {
            UcteNodeCode firstUcteNodeCode = ucteSubstation.getNodes().stream().filter(code -> code.getUcteCountryCode() != UcteCountryCode.XX).findFirst().orElse(null);
            if (firstUcteNodeCode == null) continue;
            LOGGER.trace("Create substation '{}'", (Object)ucteSubstation.getName());
            Substation substation = ((SubstationAdder)network.newSubstation().setId(ucteSubstation.getName())).setCountry(EntsoeGeographicalCode.valueOf((String)firstUcteNodeCode.getUcteCountryCode().name()).getCountry()).add();
            EntsoeGeographicalCode regionalCode = UcteImporter.getRegionalGeographicalCode(substation);
            if (regionalCode != null) {
                ((EntsoeAreaAdder)substation.newExtension(EntsoeAreaAdder.class)).withCode(regionalCode).add();
            }
            for (UcteVoltageLevel ucteVoltageLevel : ucteSubstation.getVoltageLevels()) {
                UcteVoltageLevelCode ucteVoltageLevelCode = ((UcteNodeCode)ucteVoltageLevel.getNodes().iterator().next()).getVoltageLevelCode();
                LOGGER.trace("Create voltage level '{}'", (Object)ucteVoltageLevel.getName());
                VoltageLevel voltageLevel = ((VoltageLevelAdder)substation.newVoltageLevel().setId(ucteVoltageLevel.getName())).setNominalV((double)ucteVoltageLevelCode.getVoltageLevel()).setTopologyKind(TopologyKind.BUS_BREAKER).add();
                UcteImporter.createBuses(ucteNetwork, ucteVoltageLevel, voltageLevel);
            }
        }
    }

    private static boolean isValueValid(double value) {
        return !Double.isNaN(value) && value != 0.0;
    }

    private static void createLoad(UcteNode ucteNode, VoltageLevel voltageLevel, Bus bus) {
        String loadId = bus.getId() + "_load";
        LOGGER.trace("Create load '{}'", (Object)loadId);
        double p0 = 0.0;
        if (UcteImporter.isValueValid(ucteNode.getActiveLoad())) {
            p0 = ucteNode.getActiveLoad();
        }
        double q0 = 0.0;
        if (UcteImporter.isValueValid(ucteNode.getReactiveLoad())) {
            q0 = ucteNode.getReactiveLoad();
        }
        ((LoadAdder)((LoadAdder)((LoadAdder)voltageLevel.newLoad().setId(loadId)).setBus(bus.getId())).setConnectableBus(bus.getId())).setP0(p0).setQ0(q0).add();
    }

    private static void createGenerator(UcteNode ucteNode, VoltageLevel voltageLevel, Bus bus) {
        String generatorId = bus.getId() + "_generator";
        LOGGER.trace("Create generator '{}'", (Object)generatorId);
        EnergySource energySource = EnergySource.OTHER;
        if (ucteNode.getPowerPlantType() != null) {
            energySource = switch (ucteNode.getPowerPlantType()) {
                default -> throw new MatchException(null, null);
                case UctePowerPlantType.C, UctePowerPlantType.G, UctePowerPlantType.L, UctePowerPlantType.O -> EnergySource.THERMAL;
                case UctePowerPlantType.H -> EnergySource.HYDRO;
                case UctePowerPlantType.N -> EnergySource.NUCLEAR;
                case UctePowerPlantType.W -> EnergySource.WIND;
                case UctePowerPlantType.F -> EnergySource.OTHER;
            };
        }
        double generatorP = UcteImporter.isValueValid(ucteNode.getActivePowerGeneration()) ? -ucteNode.getActivePowerGeneration() : 0.0;
        double generatorQ = UcteImporter.isValueValid(ucteNode.getReactivePowerGeneration()) ? -ucteNode.getReactivePowerGeneration() : 0.0;
        Generator generator = ((GeneratorAdder)((GeneratorAdder)((GeneratorAdder)voltageLevel.newGenerator().setId(generatorId)).setEnergySource(energySource).setBus(bus.getId())).setConnectableBus(bus.getId())).setMinP(-ucteNode.getMinimumPermissibleActivePowerGeneration()).setMaxP(-ucteNode.getMaximumPermissibleActivePowerGeneration()).setVoltageRegulatorOn(ucteNode.isRegulatingVoltage()).setTargetP(generatorP).setTargetQ(generatorQ).setTargetV(ucteNode.getVoltageReference()).add();
        generator.newMinMaxReactiveLimits().setMinQ(-ucteNode.getMinimumPermissibleReactivePowerGeneration()).setMaxQ(-ucteNode.getMaximumPermissibleReactivePowerGeneration()).add();
        if (ucteNode.getPowerPlantType() != null) {
            generator.setProperty("powerPlantType", ucteNode.getPowerPlantType().toString());
        }
    }

    private static void createDanglingLine(UcteLine ucteLine, boolean connected, UcteNode xnode, UcteNodeCode nodeCode, UcteVoltageLevel ucteVoltageLevel, Network network) {
        LOGGER.trace("Create dangling line '{}' (X-node='{}')", (Object)ucteLine.getId(), (Object)xnode.getCode());
        double p0 = UcteImporter.isValueValid(xnode.getActiveLoad()) ? xnode.getActiveLoad() : 0.0;
        double q0 = UcteImporter.isValueValid(xnode.getReactiveLoad()) ? xnode.getReactiveLoad() : 0.0;
        double targetP = UcteImporter.isValueValid(xnode.getActivePowerGeneration()) ? xnode.getActivePowerGeneration() : 0.0;
        double targetQ = UcteImporter.isValueValid(xnode.getReactivePowerGeneration()) ? xnode.getReactivePowerGeneration() : 0.0;
        VoltageLevel voltageLevel = network.getVoltageLevel(ucteVoltageLevel.getName());
        DanglingLine dl = ((DanglingLineAdder)((DanglingLineAdder)((DanglingLineAdder)((DanglingLineAdder)((DanglingLineAdder)voltageLevel.newDanglingLine().setId(ucteLine.getId().toString())).setName(xnode.getGeographicalName())).setBus(connected ? nodeCode.toString() : null)).setConnectableBus(nodeCode.toString())).setR(ucteLine.getResistance()).setX(ucteLine.getReactance()).setG(0.0).setB(UcteImporter.getSusceptance((UcteElement)ucteLine)).setP0(p0).setQ0(q0).setPairingKey(xnode.getCode().toString()).setFictitious(UcteImporter.isFictitious((UcteElement)ucteLine))).newGeneration().setTargetP(-targetP).setTargetQ(-targetQ).add().add();
        if (xnode.isRegulatingVoltage()) {
            dl.getGeneration().setTargetV(xnode.getVoltageReference()).setVoltageRegulationOn(true).setMaxP(-xnode.getMaximumPermissibleActivePowerGeneration()).setMinP(-xnode.getMinimumPermissibleActivePowerGeneration());
            dl.getGeneration().newMinMaxReactiveLimits().setMinQ(-xnode.getMinimumPermissibleReactivePowerGeneration()).setMaxQ(-xnode.getMaximumPermissibleReactivePowerGeneration()).add();
        }
        if (ucteLine.getCurrentLimit() != null) {
            ((CurrentLimitsAdder)dl.getOrCreateSelectedOperationalLimitsGroup().newCurrentLimits().setPermanentLimit((double)ucteLine.getCurrentLimit().intValue())).add();
        }
        UcteImporter.addElementNameProperty((UcteElement)ucteLine, dl);
        UcteImporter.addGeographicalNameProperty(xnode, dl);
        UcteImporter.addXnodeStatusProperty(xnode, dl);
        UcteImporter.addDanglingLineCouplerProperty(ucteLine, dl);
    }

    private static void createCoupler(UcteNetworkExt ucteNetwork, Network network, UcteLine ucteLine, UcteNodeCode nodeCode1, UcteNodeCode nodeCode2, UcteVoltageLevel ucteVoltageLevel1, UcteVoltageLevel ucteVoltageLevel2) {
        LOGGER.trace("Create coupler '{}'", (Object)ucteLine.getId());
        if (ucteVoltageLevel1 != ucteVoltageLevel2) {
            throw new UcteException("Coupler between two different voltage levels");
        }
        boolean connected = UcteImporter.isConnected((UcteElement)ucteLine);
        if (nodeCode1.getUcteCountryCode() == UcteCountryCode.XX && nodeCode2.getUcteCountryCode() != UcteCountryCode.XX) {
            UcteImporter.createDanglingLine(ucteLine, connected, ucteNetwork.getNode(nodeCode1), nodeCode2, ucteVoltageLevel2, network);
        } else if (nodeCode2.getUcteCountryCode() == UcteCountryCode.XX && nodeCode1.getUcteCountryCode() != UcteCountryCode.XX) {
            UcteImporter.createDanglingLine(ucteLine, connected, ucteNetwork.getNode(nodeCode2), nodeCode1, ucteVoltageLevel1, network);
        } else {
            double z = Math.hypot(ucteLine.getResistance(), ucteLine.getReactance());
            UcteImporter.createCouplerFromLowImpedanceLine(network, ucteLine, nodeCode1, nodeCode2, ucteVoltageLevel1, ucteVoltageLevel2, connected, z);
        }
    }

    private static void createCouplerFromLowImpedanceLine(Network network, UcteLine ucteLine, UcteNodeCode nodeCode1, UcteNodeCode nodeCode2, UcteVoltageLevel ucteVoltageLevel1, UcteVoltageLevel ucteVoltageLevel2, boolean connected, double z) {
        LOGGER.info("Create coupler '{}' from low impedance line ({} ohm)", (Object)ucteLine.getId(), (Object)z);
        if (ucteVoltageLevel1 != ucteVoltageLevel2) {
            throw new UcteException("Nodes coupled with a low impedance line are expected to be in the same voltage level");
        }
        if (nodeCode1.equals((Object)nodeCode2)) {
            LOGGER.error("Coupler '{}' has same bus at both ends: ignored", (Object)ucteLine.getId());
            return;
        }
        VoltageLevel voltageLevel = network.getVoltageLevel(ucteVoltageLevel1.getName());
        Switch couplerSwitch = ((VoltageLevel.BusBreakerView.SwitchAdder)((VoltageLevel.BusBreakerView.SwitchAdder)((VoltageLevel.BusBreakerView.SwitchAdder)voltageLevel.getBusBreakerView().newSwitch().setEnsureIdUnicity(true)).setId(ucteLine.getId().toString())).setBus1(nodeCode1.toString()).setBus2(nodeCode2.toString()).setOpen(!connected).setFictitious(UcteImporter.isFictitious((UcteElement)ucteLine))).add();
        UcteImporter.addCurrentLimitProperty(ucteLine, couplerSwitch);
        UcteImporter.addOrderCodeProperty(ucteLine, couplerSwitch);
        UcteImporter.addElementNameProperty((UcteElement)ucteLine, couplerSwitch);
    }

    private static void createStandardLine(Network network, UcteLine ucteLine, UcteNodeCode nodeCode1, UcteNodeCode nodeCode2, UcteVoltageLevel ucteVoltageLevel1, UcteVoltageLevel ucteVoltageLevel2, boolean connected) {
        LOGGER.trace("Create line '{}'", (Object)ucteLine.getId());
        Line l = ((LineAdder)((LineAdder)((LineAdder)((LineAdder)((LineAdder)((LineAdder)((LineAdder)((LineAdder)((LineAdder)network.newLine().setEnsureIdUnicity(true)).setId(ucteLine.getId().toString())).setVoltageLevel1(ucteVoltageLevel1.getName())).setVoltageLevel2(ucteVoltageLevel2.getName())).setBus1(connected ? nodeCode1.toString() : null)).setBus2(connected ? nodeCode2.toString() : null)).setConnectableBus1(nodeCode1.toString())).setConnectableBus2(nodeCode2.toString())).setR(ucteLine.getResistance()).setX(ucteLine.getReactance()).setG1(0.0).setG2(0.0).setB1(UcteImporter.getSusceptance((UcteElement)ucteLine) / 2.0).setB2(UcteImporter.getSusceptance((UcteElement)ucteLine) / 2.0).setFictitious(UcteImporter.isFictitious((UcteElement)ucteLine))).add();
        UcteImporter.addElementNameProperty((UcteElement)ucteLine, l);
        if (ucteLine.getCurrentLimit() != null) {
            int currentLimit = ucteLine.getCurrentLimit();
            ((CurrentLimitsAdder)l.getOrCreateSelectedOperationalLimitsGroup1().newCurrentLimits().setPermanentLimit((double)currentLimit)).add();
            ((CurrentLimitsAdder)l.getOrCreateSelectedOperationalLimitsGroup2().newCurrentLimits().setPermanentLimit((double)currentLimit)).add();
        }
    }

    private static void createLine(UcteNetworkExt ucteNetwork, Network network, UcteLine ucteLine, UcteNodeCode nodeCode1, UcteNodeCode nodeCode2, UcteVoltageLevel ucteVoltageLevel1, UcteVoltageLevel ucteVoltageLevel2) {
        boolean connected = UcteImporter.isConnected((UcteElement)ucteLine);
        double z = Math.hypot(ucteLine.getResistance(), ucteLine.getReactance());
        if (z < 0.05 && nodeCode1.getUcteCountryCode() != UcteCountryCode.XX && nodeCode2.getUcteCountryCode() != UcteCountryCode.XX) {
            UcteImporter.createCouplerFromLowImpedanceLine(network, ucteLine, nodeCode1, nodeCode2, ucteVoltageLevel1, ucteVoltageLevel2, connected, z);
        } else if (nodeCode1.getUcteCountryCode() != UcteCountryCode.XX && nodeCode2.getUcteCountryCode() != UcteCountryCode.XX) {
            UcteImporter.createStandardLine(network, ucteLine, nodeCode1, nodeCode2, ucteVoltageLevel1, ucteVoltageLevel2, connected);
        } else if (nodeCode1.getUcteCountryCode() == UcteCountryCode.XX && nodeCode2.getUcteCountryCode() != UcteCountryCode.XX) {
            UcteNode xnode = ucteNetwork.getNode(nodeCode1);
            UcteImporter.createDanglingLine(ucteLine, connected, xnode, nodeCode2, ucteVoltageLevel2, network);
        } else if (nodeCode1.getUcteCountryCode() != UcteCountryCode.XX && nodeCode2.getUcteCountryCode() == UcteCountryCode.XX) {
            UcteNode xnode = ucteNetwork.getNode(nodeCode2);
            UcteImporter.createDanglingLine(ucteLine, connected, xnode, nodeCode1, ucteVoltageLevel1, network);
        } else {
            throw new UcteException("Line between 2 X-nodes: '" + String.valueOf(nodeCode1) + "' and '" + String.valueOf(nodeCode2) + "'");
        }
    }

    private static void createLines(UcteNetworkExt ucteNetwork, Network network) {
        block4: for (UcteLine ucteLine : ucteNetwork.getLines()) {
            UcteNodeCode nodeCode1 = ucteLine.getId().getNodeCode1();
            UcteNodeCode nodeCode2 = ucteLine.getId().getNodeCode2();
            UcteVoltageLevel ucteVoltageLevel1 = ucteNetwork.getVoltageLevel(nodeCode1);
            UcteVoltageLevel ucteVoltageLevel2 = ucteNetwork.getVoltageLevel(nodeCode2);
            switch (ucteLine.getStatus()) {
                case BUSBAR_COUPLER_IN_OPERATION: 
                case BUSBAR_COUPLER_OUT_OF_OPERATION: {
                    UcteImporter.createCoupler(ucteNetwork, network, ucteLine, nodeCode1, nodeCode2, ucteVoltageLevel1, ucteVoltageLevel2);
                    continue block4;
                }
                case EQUIVALENT_ELEMENT_IN_OPERATION: 
                case EQUIVALENT_ELEMENT_OUT_OF_OPERATION: 
                case REAL_ELEMENT_IN_OPERATION: 
                case REAL_ELEMENT_OUT_OF_OPERATION: {
                    UcteImporter.createLine(ucteNetwork, network, ucteLine, nodeCode1, nodeCode2, ucteVoltageLevel1, ucteVoltageLevel2);
                    continue block4;
                }
            }
            throw new IllegalStateException(UNEXPECTED_UCTE_ELEMENT_STATUS + String.valueOf(ucteLine.getStatus()));
        }
    }

    private static void createRatioTapChanger(UctePhaseRegulation uctePhaseRegulation, TwoWindingsTransformer transformer) {
        LOGGER.trace("Create ratio tap changer '{}'", (Object)transformer.getId());
        UcteImporter.createRatioTapChangerAdder(uctePhaseRegulation, transformer).add();
    }

    private static RatioTapChangerAdder createRatioTapChangerAdder(UctePhaseRegulation uctePhaseRegulation, TwoWindingsTransformer transformer) {
        int lowerTap = UcteImporter.getLowTapPosition(uctePhaseRegulation, transformer);
        RatioTapChangerAdder rtca = (RatioTapChangerAdder)((RatioTapChangerAdder)((RatioTapChangerAdder)transformer.newRatioTapChanger().setLowTapPosition(lowerTap)).setTapPosition(uctePhaseRegulation.getNp().intValue())).setLoadTapChangingCapabilities(!Double.isNaN(uctePhaseRegulation.getU()));
        if (!Double.isNaN(uctePhaseRegulation.getU())) {
            ((RatioTapChangerAdder)((RatioTapChangerAdder)((RatioTapChangerAdder)rtca.setLoadTapChangingCapabilities(true)).setRegulating(true)).setTargetV(uctePhaseRegulation.getU()).setTargetDeadband(0.0)).setRegulationTerminal(transformer.getTerminal1());
        }
        for (int i = lowerTap; i <= Math.abs(lowerTap); ++i) {
            double rho = 1.0 / (1.0 + (double)i * uctePhaseRegulation.getDu() / 100.0);
            ((RatioTapChangerAdder.StepAdder)((RatioTapChangerAdder.StepAdder)((RatioTapChangerAdder.StepAdder)((RatioTapChangerAdder.StepAdder)((RatioTapChangerAdder.StepAdder)((RatioTapChangerAdder.StepAdder)rtca.beginStep()).setRho(rho)).setR(0.0)).setX(0.0)).setG(0.0)).setB(0.0)).endStep();
        }
        return rtca;
    }

    private static PhaseTapChangerAdder createPhaseTapChangerAdder(UcteAngleRegulation ucteAngleRegulation, TwoWindingsTransformer transformer, Double currentRatioTapChangerRho) {
        int lowerTap = UcteImporter.getLowTapPosition(ucteAngleRegulation, transformer);
        PhaseTapChangerAdder ptca = (PhaseTapChangerAdder)((PhaseTapChangerAdder)transformer.newPhaseTapChanger().setLowTapPosition(lowerTap)).setTapPosition(ucteAngleRegulation.getNp().intValue());
        if (!Double.isNaN(ucteAngleRegulation.getP())) {
            ((PhaseTapChangerAdder)ptca.setRegulationValue(-ucteAngleRegulation.getP()).setRegulationMode(PhaseTapChanger.RegulationMode.ACTIVE_POWER_CONTROL).setTargetDeadband(0.0)).setRegulationTerminal(transformer.getTerminal1());
        } else {
            ptca.setRegulationMode(PhaseTapChanger.RegulationMode.CURRENT_LIMITER);
        }
        ptca.setRegulating(false);
        for (int i = lowerTap; i <= Math.abs(lowerTap); ++i) {
            double dx = (double)i * ucteAngleRegulation.getDu() / 100.0 * Math.cos(Math.toRadians(ucteAngleRegulation.getTheta()));
            double dy = (double)i * ucteAngleRegulation.getDu() / 100.0 * Math.sin(Math.toRadians(ucteAngleRegulation.getTheta()));
            Pair<Double, Double> rhoAndAlpha = UcteImporter.getRhoAndAlpha(ucteAngleRegulation, dx, dy, currentRatioTapChangerRho);
            ((PhaseTapChangerAdder.StepAdder)((PhaseTapChangerAdder.StepAdder)((PhaseTapChangerAdder.StepAdder)((PhaseTapChangerAdder.StepAdder)((PhaseTapChangerAdder.StepAdder)((PhaseTapChangerAdder.StepAdder)((PhaseTapChangerAdder.StepAdder)ptca.beginStep()).setRho(((Double)rhoAndAlpha.getLeft()).doubleValue())).setAlpha(-((Double)rhoAndAlpha.getRight()).doubleValue())).setR(0.0)).setX(0.0)).setG(0.0)).setB(0.0)).endStep();
        }
        return ptca;
    }

    private static void createPhaseTapChanger(UcteAngleRegulation ucteAngleRegulation, TwoWindingsTransformer transformer) {
        LOGGER.trace("Create phase tap changer '{}'", (Object)transformer.getId());
        UcteImporter.createPhaseTapChangerAdder(ucteAngleRegulation, transformer, null).add();
    }

    private static Pair<Double, Double> getRhoAndAlpha(UcteAngleRegulation ucteAngleRegulation, double dx, double dy, Double currentRatioTapChangerRho) {
        double alpha;
        double rho;
        if (currentRatioTapChangerRho != null && currentRatioTapChangerRho.equals(0.0)) {
            throw new IllegalStateException("Unexpected non zero value for current ratio tap changer rho: " + currentRatioTapChangerRho);
        }
        switch (ucteAngleRegulation.getType()) {
            case ASYM: {
                if (currentRatioTapChangerRho == null) {
                    rho = 1.0 / Math.hypot(dy, 1.0 + dx);
                    alpha = Math.toDegrees(Math.atan2(dy, 1.0 + dx));
                    break;
                }
                double dxEq = dx + 1.0 / currentRatioTapChangerRho - 1.0;
                rho = 1.0 / Math.hypot(dy, 1.0 + dxEq) / currentRatioTapChangerRho;
                alpha = Math.toDegrees(Math.atan2(dy, 1.0 + dxEq));
                break;
            }
            case SYMM: {
                double dyHalf = dy / 2.0;
                double coeff = 1.0;
                if (currentRatioTapChangerRho != null) {
                    coeff = 2.0 * currentRatioTapChangerRho - 1.0;
                }
                double gamma = Math.toDegrees(Math.atan2(dyHalf * coeff, dx + 1.0));
                double dy22 = dyHalf * dyHalf;
                alpha = gamma + Math.toDegrees(Math.atan2(dyHalf, 1.0 + dx));
                rho = Math.sqrt((1.0 + dy22) / (1.0 + dy22 * coeff * coeff));
                break;
            }
            default: {
                throw new IllegalStateException("Unexpected UcteAngleRegulationType value: " + String.valueOf(ucteAngleRegulation.getType()));
            }
        }
        return Pair.of((Object)rho, (Object)alpha);
    }

    private static void createRatioAndPhaseTapChanger(UcteAngleRegulation ucteAngleRegulation, UctePhaseRegulation uctePhaseRegulation, TwoWindingsTransformer transformer) {
        LOGGER.trace("Create phase tap changer combining both ratio and phase tap '{}'", (Object)transformer.getId());
        UcteImporter.createRatioTapChangerAdder(uctePhaseRegulation, transformer).add();
        transformer.getRatioTapChanger().setRegulating(false);
        double currentRatioTapChangerRho = ((RatioTapChangerStep)transformer.getRatioTapChanger().getCurrentStep()).getRho();
        UcteImporter.createPhaseTapChangerAdder(ucteAngleRegulation, transformer, currentRatioTapChangerRho).add();
    }

    private static int getLowTapPosition(UctePhaseRegulation uctePhaseRegulation, TwoWindingsTransformer transformer) {
        return UcteImporter.getLowTapPosition(transformer, uctePhaseRegulation.getN(), uctePhaseRegulation.getNp());
    }

    private static int getLowTapPosition(UcteAngleRegulation ucteAngleRegulation, TwoWindingsTransformer transformer) {
        return UcteImporter.getLowTapPosition(transformer, ucteAngleRegulation.getN(), ucteAngleRegulation.getNp());
    }

    private static int getLowTapPosition(TwoWindingsTransformer transformer, int initialTapsNumber, int currentTapPosition) {
        int floor;
        if (initialTapsNumber >= Math.abs(currentTapPosition)) {
            floor = -initialTapsNumber;
        } else {
            LOGGER.warn("Tap position for transformer '{}' is '{}', absolute value should be equal or lower than number of Taps '{}'", new Object[]{transformer.getId(), currentTapPosition, initialTapsNumber});
            floor = currentTapPosition < 0 ? currentTapPosition : -currentTapPosition;
            LOGGER.info("Number of Taps for transformer '{}' is extended from '{}', to '{}'", new Object[]{transformer.getId(), initialTapsNumber, Math.abs(floor)});
        }
        return floor;
    }

    private static TwoWindingsTransformer createXnodeTransfo(UcteNetworkExt ucteNetwork, UcteTransformer ucteTransfo, boolean connected, UcteNodeCode xNodeCode, UcteNodeCode ucteOtherNodeCode, UcteVoltageLevel ucteOtherVoltageLevel, Substation substation, EntsoeFileName ucteFileName) {
        Object busId2;
        Object busId1;
        String voltageLevelId2;
        String voltageLevelId1;
        String xNodeName = xNodeCode.toString();
        String yNodeName = ucteFileName.getCountry() != null ? ucteFileName.getCountry() + "_" + xNodeName : "YNODE_" + xNodeName;
        VoltageLevel yVoltageLevel = ((VoltageLevelAdder)substation.newVoltageLevel().setId(yNodeName + "_VL")).setNominalV((double)xNodeCode.getVoltageLevelCode().getVoltageLevel()).setTopologyKind(TopologyKind.BUS_BREAKER).add();
        ((BusAdder)((BusAdder)yVoltageLevel.getBusBreakerView().newBus().setId(yNodeName)).setFictitious(true)).add();
        UcteNode ucteXnode = ucteNetwork.getNode(xNodeCode);
        LOGGER.warn("Create small impedance dangling line '{}{}' (transformer connected to X-node '{}')", new Object[]{xNodeName, yNodeName, ucteXnode.getCode()});
        double p0 = UcteImporter.isValueValid(ucteXnode.getActiveLoad()) ? ucteXnode.getActiveLoad() : 0.0;
        double q0 = UcteImporter.isValueValid(ucteXnode.getReactiveLoad()) ? ucteXnode.getReactiveLoad() : 0.0;
        double targetP = UcteImporter.isValueValid(ucteXnode.getActivePowerGeneration()) ? ucteXnode.getActivePowerGeneration() : 0.0;
        double targetQ = UcteImporter.isValueValid(ucteXnode.getReactivePowerGeneration()) ? ucteXnode.getReactivePowerGeneration() : 0.0;
        DanglingLine yDanglingLine = ((DanglingLineAdder)((DanglingLineAdder)((DanglingLineAdder)yVoltageLevel.newDanglingLine().setId(xNodeName + " " + yNodeName)).setBus(yNodeName)).setConnectableBus(yNodeName)).setR(0.0).setX(0.05).setG(0.0).setB(0.0).setP0(p0).setQ0(q0).setPairingKey(ucteXnode.getCode().toString()).newGeneration().setTargetP(-targetP).setTargetQ(-targetQ).add().add();
        UcteImporter.addXnodeStatusProperty(ucteXnode, yDanglingLine);
        UcteImporter.addGeographicalNameProperty(ucteXnode, yDanglingLine);
        if (ucteXnode.getCode().equals((Object)ucteTransfo.getId().getNodeCode1())) {
            voltageLevelId1 = ucteOtherVoltageLevel.getName();
            voltageLevelId2 = yVoltageLevel.getId();
            busId1 = ucteOtherNodeCode.toString();
            busId2 = yNodeName;
        } else {
            voltageLevelId1 = yVoltageLevel.getId();
            voltageLevelId2 = ucteOtherVoltageLevel.getName();
            busId1 = yNodeName;
            busId2 = ucteOtherNodeCode.toString();
        }
        return ((TwoWindingsTransformerAdder)((TwoWindingsTransformerAdder)((TwoWindingsTransformerAdder)((TwoWindingsTransformerAdder)((TwoWindingsTransformerAdder)((TwoWindingsTransformerAdder)((TwoWindingsTransformerAdder)((TwoWindingsTransformerAdder)substation.newTwoWindingsTransformer().setEnsureIdUnicity(true)).setId(ucteTransfo.getId().toString())).setVoltageLevel1(voltageLevelId1)).setVoltageLevel2(voltageLevelId2)).setBus1((String)(connected ? busId1 : null))).setBus2((String)(connected ? busId2 : null))).setConnectableBus1((String)busId1)).setConnectableBus2((String)busId2)).setRatedU1(ucteTransfo.getRatedVoltage2()).setRatedU2(ucteTransfo.getRatedVoltage1()).setR(ucteTransfo.getResistance()).setX(ucteTransfo.getReactance()).setG(UcteImporter.getConductance(ucteTransfo)).setB(UcteImporter.getSusceptance((UcteElement)ucteTransfo)).add();
    }

    private static boolean isConnected(UcteElement ucteElement) {
        return switch (ucteElement.getStatus()) {
            default -> throw new MatchException(null, null);
            case UcteElementStatus.EQUIVALENT_ELEMENT_IN_OPERATION, UcteElementStatus.REAL_ELEMENT_IN_OPERATION, UcteElementStatus.BUSBAR_COUPLER_IN_OPERATION -> true;
            case UcteElementStatus.EQUIVALENT_ELEMENT_OUT_OF_OPERATION, UcteElementStatus.REAL_ELEMENT_OUT_OF_OPERATION, UcteElementStatus.BUSBAR_COUPLER_OUT_OF_OPERATION -> false;
        };
    }

    private static void addTapChangers(UcteNetworkExt ucteNetwork, UcteTransformer ucteTransfo, TwoWindingsTransformer transformer, boolean combinePhaseAngleRegulation) {
        UcteRegulation ucteRegulation = ucteNetwork.getRegulation(ucteTransfo.getId());
        if (ucteRegulation != null) {
            if (combinePhaseAngleRegulation && ucteRegulation.getPhaseRegulation() != null && ucteRegulation.getAngleRegulation() != null) {
                UcteImporter.createRatioAndPhaseTapChanger(ucteRegulation.getAngleRegulation(), ucteRegulation.getPhaseRegulation(), transformer);
            } else {
                if (ucteRegulation.getPhaseRegulation() != null) {
                    UcteImporter.createRatioTapChanger(ucteRegulation.getPhaseRegulation(), transformer);
                }
                if (ucteRegulation.getAngleRegulation() != null) {
                    UcteImporter.createPhaseTapChanger(ucteRegulation.getAngleRegulation(), transformer);
                }
            }
        }
    }

    private static void createTransformers(UcteNetworkExt ucteNetwork, Network network, EntsoeFileName ucteFileName, boolean combinePhaseAngleRegulation) {
        for (UcteTransformer ucteTransfo : ucteNetwork.getTransformers()) {
            UcteNodeCode nodeCode1 = ucteTransfo.getId().getNodeCode1();
            UcteNodeCode nodeCode2 = ucteTransfo.getId().getNodeCode2();
            UcteVoltageLevel ucteVoltageLevel1 = ucteNetwork.getVoltageLevel(nodeCode1);
            UcteVoltageLevel ucteVoltageLevel2 = ucteNetwork.getVoltageLevel(nodeCode2);
            UcteSubstation ucteSubstation = ucteVoltageLevel1.getSubstation();
            Substation substation = network.getSubstation(ucteSubstation.getName());
            LOGGER.trace("Create transformer '{}'", (Object)ucteTransfo.getId());
            boolean connected = UcteImporter.isConnected((UcteElement)ucteTransfo);
            TwoWindingsTransformer transformer = nodeCode1.getUcteCountryCode() == UcteCountryCode.XX && nodeCode2.getUcteCountryCode() != UcteCountryCode.XX ? UcteImporter.createXnodeTransfo(ucteNetwork, ucteTransfo, connected, nodeCode1, nodeCode2, ucteVoltageLevel2, substation, ucteFileName) : (nodeCode2.getUcteCountryCode() == UcteCountryCode.XX && nodeCode1.getUcteCountryCode() != UcteCountryCode.XX ? UcteImporter.createXnodeTransfo(ucteNetwork, ucteTransfo, connected, nodeCode2, nodeCode1, ucteVoltageLevel1, substation, ucteFileName) : ((TwoWindingsTransformerAdder)((TwoWindingsTransformerAdder)((TwoWindingsTransformerAdder)((TwoWindingsTransformerAdder)((TwoWindingsTransformerAdder)((TwoWindingsTransformerAdder)((TwoWindingsTransformerAdder)((TwoWindingsTransformerAdder)((TwoWindingsTransformerAdder)substation.newTwoWindingsTransformer().setEnsureIdUnicity(true)).setId(ucteTransfo.getId().toString())).setVoltageLevel1(ucteVoltageLevel2.getName())).setVoltageLevel2(ucteVoltageLevel1.getName())).setBus1(connected ? nodeCode2.toString() : null)).setBus2(connected ? nodeCode1.toString() : null)).setConnectableBus1(nodeCode2.toString())).setConnectableBus2(nodeCode1.toString())).setRatedU1(ucteTransfo.getRatedVoltage2()).setRatedU2(ucteTransfo.getRatedVoltage1()).setR(ucteTransfo.getResistance()).setX(ucteTransfo.getReactance()).setG(UcteImporter.getConductance(ucteTransfo)).setB(UcteImporter.getSusceptance((UcteElement)ucteTransfo)).setFictitious(UcteImporter.isFictitious((UcteElement)ucteTransfo))).add());
            if (ucteTransfo.getCurrentLimit() != null) {
                int currentLimit = ucteTransfo.getCurrentLimit();
                ((CurrentLimitsAdder)transformer.getOrCreateSelectedOperationalLimitsGroup2().newCurrentLimits().setPermanentLimit((double)currentLimit)).add();
            }
            UcteImporter.addElementNameProperty((UcteElement)ucteTransfo, transformer);
            UcteImporter.addTapChangers(ucteNetwork, ucteTransfo, transformer, combinePhaseAngleRegulation);
            UcteImporter.addNominalPowerProperty(ucteTransfo, transformer);
        }
    }

    private static DanglingLine getMatchingDanglingLine(DanglingLine dl1, Map<String, List<DanglingLine>> danglingLinesByPairingKey) {
        String otherPairingKey = dl1.getPairingKey();
        List<DanglingLine> matchingDanglingLines = danglingLinesByPairingKey.get(otherPairingKey).stream().filter(dl -> dl != dl1).toList();
        if (matchingDanglingLines.isEmpty()) {
            return null;
        }
        if (matchingDanglingLines.size() == 1) {
            return matchingDanglingLines.get(0);
        }
        if (!dl1.getTerminal().isConnected()) {
            return null;
        }
        List<DanglingLine> connectedMatchingDanglingLines = matchingDanglingLines.stream().filter(dl -> dl.getTerminal().isConnected()).toList();
        if (connectedMatchingDanglingLines.isEmpty()) {
            return null;
        }
        if (connectedMatchingDanglingLines.size() == 1) {
            return connectedMatchingDanglingLines.get(0);
        }
        throw new UcteException("More that 2 connected dangling lines have the same pairing key " + dl1.getPairingKey());
    }

    private static void addElementNameProperty(Map<String, String> properties, DanglingLine dl1, DanglingLine dl2) {
        if (dl1.hasProperty("elementName")) {
            properties.put("elementName_1", dl1.getProperty("elementName"));
        }
        if (dl2.hasProperty("elementName")) {
            properties.put("elementName_2", dl2.getProperty("elementName"));
        }
    }

    private static void addElementNameProperty(UcteElement ucteElement, Identifiable<?> identifiable) {
        if (ucteElement.getElementName() != null && !ucteElement.getElementName().isEmpty()) {
            identifiable.setProperty("elementName", ucteElement.getElementName());
        }
    }

    private static void addCurrentLimitProperty(UcteLine ucteLine, Switch aSwitch) {
        if (ucteLine.getCurrentLimit() != null) {
            aSwitch.setProperty("currentLimit", String.valueOf(ucteLine.getCurrentLimit()));
        }
    }

    private static void addGeographicalNameProperty(UcteNode ucteNode, Identifiable<?> identifiable) {
        if (ucteNode.getGeographicalName() != null) {
            identifiable.setProperty("geographicalName", ucteNode.getGeographicalName());
        }
    }

    private static void addGeographicalNameProperty(UcteNetwork ucteNetwork, Map<String, String> properties, DanglingLine dl1) {
        Optional optUcteNodeCode = UcteNodeCode.parseUcteNodeCode((String)dl1.getPairingKey());
        if (!optUcteNodeCode.isPresent()) {
            throw new UcteException("It's not possible to import this network");
        }
        UcteNode ucteNode = ucteNetwork.getNode((UcteNodeCode)optUcteNodeCode.get());
        properties.put("geographicalName", ucteNode.getGeographicalName());
    }

    private static void addOrderCodeProperty(UcteLine ucteLine, Switch sw) {
        String ucteLineId = ucteLine.getId().toString();
        sw.setProperty("orderCode", String.valueOf(ucteLineId.charAt(ucteLineId.length() - 1)));
    }

    private static void addNominalPowerProperty(UcteTransformer transformer, TwoWindingsTransformer twoWindingsTransformer) {
        if (!Double.isNaN(transformer.getNominalPower())) {
            twoWindingsTransformer.setProperty("nomimalPower", String.valueOf(transformer.getNominalPower()));
        }
    }

    private static void addXnodeStatusProperty(UcteNode ucteNode, Identifiable<?> identifiable) {
        identifiable.setProperty("status_XNode", ucteNode.getStatus().toString());
    }

    private static void addXnodeStatusProperty(Map<String, String> properties, DanglingLine danglingLine) {
        properties.put("status_XNode", danglingLine.getProperty("status_XNode"));
    }

    private static void addDanglingLineCouplerProperty(UcteLine ucteLine, DanglingLine danglingLine) {
        switch (ucteLine.getStatus()) {
            case BUSBAR_COUPLER_IN_OPERATION: 
            case BUSBAR_COUPLER_OUT_OF_OPERATION: {
                danglingLine.setProperty("isCoupler", "true");
                break;
            }
            case EQUIVALENT_ELEMENT_IN_OPERATION: 
            case EQUIVALENT_ELEMENT_OUT_OF_OPERATION: 
            case REAL_ELEMENT_IN_OPERATION: 
            case REAL_ELEMENT_OUT_OF_OPERATION: {
                danglingLine.setProperty("isCoupler", "false");
            }
        }
    }

    public String getFormat() {
        return "UCTE";
    }

    public List<String> getSupportedExtensions() {
        return Arrays.asList(EXTENSIONS);
    }

    public String getComment() {
        return "UCTE-DEF";
    }

    public List<Parameter> getParameters() {
        return ConfiguredParameter.load(PARAMETERS, (String)this.getFormat(), (ParameterDefaultValueConfig)this.defaultValueConfig);
    }

    private String findExtension(ReadOnlyDataSource dataSource, boolean throwException) throws IOException {
        for (String ext : EXTENSIONS) {
            if (!dataSource.isDataExtension(ext) || !dataSource.exists(null, ext)) continue;
            return ext;
        }
        if (throwException) {
            throw new UcteException("File " + dataSource.getBaseName() + "." + String.join((CharSequence)"|", EXTENSIONS) + " not found");
        }
        return null;
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public boolean exists(ReadOnlyDataSource dataSource) {
        boolean bl;
        String ext = this.findExtension(dataSource, false);
        if (ext == null) return false;
        BufferedReader reader = new BufferedReader(new InputStreamReader(dataSource.newInputStream(null, ext)));
        try {
            bl = new UcteReader().checkHeader(reader);
        }
        catch (Throwable throwable) {
            try {
                try {
                    reader.close();
                    throw throwable;
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        }
        reader.close();
        return bl;
    }

    private static void mergeDanglingLines(UcteNetwork ucteNetwork, Network network) {
        HashMap<String, List<DanglingLine>> danglingLinesByPairingKey = new HashMap<String, List<DanglingLine>>();
        for (DanglingLine dl : network.getDanglingLines(DanglingLineFilter.ALL)) {
            danglingLinesByPairingKey.computeIfAbsent(dl.getPairingKey(), code -> new ArrayList()).add(dl);
        }
        HashSet danglingLinesToProcess = Sets.newHashSet((Iterable)network.getDanglingLines(DanglingLineFilter.ALL));
        while (!danglingLinesToProcess.isEmpty()) {
            DanglingLine dlToProcess = (DanglingLine)danglingLinesToProcess.iterator().next();
            DanglingLine dlMatchingDlToProcess = UcteImporter.getMatchingDanglingLine(dlToProcess, danglingLinesByPairingKey);
            if (dlMatchingDlToProcess != null) {
                boolean switchDanglingLinesOrder = dlToProcess.getId().compareTo(dlMatchingDlToProcess.getId()) > 0;
                DanglingLine dlAtSideOne = switchDanglingLinesOrder ? dlMatchingDlToProcess : dlToProcess;
                DanglingLine dlAtSideTwo = switchDanglingLinesOrder ? dlToProcess : dlMatchingDlToProcess;
                UcteImporter.createTieLine(ucteNetwork, network, dlAtSideOne, dlAtSideTwo);
                danglingLinesToProcess.remove(dlMatchingDlToProcess);
            }
            danglingLinesToProcess.remove(dlToProcess);
        }
    }

    private static void createTieLine(UcteNetwork ucteNetwork, Network network, DanglingLine dlAtSideOne, DanglingLine dlAtSideTwo) {
        String mergeLineId = dlAtSideOne.getId() + " + " + dlAtSideTwo.getId();
        TieLine mergeLine = ((TieLineAdder)network.newTieLine().setId(mergeLineId)).setDanglingLine1(dlAtSideOne.getId()).setDanglingLine2(dlAtSideTwo.getId()).add();
        HashMap<String, String> properties = new HashMap<String, String>();
        UcteImporter.addElementNameProperty(properties, dlAtSideOne, dlAtSideTwo);
        UcteImporter.addGeographicalNameProperty(ucteNetwork, properties, dlAtSideOne);
        UcteImporter.addXnodeStatusProperty(properties, dlAtSideOne);
        properties.forEach((arg_0, arg_1) -> ((TieLine)mergeLine).setProperty(arg_0, arg_1));
    }

    public void copy(ReadOnlyDataSource fromDataSource, DataSource toDataSource) {
        Objects.requireNonNull(fromDataSource);
        Objects.requireNonNull(toDataSource);
        try {
            String ext = this.findExtension(fromDataSource, true);
            try (InputStream is = fromDataSource.newInputStream(null, ext);
                 OutputStream os = toDataSource.newOutputStream(null, ext, false);){
                ByteStreams.copy((InputStream)is, (OutputStream)os);
            }
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    public Network importData(ReadOnlyDataSource dataSource, NetworkFactory networkFactory, Properties parameters, ReportNode reportNode) {
        Network network;
        String ext = this.findExtension(dataSource, true);
        BufferedReader reader = new BufferedReader(new InputStreamReader(dataSource.newInputStream(null, ext)));
        try {
            Stopwatch stopwatch = Stopwatch.createStarted();
            boolean combinePhaseAngleRegulation = Parameter.readBoolean((String)this.getFormat(), (Properties)parameters, (Parameter)COMBINE_PHASE_ANGLE_REGULATION_PARAMETER, (ParameterDefaultValueConfig)this.defaultValueConfig);
            boolean createAreas = Parameter.readBoolean((String)this.getFormat(), (Properties)parameters, (Parameter)CREATE_AREAS_PARAMETER, (ParameterDefaultValueConfig)this.defaultValueConfig);
            Set<String> areaDcXnodes = Parameter.readStringList((String)this.getFormat(), (Properties)parameters, (Parameter)AREAS_DC_XNODES_PARAMETER, (ParameterDefaultValueConfig)this.defaultValueConfig).stream().collect(Collectors.toUnmodifiableSet());
            UcteNetworkExt ucteNetwork = new UcteNetworkExt(new UcteReader().read(reader, reportNode), 0.05);
            String fileName = dataSource.getBaseName();
            EntsoeFileName ucteFileName = EntsoeFileName.parse((String)fileName);
            Network network2 = networkFactory.createNetwork(fileName, "UCTE");
            network2.setCaseDate(ucteFileName.getDate());
            network2.setForecastDistance(ucteFileName.getForecastDistance());
            UcteImporter.createBuses(ucteNetwork, network2);
            UcteImporter.createLines(ucteNetwork, network2);
            UcteImporter.createTransformers(ucteNetwork, network2, ucteFileName, combinePhaseAngleRegulation);
            UcteImporter.mergeDanglingLines((UcteNetwork)ucteNetwork, network2);
            if (createAreas) {
                UcteImporter.createAreas(network2, areaDcXnodes);
            }
            stopwatch.stop();
            LOGGER.debug("UCTE import done in {} ms", (Object)stopwatch.elapsed(TimeUnit.MILLISECONDS));
            network = network2;
        }
        catch (Throwable throwable) {
            try {
                try {
                    reader.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        }
        reader.close();
        return network;
    }

    private static void createAreas(Network network, Set<String> areaDcXnodes) {
        EnumMap countryArea = new EnumMap(Country.class);
        network.getSubstationStream().forEach(substation -> {
            Country country = (Country)substation.getCountry().orElseThrow(() -> new IllegalStateException("No country set for substation '" + substation.getId() + "'"));
            Area area = countryArea.computeIfAbsent(country, k -> ((AreaAdder)network.newArea().setAreaType("ControlArea").setId(country.toString())).add());
            substation.getVoltageLevelStream().forEach(vl -> {
                area.addVoltageLevel(vl);
                vl.getDanglingLines().forEach(dl -> area.newAreaBoundary().setBoundary(dl.getBoundary()).setAc(!areaDcXnodes.contains(dl.getPairingKey())).add());
            });
        });
    }
}

