package com.prezi.haxe.gradle;

import com.google.common.base.Charsets;
import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.io.Files;
import com.prezi.haxe.gradle.incubating.LanguageSourceSet;
import org.apache.commons.io.FileUtils;
import org.gradle.api.DomainObjectSet;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.regex.Pattern;

public class HaxeTestCompile extends HaxeCompile {
	private static final Pattern EXAMPLE_TEST_LINE = Pattern.compile("(import |\\s*add\\()ExampleTest\\)?;");
	private static final Pattern NO_JUNIT_CLIENT = Pattern.compile("\\s+runner.addResultClient.*;");
	private static final Pattern JUNIT_CLIENT = Pattern.compile("\\s+//\\s*runner.addResultClient.*;");
	private File workingDirectory;

	@Override
	public void compile() throws IOException, InterruptedException {
		File workDir = getWorkingDirectory();
		FileUtils.deleteDirectory(workDir);
		FileUtils.forceMkdir(workDir);

		File testsDir = getTestsDirectory();

		for (File sourceDir : super.getSourceDirectories(getSourceSets())) {
			if (sourceDir.exists()) {
				FileUtils.copyDirectory(sourceDir, testsDir);
			}
		}

		List<String> cmd = Arrays.asList("haxelib", "run", "munit", "gen", testsDir.getName());
		CommandExecutor.execute(cmd, workDir, new DefaultExecutionResultHandler(cmd));
		postProcessGeneratedSources(testsDir);

		super.compile();
	}

	protected void postProcessGeneratedSources(File testsDir) throws IOException {
		// Delete ExampleTest.hx generated by MUnit
		FileUtils.deleteQuietly(new File(testsDir, "ExampleTest.hx"));
		File suite = new File(testsDir, "TestSuite.hx");
		if (suite.exists()) {
			List<String> suiteText = Files.readLines(suite, Charsets.UTF_8);
			FileUtils.deleteQuietly(suite);
			Files.write(Joiner.on("\n").join(Iterables.transform(suiteText, new Function<String, String>() {
				@Override
				public String apply(String line) {
					String result;
					if (EXAMPLE_TEST_LINE.matcher(line).matches()) {
						result = "// " + line;
					} else {
						result = line;
					}
					return result;
				}
			})), suite, Charsets.UTF_8);
		}

		File main = new File(testsDir, "TestMain.hx");
		if (main.exists()) {
			List<String> mainText = Files.readLines(main, Charsets.UTF_8);
			FileUtils.deleteQuietly(main);
			Files.write(Joiner.on("\n").join(Iterables.transform(mainText, new Function<String, String>() {
				@Override
				public String apply(String line) {
					String result;
					if (NO_JUNIT_CLIENT.matcher(line).matches()) {
						result = "// " + line;
					} else if (JUNIT_CLIENT.matcher(line).matches()) {
						result = line.replaceFirst("//", "");
					} else {
						result = line;
					}
					return result;
				}
			})), main, Charsets.UTF_8);
		}
	}

	@Override
	protected HaxeCommandBuilder configureHaxeCommandBuilder(File output, DomainObjectSet<LanguageSourceSet> sources) {
		HaxeCommandBuilder builder = super.configureHaxeCommandBuilder(output, sources);
		if (getProject().hasProperty("munit.debug")) {
			builder.withFlags(new ArrayList<String>(Arrays.asList("-D testDebug")));
		}

		return builder;
	}

	@Override
	protected String getMainClass() {
		return "TestMain";
	}

	@Override
	protected Set<File> getSourceDirectories(DomainObjectSet<LanguageSourceSet> sources) {
		return Collections.singleton(getTestsDirectory());
	}

	public File getTestsDirectory() {
		return new File(getWorkingDirectory(), "tests");
	}

	public void workingDirectory(Object workingDirectory) {
		setWorkingDirectory(workingDirectory);
	}
	public File getWorkingDirectory() {
		return workingDirectory;
	}
	public void setWorkingDirectory(Object workingDirectory) {
		this.workingDirectory = getProject().file(workingDirectory);
	}
}
