/*
 * Decompiled with CFR 0.152.
 */
package com.proofpoint.stats;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Function;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.base.Ticker;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableListMultimap;
import com.google.common.collect.Iterators;
import com.google.common.collect.Multimaps;
import com.google.common.collect.Ordering;
import com.google.common.collect.PeekingIterator;
import com.google.common.util.concurrent.AtomicDouble;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import javax.annotation.concurrent.NotThreadSafe;
import org.openjdk.jol.info.ClassLayout;

@NotThreadSafe
public class QuantileDigest {
    private static final int MAX_BITS = 64;
    private static final double MAX_SIZE_FACTOR = 1.5;
    static final long RESCALE_THRESHOLD_SECONDS = 50L;
    static final double ZERO_WEIGHT_THRESHOLD = 1.0E-5;
    private final double maxError;
    private final Ticker ticker;
    private final double alpha;
    private final boolean compressAutomatically;
    private Node root;
    private double weightedCount;
    private long max = Long.MIN_VALUE;
    private long min = Long.MAX_VALUE;
    private long landmarkInSeconds;
    private int totalNodeCount = 0;
    private int nonZeroNodeCount = 0;
    private int compressions = 0;

    public QuantileDigest(double maxError) {
        this(maxError, 0.0);
    }

    public QuantileDigest(double maxError, double alpha) {
        this(maxError, alpha, Ticker.systemTicker(), true);
    }

    @VisibleForTesting
    QuantileDigest(double maxError, double alpha, Ticker ticker, boolean compressAutomatically) {
        Preconditions.checkArgument((maxError >= 0.0 && maxError <= 1.0 ? 1 : 0) != 0, (Object)"maxError must be in range [0, 1]");
        Preconditions.checkArgument((alpha >= 0.0 && alpha < 1.0 ? 1 : 0) != 0, (Object)"alpha must be in range [0, 1)");
        this.maxError = maxError;
        this.alpha = alpha;
        this.ticker = ticker;
        this.compressAutomatically = compressAutomatically;
        this.landmarkInSeconds = TimeUnit.NANOSECONDS.toSeconds(ticker.read());
    }

    public QuantileDigest(QuantileDigest quantileDigest) {
        this(quantileDigest.getMaxError(), quantileDigest.getAlpha());
        this.merge(quantileDigest);
    }

    public double getMaxError() {
        return this.maxError;
    }

    public double getAlpha() {
        return this.alpha;
    }

    public void add(long value) {
        this.add(value, 1L);
    }

    public void add(long value, long count) {
        Preconditions.checkArgument((count > 0L ? 1 : 0) != 0, (Object)"count must be > 0");
        long nowInSeconds = TimeUnit.NANOSECONDS.toSeconds(this.ticker.read());
        int maxExpectedNodeCount = 3 * this.calculateCompressionFactor();
        if (nowInSeconds - this.landmarkInSeconds >= 50L) {
            this.rescale(nowInSeconds);
            this.compress();
        } else if ((double)this.nonZeroNodeCount > 1.5 * (double)maxExpectedNodeCount && this.compressAutomatically) {
            this.compress();
        }
        double weight = this.weight(TimeUnit.NANOSECONDS.toSeconds(this.ticker.read())) * (double)count;
        this.max = Math.max(this.max, value);
        this.min = Math.min(this.min, value);
        this.insert(QuantileDigest.longToBits(value), weight);
    }

    public void merge(QuantileDigest other) {
        this.rescaleToCommonLandmark(this, other);
        this.root = this.merge(this.root, other.root);
        this.max = Math.max(this.max, other.max);
        this.min = Math.min(this.min, other.min);
        this.compress();
    }

    public List<Long> getQuantilesLowerBound(List<Double> quantiles) {
        Preconditions.checkArgument((boolean)Ordering.natural().isOrdered(quantiles), (Object)"quantiles must be sorted in increasing order");
        for (double quantile : quantiles) {
            Preconditions.checkArgument((quantile >= 0.0 && quantile <= 1.0 ? 1 : 0) != 0, (Object)"quantile must be between [0,1]");
        }
        ImmutableList reversedQuantiles = ImmutableList.copyOf(quantiles).reverse();
        final ImmutableList.Builder builder = ImmutableList.builder();
        final PeekingIterator iterator = Iterators.peekingIterator(reversedQuantiles.iterator());
        this.postOrderTraversal(this.root, new Callback(){
            private double sum;

            @Override
            public boolean process(Node node) {
                this.sum += node.weightedCount;
                while (iterator.hasNext() && this.sum > (1.0 - (Double)iterator.peek()) * QuantileDigest.this.weightedCount) {
                    iterator.next();
                    long value = Math.max(node.getLowerBound(), QuantileDigest.this.min);
                    builder.add((Object)value);
                }
                return iterator.hasNext();
            }
        }, TraversalOrder.REVERSE);
        while (iterator.hasNext()) {
            builder.add((Object)this.min);
            iterator.next();
        }
        return builder.build().reverse();
    }

    public List<Long> getQuantilesUpperBound(List<Double> quantiles) {
        Preconditions.checkArgument((boolean)Ordering.natural().isOrdered(quantiles), (Object)"quantiles must be sorted in increasing order");
        for (double quantile : quantiles) {
            Preconditions.checkArgument((quantile >= 0.0 && quantile <= 1.0 ? 1 : 0) != 0, (Object)"quantile must be between [0,1]");
        }
        final ImmutableList.Builder builder = ImmutableList.builder();
        final PeekingIterator iterator = Iterators.peekingIterator(quantiles.iterator());
        this.postOrderTraversal(this.root, new Callback(){
            private double sum = 0.0;

            @Override
            public boolean process(Node node) {
                this.sum += node.weightedCount;
                while (iterator.hasNext() && this.sum > (Double)iterator.peek() * QuantileDigest.this.weightedCount) {
                    iterator.next();
                    long value = Math.min(node.getUpperBound(), QuantileDigest.this.max);
                    builder.add((Object)value);
                }
                return iterator.hasNext();
            }
        });
        while (iterator.hasNext()) {
            builder.add((Object)this.max);
            iterator.next();
        }
        return builder.build();
    }

    public List<Long> getQuantiles(List<Double> quantiles) {
        return this.getQuantilesUpperBound(quantiles);
    }

    public long getQuantile(double quantile) {
        return this.getQuantiles((List<Double>)ImmutableList.of((Object)quantile)).get(0);
    }

    public long getQuantileLowerBound(double quantile) {
        return this.getQuantilesLowerBound((List<Double>)ImmutableList.of((Object)quantile)).get(0);
    }

    public long getQuantileUpperBound(double quantile) {
        return this.getQuantilesUpperBound((List<Double>)ImmutableList.of((Object)quantile)).get(0);
    }

    public double getCount() {
        return this.weightedCount / this.weight(TimeUnit.NANOSECONDS.toSeconds(this.ticker.read()));
    }

    public List<Bucket> getHistogram(List<Long> bucketUpperBounds) {
        Preconditions.checkArgument((boolean)Ordering.natural().isOrdered(bucketUpperBounds), (Object)"buckets must be sorted in increasing order");
        final ImmutableList.Builder builder = ImmutableList.builder();
        final PeekingIterator iterator = Iterators.peekingIterator(bucketUpperBounds.iterator());
        final AtomicDouble sum = new AtomicDouble();
        final AtomicDouble lastSum = new AtomicDouble();
        final AtomicDouble bucketWeightedSum = new AtomicDouble();
        final double normalizationFactor = this.weight(TimeUnit.NANOSECONDS.toSeconds(this.ticker.read()));
        this.postOrderTraversal(this.root, new Callback(){

            @Override
            public boolean process(Node node) {
                while (iterator.hasNext() && (Long)iterator.peek() <= node.getUpperBound()) {
                    double bucketCount = sum.get() - lastSum.get();
                    Bucket bucket = new Bucket(bucketCount / normalizationFactor, bucketWeightedSum.get() / bucketCount);
                    builder.add((Object)bucket);
                    lastSum.set(sum.get());
                    bucketWeightedSum.set(0.0);
                    iterator.next();
                }
                bucketWeightedSum.addAndGet((double)node.getMiddle() * node.weightedCount);
                sum.addAndGet(node.weightedCount);
                return iterator.hasNext();
            }
        });
        while (iterator.hasNext()) {
            double bucketCount = sum.get() - lastSum.get();
            Bucket bucket = new Bucket(bucketCount / normalizationFactor, bucketWeightedSum.get() / bucketCount);
            builder.add((Object)bucket);
            iterator.next();
        }
        return builder.build();
    }

    public long getMin() {
        final AtomicLong chosen = new AtomicLong(this.min);
        this.postOrderTraversal(this.root, new Callback(){

            @Override
            public boolean process(Node node) {
                if (node.weightedCount >= 1.0E-5) {
                    chosen.set(node.getLowerBound());
                    return false;
                }
                return true;
            }
        }, TraversalOrder.FORWARD);
        return Math.max(this.min, chosen.get());
    }

    public long getMax() {
        final AtomicLong chosen = new AtomicLong(this.max);
        this.postOrderTraversal(this.root, new Callback(){

            @Override
            public boolean process(Node node) {
                if (node.weightedCount >= 1.0E-5) {
                    chosen.set(node.getUpperBound());
                    return false;
                }
                return true;
            }
        }, TraversalOrder.REVERSE);
        return Math.min(this.max, chosen.get());
    }

    public int estimatedInMemorySizeInBytes() {
        return SizeOf.QUANTILE_DIGEST + this.totalNodeCount * SizeOf.NODE;
    }

    public int estimatedSerializedSizeInBytes() {
        int estimatedNodeSize = 18;
        return 44 + this.totalNodeCount * estimatedNodeSize;
    }

    public void serialize(final DataOutput output) {
        try {
            output.writeDouble(this.maxError);
            output.writeDouble(this.alpha);
            output.writeLong(this.landmarkInSeconds);
            output.writeLong(this.min);
            output.writeLong(this.max);
            output.writeInt(this.totalNodeCount);
            this.postOrderTraversal(this.root, new Callback(){

                @Override
                public boolean process(Node node) {
                    try {
                        QuantileDigest.this.serializeNode(output, node);
                    }
                    catch (IOException e) {
                        Throwables.propagate((Throwable)e);
                    }
                    return true;
                }
            });
        }
        catch (IOException e) {
            Throwables.propagate((Throwable)e);
        }
    }

    private void serializeNode(DataOutput output, Node node) throws IOException {
        int flags = 0;
        if (node.left != null) {
            flags |= 1;
        }
        if (node.right != null) {
            flags |= 2;
        }
        output.writeByte(flags);
        output.writeByte(node.level);
        output.writeLong(node.bits);
        output.writeDouble(node.weightedCount);
    }

    public static QuantileDigest deserialize(DataInput input) {
        try {
            double maxError = input.readDouble();
            double alpha = input.readDouble();
            QuantileDigest result = new QuantileDigest(maxError, alpha);
            result.landmarkInSeconds = input.readLong();
            result.min = input.readLong();
            result.max = input.readLong();
            result.totalNodeCount = input.readInt();
            ArrayDeque<Node> stack = new ArrayDeque<Node>();
            for (int i = 0; i < result.totalNodeCount; ++i) {
                byte flags = input.readByte();
                Node node = QuantileDigest.deserializeNode(input);
                if ((flags & 2) != 0) {
                    node.right = (Node)stack.pop();
                }
                if ((flags & 1) != 0) {
                    node.left = (Node)stack.pop();
                }
                stack.push(node);
                result.weightedCount += node.weightedCount;
                if (!(node.weightedCount >= 1.0E-5)) continue;
                ++result.nonZeroNodeCount;
            }
            if (!stack.isEmpty()) {
                Preconditions.checkArgument((stack.size() == 1 ? 1 : 0) != 0, (Object)"Tree is corrupted. Expected a single root node");
                result.root = (Node)stack.pop();
            }
            return result;
        }
        catch (IOException e) {
            throw Throwables.propagate((Throwable)e);
        }
    }

    private static Node deserializeNode(DataInput input) throws IOException {
        int level = input.readUnsignedByte();
        long value = input.readLong();
        double weight = input.readDouble();
        return new Node(value, level, weight);
    }

    @VisibleForTesting
    int getTotalNodeCount() {
        return this.totalNodeCount;
    }

    @VisibleForTesting
    int getNonZeroNodeCount() {
        return this.nonZeroNodeCount;
    }

    @VisibleForTesting
    int getCompressions() {
        return this.compressions;
    }

    @VisibleForTesting
    void compress() {
        ++this.compressions;
        final int compressionFactor = this.calculateCompressionFactor();
        this.postOrderTraversal(this.root, new Callback(){

            @Override
            public boolean process(Node node) {
                Node node2;
                if (node.isLeaf()) {
                    return true;
                }
                double leftWeight = 0.0;
                if (node.left != null) {
                    leftWeight = node.left.weightedCount;
                }
                double rightWeight = 0.0;
                if (node.right != null) {
                    rightWeight = node.right.weightedCount;
                }
                boolean shouldCompress = node.weightedCount + leftWeight + rightWeight < (double)((int)(QuantileDigest.this.weightedCount / (double)compressionFactor));
                double oldNodeWeight = node.weightedCount;
                if (shouldCompress || leftWeight < 1.0E-5) {
                    node.left = QuantileDigest.this.tryRemove(node.left);
                    QuantileDigest.this.weightedCount = QuantileDigest.this.weightedCount + leftWeight;
                    node2 = node;
                    node2.weightedCount = node2.weightedCount + leftWeight;
                }
                if (shouldCompress || rightWeight < 1.0E-5) {
                    node.right = QuantileDigest.this.tryRemove(node.right);
                    QuantileDigest.this.weightedCount = QuantileDigest.this.weightedCount + rightWeight;
                    node2 = node;
                    node2.weightedCount = node2.weightedCount + rightWeight;
                }
                if (oldNodeWeight < 1.0E-5 && node.weightedCount >= 1.0E-5) {
                    ++QuantileDigest.this.nonZeroNodeCount;
                }
                return true;
            }
        });
        if (this.root != null && this.root.weightedCount < 1.0E-5) {
            this.root = this.tryRemove(this.root);
        }
    }

    private double weight(long timestamp) {
        return Math.exp(this.alpha * (double)(timestamp - this.landmarkInSeconds));
    }

    private void rescale(long newLandmarkInSeconds) {
        final double factor = Math.exp(-this.alpha * (double)(newLandmarkInSeconds - this.landmarkInSeconds));
        this.weightedCount *= factor;
        this.postOrderTraversal(this.root, new Callback(){

            @Override
            public boolean process(Node node) {
                double oldWeight = node.weightedCount;
                Node node2 = node;
                node2.weightedCount = node2.weightedCount * factor;
                if (oldWeight >= 1.0E-5 && node.weightedCount < 1.0E-5) {
                    --QuantileDigest.this.nonZeroNodeCount;
                }
                return true;
            }
        });
        this.landmarkInSeconds = newLandmarkInSeconds;
    }

    private int calculateCompressionFactor() {
        if (this.root == null) {
            return 1;
        }
        return Math.max((int)((double)(this.root.level + 1) / this.maxError), 1);
    }

    private void insert(long bits, double weight) {
        long lastBranch = 0L;
        Node parent = null;
        Node current = this.root;
        while (true) {
            if (current == null) {
                this.setChild(parent, lastBranch, this.createLeaf(bits, weight));
                return;
            }
            if (!QuantileDigest.inSameSubtree(bits, current.bits, current.level)) {
                this.setChild(parent, lastBranch, this.makeSiblings(current, this.createLeaf(bits, weight)));
                return;
            }
            if (current.level == 0 && current.bits == bits) {
                double oldWeight = current.weightedCount;
                Node node = current;
                node.weightedCount = node.weightedCount + weight;
                if (current.weightedCount >= 1.0E-5 && oldWeight < 1.0E-5) {
                    ++this.nonZeroNodeCount;
                }
                this.weightedCount += weight;
                return;
            }
            long branch = bits & current.getBranchMask();
            parent = current;
            lastBranch = branch;
            if (branch == 0L) {
                current = current.left;
                continue;
            }
            current = current.right;
        }
    }

    private void setChild(Node parent, long branch, Node child) {
        if (parent == null) {
            this.root = child;
        } else if (branch == 0L) {
            parent.left = child;
        } else {
            parent.right = child;
        }
    }

    private Node makeSiblings(Node node, Node sibling) {
        int parentLevel = 64 - Long.numberOfLeadingZeros(node.bits ^ sibling.bits);
        Node parent = this.createNode(node.bits, parentLevel, 0.0);
        long branch = sibling.bits & parent.getBranchMask();
        if (branch == 0L) {
            parent.left = sibling;
            parent.right = node;
        } else {
            parent.left = node;
            parent.right = sibling;
        }
        return parent;
    }

    private Node createLeaf(long bits, double weight) {
        return this.createNode(bits, 0, weight);
    }

    private Node createNode(long bits, int level, double weight) {
        this.weightedCount += weight;
        ++this.totalNodeCount;
        if (weight >= 1.0E-5) {
            ++this.nonZeroNodeCount;
        }
        return new Node(bits, level, weight);
    }

    private Node merge(Node node, Node other) {
        if (node == null) {
            return this.copyRecursive(other);
        }
        if (other == null) {
            return node;
        }
        if (!QuantileDigest.inSameSubtree(node.bits, other.bits, Math.max(node.level, other.level))) {
            return this.makeSiblings(node, this.copyRecursive(other));
        }
        if (node.level > other.level) {
            long branch = other.bits & node.getBranchMask();
            if (branch == 0L) {
                node.left = this.merge(node.left, other);
            } else {
                node.right = this.merge(node.right, other);
            }
            return node;
        }
        if (node.level < other.level) {
            Node result = this.createNode(other.bits, other.level, other.weightedCount);
            long branch = node.bits & other.getBranchMask();
            if (branch == 0L) {
                result.left = this.merge(node, other.left);
                result.right = this.copyRecursive(other.right);
            } else {
                result.left = this.copyRecursive(other.left);
                result.right = this.merge(node, other.right);
            }
            return result;
        }
        double oldWeight = node.weightedCount;
        this.weightedCount += other.weightedCount;
        node.weightedCount = node.weightedCount + other.weightedCount;
        node.left = this.merge(node.left, other.left);
        node.right = this.merge(node.right, other.right);
        if (oldWeight < 1.0E-5 && node.weightedCount >= 1.0E-5) {
            ++this.nonZeroNodeCount;
        }
        return node;
    }

    private static boolean inSameSubtree(long bitsA, long bitsB, int level) {
        return level == 64 || bitsA >>> level == bitsB >>> level;
    }

    private Node copyRecursive(Node node) {
        Node result = null;
        if (node != null) {
            result = this.createNode(node.bits, node.level, node.weightedCount);
            result.left = this.copyRecursive(node.left);
            result.right = this.copyRecursive(node.right);
        }
        return result;
    }

    private Node tryRemove(Node node) {
        if (node == null) {
            return null;
        }
        if (node.weightedCount >= 1.0E-5) {
            --this.nonZeroNodeCount;
        }
        this.weightedCount -= node.weightedCount;
        Node result = null;
        if (node.isLeaf()) {
            --this.totalNodeCount;
        } else if (node.hasSingleChild()) {
            result = node.getSingleChild();
            --this.totalNodeCount;
        } else {
            node.weightedCount = 0.0;
            result = node;
        }
        return result;
    }

    private boolean postOrderTraversal(Node node, Callback callback) {
        return this.postOrderTraversal(node, callback, TraversalOrder.FORWARD);
    }

    private boolean postOrderTraversal(Node node, Callback callback, TraversalOrder order) {
        Node second;
        Node first;
        if (node == null) {
            return false;
        }
        if (order == TraversalOrder.FORWARD) {
            first = node.left;
            second = node.right;
        } else {
            first = node.right;
            second = node.left;
        }
        if (first != null && !this.postOrderTraversal(first, callback, order)) {
            return false;
        }
        if (second != null && !this.postOrderTraversal(second, callback, order)) {
            return false;
        }
        return callback.process(node);
    }

    public double getConfidenceFactor() {
        return this.computeMaxPathWeight(this.root) * 1.0 / this.weightedCount;
    }

    public boolean equivalent(QuantileDigest other) {
        this.rescaleToCommonLandmark(this, other);
        return this.totalNodeCount == other.totalNodeCount && this.nonZeroNodeCount == other.nonZeroNodeCount && this.min == other.min && this.max == other.max && this.weightedCount == other.weightedCount && Objects.equals(this.root, other.root);
    }

    private void rescaleToCommonLandmark(QuantileDigest one, QuantileDigest two) {
        long targetLandmark;
        long nowInSeconds = TimeUnit.NANOSECONDS.toSeconds(this.ticker.read());
        if (nowInSeconds - (targetLandmark = Math.max(one.landmarkInSeconds, two.landmarkInSeconds)) >= 50L) {
            targetLandmark = nowInSeconds;
        }
        if (targetLandmark != one.landmarkInSeconds) {
            one.rescale(targetLandmark);
        }
        if (targetLandmark != two.landmarkInSeconds) {
            two.rescale(targetLandmark);
        }
    }

    private double computeMaxPathWeight(Node node) {
        if (node == null || node.level == 0) {
            return 0.0;
        }
        double leftMaxWeight = this.computeMaxPathWeight(node.left);
        double rightMaxWeight = this.computeMaxPathWeight(node.right);
        return Math.max(leftMaxWeight, rightMaxWeight) + node.weightedCount;
    }

    @VisibleForTesting
    void validate() {
        final AtomicDouble sumOfWeights = new AtomicDouble();
        final AtomicInteger actualNodeCount = new AtomicInteger();
        final AtomicInteger actualNonZeroNodeCount = new AtomicInteger();
        if (this.root != null) {
            this.validateStructure(this.root);
            this.postOrderTraversal(this.root, new Callback(){

                @Override
                public boolean process(Node node) {
                    sumOfWeights.addAndGet(node.weightedCount);
                    actualNodeCount.incrementAndGet();
                    if (node.weightedCount >= 1.0E-5) {
                        actualNonZeroNodeCount.incrementAndGet();
                    }
                    return true;
                }
            });
        }
        Preconditions.checkState((Math.abs(sumOfWeights.get() - this.weightedCount) < 1.0E-5 ? 1 : 0) != 0, (String)"Computed weight (%s) doesn't match summary (%s)", (Object)sumOfWeights.get(), (Object)this.weightedCount);
        Preconditions.checkState((actualNodeCount.get() == this.totalNodeCount ? 1 : 0) != 0, (String)"Actual node count (%s) doesn't match summary (%s)", (int)actualNodeCount.get(), (int)this.totalNodeCount);
        Preconditions.checkState((actualNonZeroNodeCount.get() == this.nonZeroNodeCount ? 1 : 0) != 0, (String)"Actual non-zero node count (%s) doesn't match summary (%s)", (int)actualNonZeroNodeCount.get(), (int)this.nonZeroNodeCount);
    }

    private void validateStructure(Node node) {
        Preconditions.checkState((node.level >= 0 ? 1 : 0) != 0);
        if (node.left != null) {
            this.validateBranchStructure(node, node.left, node.right, true);
            this.validateStructure(node.left);
        }
        if (node.right != null) {
            this.validateBranchStructure(node, node.right, node.left, false);
            this.validateStructure(node.right);
        }
    }

    private void validateBranchStructure(Node parent, Node child, Node otherChild, boolean isLeft) {
        Preconditions.checkState((child.level < parent.level ? 1 : 0) != 0, (String)"Child level (%s) should be smaller than parent level (%s)", (int)child.level, (int)parent.level);
        long branch = child.bits & 1L << parent.level - 1;
        Preconditions.checkState((branch == 0L && isLeft || branch != 0L && !isLeft ? 1 : 0) != 0, (Object)"Value of child node is inconsistent with its branch");
        Preconditions.checkState((parent.weightedCount >= 1.0E-5 || child.weightedCount >= 1.0E-5 || otherChild != null ? 1 : 0) != 0, (Object)"Found a linear chain of zero-weight nodes");
    }

    @SuppressFBWarnings(value={"VA_FORMAT_STRING_USES_NEWLINE"}, justification="don't care")
    public String toGraphviz() {
        StringBuilder builder = new StringBuilder();
        builder.append("digraph QuantileDigest {\n").append("\tgraph [ordering=\"out\"];");
        final ArrayList nodes = new ArrayList();
        this.postOrderTraversal(this.root, new Callback(){

            @Override
            public boolean process(Node node) {
                nodes.add(node);
                return true;
            }
        });
        ImmutableListMultimap nodesByLevel = Multimaps.index(nodes, (Function)new Function<Node, Integer>(){

            public Integer apply(Node input) {
                return input.level;
            }
        });
        for (Map.Entry entry : nodesByLevel.asMap().entrySet()) {
            builder.append("\tsubgraph level_" + entry.getKey() + " {\n").append("\t\trank = same;\n");
            for (Node node : (Collection)entry.getValue()) {
                builder.append(String.format("\t\t%s [label=\"[%s..%s]@%s\\n%s\", shape=rect, style=filled,color=%s];\n", QuantileDigest.idFor(node), node.getLowerBound(), node.getUpperBound(), node.level, node.weightedCount, node.weightedCount > 0.0 ? "salmon2" : "white"));
            }
            builder.append("\t}\n");
        }
        for (Node node : nodes) {
            if (node.left != null) {
                builder.append(String.format("\t%s -> %s;\n", QuantileDigest.idFor(node), QuantileDigest.idFor(node.left)));
            }
            if (node.right == null) continue;
            builder.append(String.format("\t%s -> %s;\n", QuantileDigest.idFor(node), QuantileDigest.idFor(node.right)));
        }
        builder.append("}\n");
        return builder.toString();
    }

    private static String idFor(Node node) {
        return String.format("node_%x_%x", node.bits, node.level);
    }

    private static long longToBits(long value) {
        return value ^ Long.MIN_VALUE;
    }

    private static long bitsToLong(long bits) {
        return bits ^ Long.MIN_VALUE;
    }

    private static class Flags {
        public static final int HAS_LEFT = 1;
        public static final int HAS_RIGHT = 2;

        private Flags() {
        }
    }

    private static class SizeOf {
        public static final int BYTE = 1;
        public static final int INTEGER = 4;
        public static final int LONG = 8;
        public static final int DOUBLE = 8;
        public static final int QUANTILE_DIGEST = ClassLayout.parseClass(QuantileDigest.class).instanceSize();
        public static final int NODE = ClassLayout.parseClass(Node.class).instanceSize();

        private SizeOf() {
        }
    }

    private static interface Callback {
        public boolean process(Node var1);
    }

    private static class Node {
        private double weightedCount;
        private final int level;
        private final long bits;
        private Node left;
        private Node right;

        private Node(long bits, int level, double weightedCount) {
            this.bits = bits;
            this.level = level;
            this.weightedCount = weightedCount;
        }

        public boolean isLeaf() {
            return this.left == null && this.right == null;
        }

        public boolean hasSingleChild() {
            return this.left == null && this.right != null || this.left != null && this.right == null;
        }

        public Node getSingleChild() {
            Preconditions.checkState((boolean)this.hasSingleChild(), (Object)"Node does not have a single child");
            return (Node)MoreObjects.firstNonNull((Object)this.left, (Object)this.right);
        }

        public long getUpperBound() {
            long mask = 0L;
            if (this.level > 0) {
                mask = -1L >>> 64 - this.level;
            }
            return QuantileDigest.bitsToLong(this.bits | mask);
        }

        public long getBranchMask() {
            return 1L << this.level - 1;
        }

        public long getLowerBound() {
            long mask = 0L;
            if (this.level > 0) {
                mask = -1L >>> 64 - this.level;
            }
            return QuantileDigest.bitsToLong(this.bits & (mask ^ 0xFFFFFFFFFFFFFFFFL));
        }

        public long getMiddle() {
            return this.getLowerBound() + (this.getUpperBound() - this.getLowerBound()) / 2L;
        }

        public String toString() {
            return String.format("%s (level = %d, count = %s, left = %s, right = %s)", this.bits, this.level, this.weightedCount, this.left != null, this.right != null);
        }

        public int hashCode() {
            return Objects.hash(this.weightedCount, this.level, this.bits, this.left, this.right);
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null || this.getClass() != obj.getClass()) {
                return false;
            }
            Node other = (Node)obj;
            return Objects.equals(this.weightedCount, other.weightedCount) && Objects.equals(this.level, other.level) && Objects.equals(this.bits, other.bits) && Objects.equals(this.left, other.left) && Objects.equals(this.right, other.right);
        }
    }

    public static class Bucket {
        private final double count;
        private final double mean;

        public Bucket(double count, double mean) {
            this.count = count;
            this.mean = mean;
        }

        public double getCount() {
            return this.count;
        }

        public double getMean() {
            return this.mean;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            Bucket bucket = (Bucket)o;
            if (Double.compare(bucket.count, this.count) != 0) {
                return false;
            }
            return Double.compare(bucket.mean, this.mean) == 0;
        }

        public int hashCode() {
            long temp = this.count != 0.0 ? Double.doubleToLongBits(this.count) : 0L;
            int result = (int)(temp ^ temp >>> 32);
            temp = this.mean != 0.0 ? Double.doubleToLongBits(this.mean) : 0L;
            result = 31 * result + (int)(temp ^ temp >>> 32);
            return result;
        }

        public String toString() {
            return String.format("[count: %f, mean: %f]", this.count, this.mean);
        }
    }

    private static enum TraversalOrder {
        FORWARD,
        REVERSE;

    }
}

