package org.qas.api;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;

import java.io.*;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.Map;
import java.util.logging.Logger;

/**
 * @author trongle
 * @version $Id 4/21/2017 3:42 PM
 */
public class JsonMapper {
  private static final Logger LOG = Logger.getLogger(JsonMapper.class.getName());
  /**
   * Use for JSON
   */
  private static final ObjectMapper mapper = new ObjectMapper();
  
  static {
    // mapper
    mapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
  }
  
  private JsonMapper() {
  }
  
  /**
   * Create new ObjectNode
   *
   * @return
   */
  public static ObjectNode newNode() {
    return mapper.createObjectNode();
  }
  
  /**
   * Get instance of valueType from JSON data
   *
   * @param body JSON string
   * @param valueType class type to cast
   *
   * @return instance of class valueType
   */
  public static <T> T fromJson(String body, Class<T> valueType) throws IOException {
    return parseJson(body, valueType);
  }
  
  /**
   * @param body
   * @param valueType
   * @param <T>
   *
   * @return
   *
   * @throws IOException
   */
  public static <T> T parseJson(String body, Class<T> valueType) throws IOException {
    if (isEmpty(body))
      return null;
    return mapper.readValue(body, valueType);
  }
  
  public static <T> T parseJson(String body, TypeReference<T> type) throws IOException {
    if (isEmpty(body))
      return null;
    return mapper.readValue(body, type);
  }
  
  public static <T> T fromJson(String body, TypeReference<T> type) {
    try {
      if (isEmpty(body))
        return null;
      return mapper.readValue(body, type);
    } catch (IOException e) {
      LOG.info("ERROR: cannot parse json" + e.getMessage());
      return null;
    }
  }
  
  /**
   * Get JsonNode from object append field extraData to JsonNode
   *
   * @return JsonNode, if data is a instance of POJO. Otherwise return empty string
   */
  public static ObjectNode toJsonNode(Object data) {
    if (null == data)
      return newNode();
    
    ObjectNode node = null;
    
    try {
      // convert data to ObjectNode
      node = mapper.valueToTree(data);
    } catch (IllegalArgumentException e) {
      return newNode();
    }
    return node;
  }
  
  /**
   * serial object to JSON string
   *
   * @param data
   *
   * @return
   */
  public static String toJson(Object data) {
    if (null == data)
      return "";
    try {
      return mapper.writeValueAsString(data);
    } catch (JsonProcessingException e) {
      return "";
    }
  }
  
  public static Map<String, Object> toMap(Object data) {
    if (null == data) {
      return Collections.emptyMap();
    }
    try {
      return mapper.convertValue(data, new TypeReference<Map<String, Object>>() {
      });
    } catch (Exception e) {
      return Collections.emptyMap();
    }
  }
  
  private static boolean isEmpty(String s) {
    return null == s || "".equals(s);
  }
  
  
  public static String toString(final InputStream stream) {
    try {
      ByteArrayOutputStream result = new ByteArrayOutputStream();
      byte[] buffer = new byte[1024];
      int length;
      while ((length = stream.read(buffer)) != -1) {
        result.write(buffer, 0, length);
      }
      return result.toString("UTF-8");
    } catch (Exception e) {
      throw new AuthClientException("Unable to read error response: " + e.getMessage(), e);
    } finally {
      try {
        stream.close();
      } catch (Exception ex) {
      }
    }
  }
}
