package org.qas.api.http.basic;

import org.qas.api.Request;
import org.qas.api.http.AbstractHttpRequest;
import org.qas.api.http.HttpMethod;
import org.qas.api.http.HttpRequest;
import org.qas.api.internal.util.google.base.Preconditions;

import java.io.InputStream;
import java.net.HttpURLConnection;
import java.util.Map;

/**
 * HttpUrlConnectionRequest
 *
 * @author Dzung Nguyen
 * @version $Id HttpUrlConnectionRequest 2014-03-27 12:01:30z dungvnguyen $
 * @since 1.0
 */
public final class HttpUrlConnectionRequest extends AbstractHttpRequest<HttpURLConnection>
    implements HttpRequest<HttpURLConnection> {
  //~ class properties ========================================================
  private final HttpURLConnection connection;

  //~ class members ===========================================================
  /**
   * Constructs {@link HttpUrlConnectionRequest} associated with the specified
   * request and connection.
   *
   * @param request the associated request that generated this request.
   * @param connection the associated connection that generated this request.
   */
  public HttpUrlConnectionRequest(Request request, HttpURLConnection connection) {
    super(request);
    Preconditions.checkNotNull(connection);

    this.connection = connection;
  }

  @Override
  public HttpURLConnection getUnderlying() {
    return connection;
  }

  @Override
  public HttpUrlConnectionRequest withHttpMethod(HttpMethod httpMethod) {
    setHttpMethod(httpMethod);
    return this;
  }

  @Override
  public HttpUrlConnectionRequest withContent(InputStream content) {
    setContent(content);
    return this;
  }

  @Override
  public int getContentLength() {
    try {
      return Integer.valueOf(connection.getRequestProperty("Content-Length"));
    } catch (Exception ex) {
      return -1;
    }
  }

  @Override
  public HttpUrlConnectionRequest withTimeOffset(int timeOffset) {
    setTimeOffset(timeOffset);
    return this;
  }

  @Override
  public String getContentType() {
    return connection.getContentType();
  }

  @Override
  public HttpUrlConnectionRequest withResourcePath(String resourcePath) {
    setResourcePath(resourcePath);
    return this;
  }

  @Override
  public HttpUrlConnectionRequest withHeader(String name, String value) {
    setHeader(name, value);
    return this;
  }

  @Override
  public HttpUrlConnectionRequest withHeaders(Map<String, String> headers) {
    setHeaders(headers);
    return this;
  }

  @Override
  public HttpUrlConnectionRequest withParameter(String name, String value) {
    addParameter(name, value);
    return this;
  }

  @Override
  public HttpUrlConnectionRequest withParameters(Map<String, String> parameters) {
    setParameters(parameters);
    return this;
  }
}
