package org.qas.qtest.api.auth;

import org.qas.api.AuthClientException;

import java.io.IOException;
import java.io.InputStream;

/**
 * ClasspathPropertiesQTestCredentialsProvider
 *
 * @author Dzung Nguyen
 * @version $Id ClasspathPropertiesQTestCredentialsProvider 2014-03-27 13:41:30z dungvnguyen $
 * @since 1.0
 */
public class ClasspathPropertiesQTestCredentialsProvider implements QTestCredentialsProvider {
  //~ class properties ========================================================
  private static final String DEFAULT_PROPERTIES_FILE = "qTestCredentials.properties";
  private final String credentialsFilePath;

  //~ class members ===========================================================
  public ClasspathPropertiesQTestCredentialsProvider() {
    this(DEFAULT_PROPERTIES_FILE);
  }

  /**
   * Creates a new ClasspathPropertiesQTestCredentialsProvider that will
   * attempt to load a custom file from the classpath to read qTest security
   * credentials.
   *
   * @param credentialsFilePath
   *            The custom classpath resource path to a properties file
   *            from which the qTest security credentials should be loaded.
   *
   *            For example,
   *            <ul>
   *              <li>com/mycompany/credentials.properties</li>
   *              <li>qtest-credentials.properties</li>
   *              <li>qTestCredentials.properties</li>
   *            </ul>
   */
  public ClasspathPropertiesQTestCredentialsProvider(String credentialsFilePath) {
    if (credentialsFilePath == null)
      throw new IllegalArgumentException("Credentials file path cannot be null.");

    if (!credentialsFilePath.startsWith("/")) {
      this.credentialsFilePath = "/" + credentialsFilePath;
    } else {
      this.credentialsFilePath = credentialsFilePath;
    }
  }

  public QTestCredentials getCredentials() {
    InputStream inputStream = getClass().getResourceAsStream(credentialsFilePath);
    if (inputStream == null) {
      throw new AuthClientException("Unable to load qTest credentials from the " + credentialsFilePath
          + " file on the classpath");
    }

    try {
      return new PropertiesQTestCredentials(inputStream);
    } catch (IOException ioex) {
      throw new AuthClientException("Unable to load qTest credentials from the " + credentialsFilePath
          + " file on the classpath", ioex);
    }
  }

  public void refresh() {}

  @Override
  public String toString() {
    final StringBuilder sb = new StringBuilder("ClasspathPropertiesQTestCredentialsProvider{");
    sb.append("credentialsFilePath='").append(credentialsFilePath).append('\'');
    sb.append('}');
    return sb.toString();
  }
}
