package org.qas.qtest.api.http;

import org.qas.api.ApiServiceResponse;
import org.qas.api.ResponseMetadata;
import org.qas.api.http.HttpResponse;
import org.qas.api.http.HttpResponseHandler;
import org.qas.api.internal.util.UniqueKeys;
import org.qas.api.internal.util.google.io.ByteStreams;
import org.qas.api.transform.AbstractUnmarshaller;
import org.qas.api.transform.Unmarshaller;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * StreamResponseHandler
 *
 * @author Dzung Nguyen
 * @version $Id StreamResponseHandler 2014-03-27 13:55:30z dungvnguyen $
 * @since 1.0
 */
public class StreamResponseHandler<T> implements HttpResponseHandler<ApiServiceResponse<T>> {
  //~ class properties ========================================================
  /* shared logger for profiling information. */
  private static final Logger LOG = Logger.getLogger(StreamResponseHandler.class.getName());

  /* the response unmarshaller. */
  private Unmarshaller<T, InputStream> responseUnmarshaller;

  //~ class members ===========================================================
  public StreamResponseHandler(Unmarshaller<T, InputStream> responseUnmarshaller) {
    this.responseUnmarshaller = responseUnmarshaller;

    /*
     * Even if the invoked operation just returns null, we still need an
     * unmarshaller to run so we can pull out response metadata.
     *
     * We might want to pass this in through the client class so that we
     * don't have to do this check here.
     */
    if (responseUnmarshaller == null) {
      this.responseUnmarshaller = new Unmarshaller<T, InputStream>() {
        @Override
        public boolean isParseJson() {
          return false;
        }
  
        @Override
        public T unmarshall(InputStream in) throws Exception {
          return null;
        }
  
        @Override
        public T parse(String data) throws Exception {
          return null;
        }
      };
    }
  }

  public ApiServiceResponse<T> handle(HttpResponse<?> httpResponse) throws Exception {
    if (LOG.isLoggable(Level.ALL)) {
      LOG.log(Level.ALL, "Handle the service response.");
    }

    InputStream content = httpResponse.getContent();
    File file = File.createTempFile(UniqueKeys.generateNonce(), ".stream");
    FileOutputStream outputStream = new FileOutputStream(file);

    try {
      ApiServiceResponse<T> awsResponse = new ApiServiceResponse<T>();

      // process content.
      if (content != null) {
        ByteStreams.copy(content, outputStream);
        outputStream.close();
        outputStream = null;
        awsResponse.setResult(responseUnmarshaller.unmarshall(new FileInputStream(file)));
      } else {
        file.delete();
        awsResponse.setResult(null);
      }

      // read metadata.
      Map<String, String> metadata = new HashMap<String, String>();
      metadata.put(ResponseMetadata.REQUEST_ID, httpResponse.getHeaders().get("x-qtest-request-id"));
      awsResponse.setMetadata(new ResponseMetadata(metadata));

      // aws response.
      return awsResponse;
    } finally {
      try { if (outputStream != null) outputStream.close(); } catch (Exception ex) {}
    }
  }

  public boolean needsConnectionLeftOpen() {
    return false;
  }
}
