package org.qas.qtest.api.internal.model.transform;

import org.qas.api.ApiServiceRequest;
import org.qas.api.AuthClientException;
import org.qas.api.Constants;
import org.qas.api.Request;
import org.qas.api.http.HttpMethod;
import org.qas.api.internal.util.UniqueKeys;
import org.qas.api.internal.util.google.net.HttpHeaders;
import org.qas.api.support.RequestSupport;
import org.qas.api.transform.Marshaller;

import static org.qas.qtest.api.internal.util.ApiPreconditions.checkArgument;

/**
 * AbstractQTestApiRequestMarshaller
 *
 * @author Dzung Nguyen
 * @version $Id AbstractQTestApiRequestMarshaller 2015-06-01 11:07:30z dzungvnguyen $
 * @since 1.0
 */
public abstract class AbstractQTestApiRequestMarshaller<T extends ApiServiceRequest>
  implements Marshaller<Request, T> {
  //~ class members ===========================================================

  /**
   * Validate project identifier.
   *
   * @param projectId the given project identifier to validate.
   * @throws AuthClientException if an error occurs during validate project identifier.
   */
  protected static void validateProjectId(Long projectId) throws AuthClientException {
    validateId(projectId, "Invalid project id passed to marshall(...)");
  }

  /**
   * Validate entity identifier.
   *
   * @param id      the given entity identifier.
   * @param message the given exception message.
   * @throws AuthClientException if an error occurs during validate entity identifier.
   */
  protected static void validateId(Long id, String message) throws AuthClientException {
    checkArgument(id != null && id > 0L, message);
  }

  /**
   * Creates json request from the given request type, request action and request method.
   *
   * @param requestType   the given request type.
   * @param serviceName   the given service name.
   * @param requestAction the given request action.
   * @param requestMethod the given request method.
   * @return the request instance.
   */
  protected Request createJsonRequest(T requestType, String serviceName,
                                      String requestAction, HttpMethod requestMethod) {
    Request request = new RequestSupport(requestType, serviceName);

    // set additional parameter.
    request.setHeader(Constants.Header.X_ACTION, requestAction);
    request.setHeader(Constants.Header.X_API_VERSION, "V3");
    request.setHeader(Constants.Header.X_NONCE, UniqueKeys.generateNonce());
    request.setHeader(Constants.Header.X_SERVICE_NAME, request.getServiceName());

    request.setHeader(HttpHeaders.CONTENT_TYPE, Constants.ContentType.JSON);
    request.setHeader(HttpHeaders.ACCEPT, Constants.ContentType.JSON);

    // set request method.
    request.setHttpMethod(requestMethod);

    return request;
  }

  /**
   * @return V3 base path request.
   */
  protected String getApiV3BasePath() {
    return "/api/v3";
  }

  /**
   * @return the based request from the given project identifier.
   */
  protected String createApiBasePathWithProject(Long projectId) {
    return String.format("%s/projects/%s", getApiV3BasePath(), projectId);
  }
}
